
////////full map
var gDragSystemOriginal = gDragSystem;
var gCanvasResizeTimeout;
var gZoneFullMap_Quadrilaterals = [];
var gZoneFullMap_ActiveRectangles = [];
var gZoneFullMap_DeletedIds = [];
var gZoneFullMap_LastSelectedQuad = null;
var gZoneFullMap_DraggingQuadrilateral = null;
var gZoneFullMap_DraggingCornerIndex = -1;
var gZoneFullMap_IsDrawing = false;
var gZoneFullMap_StartPoint = { x: 0, y: 0 };
var gZoneFullMap_DrawingMode = false;
var gZoneFullMap_SelectedMicrophone = null;
var gZoneFullMap_CornerDragStart = null; 
var gSelectedDevice = null;
var gIsDraggingDevice = false;
var gDeviceDragOffset = { x: 0, y: 0 };
var gZoneFullMap_GlobalDrawingMode = false;
var gZoneFullMap_MicrophoneZoneCounters = {};
var gSimpleZoneStates = {};
var gSimpleSoundPoints = {};
var gOverlappingDeviceSelector = null;
var gZoneFullMap_GroupSelectionMode = false;
var gZoneFullMap_SelectedGroup = [];
var gZoneFullMap_IsCtrlPressed = false;
var gZoneFullMap_DraggedDeviceType = null;
var gZoneFullMap_DragPreview = null;
var gZoneFullMap_IsDraggingFromButton = false;
var gZoneFullMap_DragStartPos = { x: 0, y: 0 };

const MICROPHONE_COLOR_SCHEMES = [
    {
        name: 'Electric Blue',
        primary: '#2196F3',
        light: '#64B5F6', 
        dark: '#1565C0',
        zone_fill: 'rgba(33, 150, 243, 0.3)',
        zone_stroke: '#1976D2',
        highlight: '#42A5F5'
    },
    {
        name: 'Forest Green',
        primary: '#4CAF50',
        light: '#81C784',
        dark: '#2E7D32', 
        zone_fill: 'rgba(76, 175, 80, 0.3)',
        zone_stroke: '#388E3C',
        highlight: '#66BB6A'
    },
    {
        name: 'Sunset Orange',
        primary: '#FF9800',
        light: '#FFB74D',
        dark: '#F57C00',
        zone_fill: 'rgba(255, 152, 0, 0.3)',
        zone_stroke: '#F9A825',
        highlight: '#FFA726'
    },
    {
        name: 'Royal Purple',
        primary: '#9C27B0',
        light: '#BA68C8',
        dark: '#6A1B9A',
        zone_fill: 'rgba(156, 39, 176, 0.3)',
        zone_stroke: '#7B1FA2',
        highlight: '#AB47BC'
    },
    {
        name: 'Crimson Red',
        primary: '#F44336',
        light: '#E57373',
        dark: '#C62828',
        zone_fill: 'rgba(244, 67, 54, 0.3)',
        zone_stroke: '#D32F2F',
        highlight: '#EF5350'
    },
    {
        name: 'Chocolate Brown',
        primary: '#8D4E85',
        light: '#A1887F',
        dark: '#5D4037',
        zone_fill: 'rgba(141, 78, 133, 0.3)',
        zone_stroke: '#6D4C41',
        highlight: '#8D6E63'
    },
    {
        name: 'Deep Pink',
        primary: '#E91E63',
        light: '#F06292',
        dark: '#AD1457',
        zone_fill: 'rgba(233, 30, 99, 0.3)',
        zone_stroke: '#C2185B',
        highlight: '#EC407A'
    },
    {
        name: 'Amber Gold',
        primary: '#FFC107',
        light: '#FFD54F',
        dark: '#F57F17',
        zone_fill: 'rgba(255, 193, 7, 0.3)',
        zone_stroke: '#FFA000',
        highlight: '#FFCA28'
    }
];

function ZoneFullMap_getMicrophoneColorScheme(deviceId) 
{
    const micMatch = deviceId.match(/mic_(\d+)/);
    if (micMatch) {
        const micNumber = parseInt(micMatch[1]) - 1;
        const colorIndex = micNumber % MICROPHONE_COLOR_SCHEMES.length;
        return MICROPHONE_COLOR_SCHEMES[colorIndex];
    }
    
    let hash = 0;
    for (let i = 0; i < deviceId.length; i++) {
        hash = deviceId.charCodeAt(i) + ((hash << 5) - hash);
    }
    const index = Math.abs(hash) % MICROPHONE_COLOR_SCHEMES.length;
    return MICROPHONE_COLOR_SCHEMES[index];
};

function ZoneFullMap_getCameraColorScheme(deviceId) 
{
    const micMatch = deviceId.match(/camera_(\d+)/);
    if (micMatch) {
        const micNumber = parseInt(micMatch[1]) - 1;
        const colorIndex = micNumber % MICROPHONE_COLOR_SCHEMES.length;
        return MICROPHONE_COLOR_SCHEMES[colorIndex];
    }
    
    let hash = 0;
    for (let i = 0; i < deviceId.length; i++) {
        hash = deviceId.charCodeAt(i) + ((hash << 5) - hash);
    }
    const index = Math.abs(hash) % MICROPHONE_COLOR_SCHEMES.length;
    return MICROPHONE_COLOR_SCHEMES[index];
};

function ZoneFullMap_getVisionColorScheme(deviceId) 
{
    const micMatch = deviceId.match(/vision_(\d+)/);
    if (micMatch) {
        const micNumber = parseInt(micMatch[1]) - 1;
        const colorIndex = micNumber % MICROPHONE_COLOR_SCHEMES.length;
        return MICROPHONE_COLOR_SCHEMES[colorIndex];
    }
    
    let hash = 0;
    for (let i = 0; i < deviceId.length; i++) {
        hash = deviceId.charCodeAt(i) + ((hash << 5) - hash);
    }
    const index = Math.abs(hash) % MICROPHONE_COLOR_SCHEMES.length;
    return MICROPHONE_COLOR_SCHEMES[index];
};


var gAnchorMovementTracker = {
    positions: {},
    isMoving: {},
    checkInterval: 200,
    movementThreshold: 2,
    stabilityTime: 1000,
    lastMovingStates: {} // Track previous moving states for change detection
};

var gDragSystem = {
    isInitialized: false,
    isDragging: false,
    isResizing: false,
    hasStarted: false,
    element: null,
    resizeDirection: '',
    dragThreshold: 9,
    minWidth: 800,
    minHeight: 600,
    maxWidth: 3200,
    maxHeight: 2400,
    currentScale: 1.0,
    minScale: 0.7,
    maxScale: 12.0,
    scaleStep: 0.1,
    gridConfig: {
        enabled: true,
        majorSpacing: 60,    // 60cm 主格線 (假設 1cm = 1px)
        minorSpacing: 20,    // 20cm 次格線
        majorColor: '#444',  // 主格線顏色
        minorColor: '#222',  // 次格線顏色  
        axisColor: '#666',   // 坐標軸顏色
        majorWidth: 2,       // 主格線寬度 (60cm線條)
        minorWidth: 0.1,     // 次格線寬度 (中間線條)
        axisWidth: 4,        // 坐標軸寬度 (0,0線最粗)
        showLabels: true,
        labelColor: '#000000',
        labelFont: '20px Arial',
        labelBackground: {
            enabled: true,
            color: 'rgba(255, 255, 255,1)',
            xAxis: {
                height: 28,
                offsetFromBottom: 0,
                marginLeft: 0,
                marginRight: 0
            },
            yAxis: {
                width: 70,
                offsetFromLeft: 0,
                marginTop: 0,
                marginBottom: 0
            }
        }
    },
    elementRect: { width: 0, height: 0 },
    windowSize: { width: 0, height: 0 },
    boundaries: { minX: 0, minY: 0, maxX: 0, maxY: 0 },
    mouseStart: { x: 0, y: 0 },
    mouseCurrent: { x: 0, y: 0 },
    dragOffset: { x: 0, y: 0 },
    currentPos: { x: 0, y: 0 },
    initialSize: { width: 0, height: 0 },
    initialPos: { x: 0, y: 0 },
    needsBoundaryUpdate: true,
    savedState: {
        scale: 1.0,
        position: { x: 0, y: 0 },
        windowSize: { width: 0, height: 0 },
        canvasTransform: { x: 0, y: 0, scale: 1.0 }
    },
    canvasTransform: {
        x: 0,
        y: 0,
        scale: 1.0
    },
    isCanvasDragging: false,
    lastRedrawTime: 0,
    lastLabelTransform: {
        x: null,
        y: null,
        scale: null
    },

    init: function(element, forceFullScreen = true, goToHome = false) {
        this.element = element;
        if (!this.element) {
            return;
        }
        this.isInitialized = true;
        this.updateCachedValues();
        this.setupGPUAcceleration();
        this.setupWheelZoom();
        this.setupCanvasDrag();
        this.calculateMaxScale();
        
        this.startBrowserZoomMonitoring();
        
        this._gridRedrawPending = false;
        this._moveRedrawPending = false;
        this._wheelRedrawPending = false;
        this._dragRedrawPending = false;
        this._batchRedrawTimeout = null;
        this.lastLabelTransform = {
            x: null,
            y: null,
            scale: null
        };
        
        this.createGridCanvas();
        
        if (forceFullScreen) {
            if (goToHome) {
                const canvas = this.gridCanvas;
                if (canvas) {
                    this.canvasTransform.x = canvas.width / 2;
                    this.canvasTransform.y = canvas.height * 5 / 10;
                    this.canvasTransform.scale = 1.5;
                }
            }
            this.resetToFullScreen(true); 
        } else {
            this.loadSavedState();
            requestAnimationFrame(() => {
                resizeCanvases();
            });
        }
    },

    initializeGrid: function() {
        this.createGridCanvas();
        requestAnimationFrame(() => {
            resizeCanvases();
        });
    },

    createGridCanvas: function(attempt = 0, maxAttempts = 5) {
        const canvasContainer = document.querySelector('.ZoneFullMap_canvas_td');
        if (!canvasContainer) {
            if (attempt < maxAttempts) {
                console.warn(`Canvas container not found, retrying (${attempt + 1}/${maxAttempts})`);
                setTimeout(() => this.createGridCanvas(attempt + 1, maxAttempts), 100);
            } else {
                console.error('Failed to find canvas container after maximum retries');
            }
            return;
        }
      
        let gridCanvas = document.getElementById('ZoneFullMap_xy_canvas_grid');
        if (!gridCanvas) {
            gridCanvas = document.createElement('canvas');
            gridCanvas.id = 'ZoneFullMap_xy_canvas_grid';
            gridCanvas.className = 'ZoneFullMap_canvas_border';
            gridCanvas.style.position = 'absolute';
            gridCanvas.style.zIndex = '0';
            canvasContainer.appendChild(gridCanvas);
        }
      
        this.gridCanvas = gridCanvas;
        this.gridContext = gridCanvas.getContext('2d');
      
        const containerRect = canvasContainer.getBoundingClientRect();
        gridCanvas.width = containerRect.width;
        gridCanvas.height = containerRect.height;
      
        //console.log('Grid canvas created with size:', gridCanvas.width, 'x', gridCanvas.height);
    },

    coordinateTransform: {
        mathToWeb: function(mathX, mathY, centerX, centerY, scale = 1) {
            return {
                webX: centerX + mathX * scale,
                webY: centerY - mathY * scale
            };
        },
        webToMath: function(webX, webY, centerX, centerY, scale = 1) {
            return {
                mathX: (webX - centerX) / scale,
                mathY: (centerY - webY) / scale
            };
        }
    },

    drawGrid: function() {
        if (!this.gridCanvas || !this.gridContext || !this.gridConfig.enabled) {
            return;
        }
        
        if (this._gridRedrawPending) {
            return;
        }
        
        if (ZoneFullMapRenderController.isMoving) {
            ZoneFullMapRenderController.requestGridRedraw();
            this._gridRedrawPending = false;
            return;
        }
    
        this._gridRedrawPending = true;
        
        requestAnimationFrame(() => {
            this._performGridRedraw();
            this._gridRedrawPending = false;
        });
    },
 

    drawOriginPoint: function(ctx, centerX, centerY, scale) {
        const originPos = this.coordinateTransform.mathToWeb(0, 0, centerX, centerY, scale);
        
        if (originPos.webX >= 0 && originPos.webX <= ctx.canvas.width &&
            originPos.webY >= 0 && originPos.webY <= ctx.canvas.height) {
            
            let size;
            if (scale <= 0.5) {
                size = Math.max(8, Math.min(10, 3 / scale));
            } else if (scale <= 1.0) {
                size = Math.max(6, Math.min(8, 2 / scale));
            } else if (scale <= 2.0) {
                size = Math.max(5, Math.min(7, 2 / scale));
            } else if (scale <= 5.0) {
                size = Math.max(5, Math.min(6, 1.5 / scale));
            } else {
                size = Math.max(5, Math.min(5.5, 1 / scale));
            }
            
            ctx.beginPath();
            ctx.moveTo(originPos.webX, originPos.webY - size);
            ctx.lineTo(originPos.webX + size, originPos.webY);
            ctx.lineTo(originPos.webX, originPos.webY + size);
            ctx.lineTo(originPos.webX - size, originPos.webY);
            ctx.closePath();
            
            ctx.fillStyle = 'white';
            ctx.fill();
            
            ctx.strokeStyle = '#000';
            ctx.lineWidth = Math.max(0.5, Math.min(3, 2 / scale));
            ctx.stroke();
        }
    },

    drawGridLines: function(ctx, centerX, centerY, spacing, color, width, bounds) {
        ctx.strokeStyle = color;
        ctx.lineWidth = width;
        ctx.beginPath();
    
        const gridSpacingMath = spacing / this.canvasTransform.scale;
    
        const startX = Math.floor(bounds.left / gridSpacingMath) * gridSpacingMath;
        const endX = Math.ceil(bounds.right / gridSpacingMath) * gridSpacingMath;
        
        for (let mathX = startX; mathX <= endX; mathX += gridSpacingMath) {
            const webPos = this.coordinateTransform.mathToWeb(mathX, 0, centerX, centerY, this.canvasTransform.scale);
            if (webPos.webX >= 0 && webPos.webX <= ctx.canvas.width) {
                ctx.moveTo(webPos.webX, 0);
                ctx.lineTo(webPos.webX, ctx.canvas.height);
            }
        }
    
        const startY = Math.floor(bounds.bottom / gridSpacingMath) * gridSpacingMath;
        const endY = Math.ceil(bounds.top / gridSpacingMath) * gridSpacingMath;
        
        for (let mathY = startY; mathY <= endY; mathY += gridSpacingMath) {
            const webPos = this.coordinateTransform.mathToWeb(0, mathY, centerX, centerY, this.canvasTransform.scale);
            if (webPos.webY >= 0 && webPos.webY <= ctx.canvas.height) {
                ctx.moveTo(0, webPos.webY);
                ctx.lineTo(ctx.canvas.width, webPos.webY);
            }
        }
    
        ctx.stroke();
    },

    drawGridLabelsImproved: function(ctx, centerX, centerY, majorSpacing, scale, bounds) {
        const gridSpacingMath = majorSpacing / scale;
        
        // 根據縮放動態調整標籤步長
        let labelStepMultiplier = 1;
        const baseFont = 24;
        let fontSize = baseFont;
        const currentScale = this.canvasTransform.scale;
        
        // 動態字體大小調整
        if (currentScale <= 0.3) {
            labelStepMultiplier = 10;
            fontSize = Math.max(12, baseFont * 0.7);
        } else if (currentScale <= 0.5) {
            labelStepMultiplier = 5;
            fontSize = Math.max(14, baseFont * 0.8);
        } else if (currentScale <= 0.7) {
            labelStepMultiplier = 3;
            fontSize = Math.max(16, baseFont * 0.85);
        } else if (currentScale <= 1.0) {
            labelStepMultiplier = 3;
            fontSize = Math.max(18, baseFont * 0.9);
        } else if (currentScale <= 1.3) {
            labelStepMultiplier = 2;
            fontSize = Math.max(20, baseFont * 0.95);
        } else if (currentScale <= 1.6) {
            labelStepMultiplier = 2;
            fontSize = Math.max(22, baseFont * 1);
        } else if (currentScale <= 1.9) {
            labelStepMultiplier = 1;
            fontSize = Math.max(24, baseFont * 1.05);
        } else if (currentScale <= 2.2) {
            labelStepMultiplier = 1;
            fontSize = Math.max(26, baseFont * 1.1);
        } else if (currentScale <= 2.5) {
            labelStepMultiplier = 1;
            fontSize = Math.max(28, baseFont * 1.15);
        }  else {
            labelStepMultiplier = 1;
            fontSize = Math.min(30, baseFont * 1.2);
        } 
          
        ctx.font = `${fontSize}px Arial`;
        const labelSpacing = gridSpacingMath * labelStepMultiplier;
        
        // 標籤樣式配置
        const labelStyle = {
            textColor: '#000000',
            backgroundColor: 'rgba(255, 255, 255, 0.9)',
            borderColor: 'rgba(0, 0, 0, 0.2)',
            borderWidth: 1,
            padding: {
                x: 8,
                y: 4
            },
            borderRadius: 6,
            shadowColor: 'rgba(0, 0, 0, 0.15)',
            shadowBlur: 4,
            shadowOffset: { x: 0, y: 2 }
        };
    
        // console.log('Drawing labels with style:', {
        //     currentScale: currentScale.toFixed(3),
        //     fontSize: fontSize,
        //     labelStepMultiplier,
        //     effectiveLabelSpacing: labelSpacing.toFixed(2)
        // });
        
        // === 繪製 X軸標籤（固定在底部） ===
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        
        const startX = Math.floor(bounds.left / labelSpacing) * labelSpacing;
        const endX = Math.ceil(bounds.right / labelSpacing) * labelSpacing;
        
        // X軸標籤固定在畫布底部
        const xLabelY = ctx.canvas.height - 20;
        
        for (let mathX = startX; mathX <= endX; mathX += labelSpacing) {
            
            const webPos = this.coordinateTransform.mathToWeb(mathX, 0, centerX, centerY, scale);
            
            // 確保標籤在可見範圍內
            if (webPos.webX >= 40 && webPos.webX <= ctx.canvas.width - 40) {
                const meterValue = (mathX / 100).toFixed(1);

                const labelText = meterValue + 'm';
                
                // 測量文字尺寸
                const textMetrics = ctx.measureText(labelText);
                const textWidth = textMetrics.width;
                const textHeight = fontSize;
                
                // 計算背景框尺寸和位置
                const boxWidth = textWidth + labelStyle.padding.x * 2;
                const boxHeight = textHeight + labelStyle.padding.y * 2;
                const boxX = webPos.webX - boxWidth / 2;
                const boxY = xLabelY - boxHeight / 2;
                
                // 繪製圓角背景框
                this.drawRoundedRect(ctx, boxX, boxY, boxWidth, boxHeight, labelStyle.borderRadius, labelStyle);
                
                // 繪製文字
                ctx.fillStyle = labelStyle.textColor;
                ctx.fillText(labelText, webPos.webX, xLabelY);
            }
        }
        
        // === 繪製 Y軸標籤（固定在左側） ===
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        
        const startY = Math.floor(bounds.bottom / labelSpacing) * labelSpacing;
        const endY = Math.ceil(bounds.top / labelSpacing) * labelSpacing;
        
        const yLabelX = 35;
        
        for (let mathY = startY; mathY <= endY; mathY += labelSpacing) {
            
            const webPos = this.coordinateTransform.mathToWeb(0, mathY, centerX, centerY, scale);
            
            if (webPos.webY >= 25 && webPos.webY <= ctx.canvas.height - 45) {
                const meterValue = (mathY / 100).toFixed(1);
                
                const labelText = meterValue + 'm';
                
                const textMetrics = ctx.measureText(labelText);
                const textWidth = textMetrics.width;
                const textHeight = fontSize;
                
                const boxWidth = textWidth + labelStyle.padding.x * 2;
                const boxHeight = textHeight + labelStyle.padding.y * 2;
                const boxX = yLabelX - boxWidth / 2;
                const boxY = webPos.webY - boxHeight / 2;
                
                this.drawRoundedRect(ctx, boxX, boxY, boxWidth, boxHeight, labelStyle.borderRadius, labelStyle);
                
                ctx.fillStyle = labelStyle.textColor;
                ctx.fillText(labelText, yLabelX, webPos.webY);
            }
        }
    },
    
    drawRoundedRect: function(ctx, x, y, width, height, radius, style) {
        ctx.save();
        
        if (style.shadowBlur) {
            ctx.shadowColor = style.shadowColor || 'rgba(0, 0, 0, 0.5)';
            ctx.shadowBlur = style.shadowBlur;
            ctx.shadowOffsetX = style.shadowOffset?.x || 0;
            ctx.shadowOffsetY = style.shadowOffset?.y || 0;
        }
        
        ctx.beginPath();
        ctx.moveTo(x + radius, y);
        ctx.lineTo(x + width - radius, y);
        ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
        ctx.lineTo(x + width, y + height - radius);
        ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
        ctx.lineTo(x + radius, y + height);
        ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
        ctx.lineTo(x, y + radius);
        ctx.quadraticCurveTo(x, y, x + radius, y);
        ctx.closePath();
        
        if (style.backgroundColor) {
            ctx.fillStyle = style.backgroundColor;
            ctx.fill();
        }
        
        ctx.shadowColor = 'transparent';
        ctx.shadowBlur = 0;
        ctx.shadowOffsetX = 0;
        ctx.shadowOffsetY = 0;
        
        if (style.borderColor && style.borderWidth) {
            ctx.strokeStyle = style.borderColor;
            ctx.lineWidth = style.borderWidth;
            ctx.stroke();
        }

        ctx.restore();
    },

    drawLabelBackgroundBars: function(ctx, centerX, centerY, majorSpacing, scale, bounds) {
        const config = this.gridConfig.labelBackground;
        
        ctx.fillStyle = config.color;
        
        // 繪製X軸標籤背景條（底部）
        const xBarHeight = config.xAxis.height;
        const xBarY = ctx.canvas.height - xBarHeight - config.xAxis.offsetFromBottom;
        const xBarX = config.xAxis.marginLeft;
        const xBarWidth = ctx.canvas.width - config.xAxis.marginLeft - config.xAxis.marginRight;
        
        if (xBarHeight > 0 && xBarWidth > 0) {
            ctx.fillRect(xBarX, xBarY, xBarWidth, xBarHeight);
        }
        
        // 繪製Y軸標籤背景條（左側）
        const yBarWidth = config.yAxis.width;
        const yBarX = config.yAxis.offsetFromLeft;
        const yBarY = config.yAxis.marginTop;
        const yBarHeight = ctx.canvas.height - config.yAxis.marginTop - config.yAxis.marginBottom;
        
        if (yBarWidth > 0 && yBarHeight > 0) {
            ctx.fillRect(yBarX, yBarY, yBarWidth, yBarHeight);
        }
    },

    drawAxes: function(ctx, centerX, centerY, color, width) {
        ctx.strokeStyle = color;
        ctx.lineWidth = width;
        ctx.beginPath();
    
        const xAxisY = this.coordinateTransform.mathToWeb(0, 0, centerX, centerY, this.canvasTransform.scale).webY;
        if (xAxisY >= 0 && xAxisY <= ctx.canvas.height) {
            ctx.moveTo(0, xAxisY);
            ctx.lineTo(ctx.canvas.width, xAxisY);
        }
    
        const yAxisX = this.coordinateTransform.mathToWeb(0, 0, centerX, centerY, this.canvasTransform.scale).webX;
        if (yAxisX >= 0 && yAxisX <= ctx.canvas.width) {
            ctx.moveTo(yAxisX, 0);
            ctx.lineTo(yAxisX, ctx.canvas.height);
        }
    
        ctx.stroke();
        
        // console.log('Drawing axes at:', {
        //     xAxisY: xAxisY.toFixed(2),
        //     yAxisX: yAxisX.toFixed(2),
        //     width: width
        // });
    },

    startBrowserZoomMonitoring: function() {
        this.detectBrowserZoom();
        this.lastBrowserZoom = this.browserZoom || 1.0;
      
        this.browserZoomCheckInterval = setInterval(() => {
            const previousZoom = this.browserZoom;
            this.detectBrowserZoom();
            if (Math.abs(this.browserZoom - previousZoom) > 0.01) {
                this.handleBrowserZoomChange();
                this.lastBrowserZoom = this.browserZoom;
            }
        }, 500);
    },

    detectBrowserZoom: function() {
        const screenWidth = screen.width || window.innerWidth || 800;
        const windowWidth = window.outerWidth || window.innerWidth || 800;
        const deviceRatio = window.devicePixelRatio || 1;
        const baseWidth = window.screen.availWidth || window.innerWidth || 800;
        const currentWidth = window.innerWidth || window.outerWidth || 800;
      
        let zoom = 1;
        if (window.chrome && window.chrome.runtime) {
            zoom = window.outerWidth && window.innerWidth ? Math.round((window.outerWidth / window.innerWidth) * 100) / 100 : 1;
        } else {
            zoom = deviceRatio;
        }
      
        this.browserZoom = isNaN(zoom) || zoom <= 0 ? 1.0 : Math.max(0.25, Math.min(5.0, zoom));
    },

    handleBrowserZoomChange: function() {
        this.calculateMaxScale();
        if (this.element) {
            this.adaptToBrowserZoomChange();
        }
      
        requestAnimationFrame(() => {
            resizeCanvases();
            updateScaleDisplay();
        });
    },

    adaptToBrowserZoomChange: function() {
        if (!this.element) return;
        const realWidth = window.innerWidth || 800;
        const realHeight = window.innerHeight || 600;
        this.currentScale = 1.0;
        this.element.style.width = realWidth + 'px';
        this.element.style.height = realHeight + 'px';
        this.elementRect.width = realWidth;
        this.elementRect.height = realHeight;
        this.currentPos.x = 0;
        this.currentPos.y = 0;
        this.element.style.transform = 'translate3d(0px, 0px, 0)';
      
        this.canvasTransform.x = realWidth / 2;
        this.canvasTransform.y = realHeight / 2;
      
        this.updateBoundaries();
        this.constrainPosition();
      
        requestAnimationFrame(() => {
            resizeCanvases();
            this.drawGrid();
        });
    },

    applyScaleWithBrowserZoom: function(newScale, mouseX = null, mouseY = null) {
        if (!this.element) return;
      
        this.currentScale = isNaN(newScale) ? 1.0 : newScale;
      
        const baseWidth = window.innerWidth || 800;
        const baseHeight = window.innerHeight || 600;
      
        const zoomFactor = 1 / (this.browserZoom || 1.0);
      
        let newWidth = Math.max(this.minWidth * zoomFactor, baseWidth * this.currentScale);
        let newHeight = Math.max(this.minHeight * zoomFactor, baseHeight * this.currentScale);
      
        const maxWidthAdjusted = this.maxWidth * zoomFactor;
        const maxHeightAdjusted = this.maxHeight * zoomFactor;
      
        if (maxWidthAdjusted && newWidth > maxWidthAdjusted) {
            newWidth = maxWidthAdjusted;
            this.currentScale = Math.min(this.currentScale, maxWidthAdjusted / baseWidth);
        }
        if (maxHeightAdjusted && newHeight > maxHeightAdjusted) {
            newHeight = maxHeightAdjusted;
            this.currentScale = Math.min(this.currentScale, maxHeightAdjusted / baseHeight);
        }
      
        this.currentPos.x = isNaN(this.currentPos.x) ? 0 : this.currentPos.x;
        this.currentPos.y = isNaN(this.currentPos.y) ? 0 : this.currentPos.y;
      
        this.element.style.width = newWidth + 'px';
        this.element.style.height = newHeight + 'px';
      
        this.elementRect.width = newWidth;
        this.elementRect.height = newHeight;
        this.updateBoundaries();
        this.constrainPosition();
      
        this.element.style.transform = `translate3d(${this.currentPos.x}px, ${this.currentPos.y}px, 0)`;
      
        requestAnimationFrame(() => {
            resizeCanvases();
            this.drawGrid();
        });
    },

    saveCurrentState: function() {
        this.savedState = {
            scale: this.currentScale,
            position: { x: this.currentPos.x, y: this.currentPos.y },
            windowSize: { width: window.innerWidth || 800, height: window.innerHeight || 600 },
            canvasTransform: {
                x: this.canvasTransform.x,
                y: this.canvasTransform.y,
                scale: this.canvasTransform.scale
            }
        };
        localStorage.setItem('ZoneFullMap_ViewState', JSON.stringify(this.savedState));
    },

    loadSavedState: function() {
        try {
            const saved = localStorage.getItem('ZoneFullMap_ViewState');
            if (saved) {
                this.savedState = JSON.parse(saved);
                const screenChanged = (
                    this.savedState.windowSize.width !== (window.innerWidth || 800) ||
                    this.savedState.windowSize.height !== (window.innerHeight || 600)
                );
                if (!screenChanged) {
                    this.restoreState();
                } else {
                    this.adaptStateToNewScreen();
                }
            }
        } catch (e) {
            console.warn('Failed to load ZoneFullMap view state:', e);
        }
    },

    restoreState: function() {
        const scale = isNaN(this.savedState.scale) ? 1.0 : this.savedState.scale;
        this.currentScale = Math.max(this.minScale, Math.min(this.maxScale, scale));
        this.applyScaleWithBrowserZoom(this.currentScale);
      
        const zoomFactor = 1 / (this.browserZoom || 1.0);
        this.currentPos.x = isNaN(this.savedState.position.x) ? 0 : this.savedState.position.x * zoomFactor;
        this.currentPos.y = isNaN(this.savedState.position.y) ? 0 : this.savedState.position.y * zoomFactor;
        this.constrainPosition();
        this.element.style.transform = `translate3d(${this.currentPos.x}px, ${this.currentPos.y}px, 0)`;
      
        if (this.savedState.canvasTransform) {
            this.canvasTransform = {
                x: isNaN(this.savedState.canvasTransform.x) ? this.gridCanvas.width / 2 : this.savedState.canvasTransform.x,
                y: isNaN(this.savedState.canvasTransform.y) ? this.gridCanvas.height / 2 : this.savedState.canvasTransform.y,
                scale: isNaN(this.savedState.canvasTransform.scale) ? 1.0 : this.savedState.canvasTransform.scale
            };
        }
    },

    adaptStateToNewScreen: function() {
        const oldScreenW = this.savedState.windowSize.width || window.innerWidth || 800;
        const oldScreenH = this.savedState.windowSize.height || window.innerHeight || 600;
        const newScreenW = window.innerWidth || 800;
        const newScreenH = window.innerHeight || 600;
      
        const screenScaleW = newScreenW / oldScreenW;
        const screenScaleH = newScreenH / oldScreenH;
        const screenScale = Math.min(screenScaleW, screenScaleH);
      
        const adaptedScale = (this.savedState.scale || 1.0) * screenScale;
        this.currentScale = Math.max(this.minScale, Math.min(this.maxScale, adaptedScale));
      
        this.applyScaleWithBrowserZoom(this.currentScale);
      
        if (this.savedState.canvasTransform) {
            this.canvasTransform.scale = this.savedState.canvasTransform.scale * screenScale;
            this.canvasTransform.x *= screenScaleW;
            this.canvasTransform.y *= screenScaleH;
        }
    },

    calculateMaxScale: function() {
        const zoomFactor = 1 / (this.browserZoom || 1.0);
        const adjustedMaxWidth = this.maxWidth * zoomFactor;
        const adjustedMaxHeight = this.maxHeight * zoomFactor;
      
        const screenRatioW = adjustedMaxWidth / (window.innerWidth || 800);
        const screenRatioH = adjustedMaxHeight / (window.innerHeight || 600);
      
        this.maxScale = isNaN(screenRatioW) || isNaN(screenRatioH) ? 2.0 : Math.min(screenRatioW, screenRatioH, 2.0);
    },

    updateBoundaries: function() {
        const zoomFactor = 1 / (this.browserZoom || 1.0);
        this.boundaries.minX = -(this.elementRect.width - (window.innerWidth || 800) * zoomFactor);
        this.boundaries.minY = -(this.elementRect.height - (window.innerHeight || 600) * zoomFactor);
        this.boundaries.maxX = (window.innerWidth || 800) * zoomFactor;
        this.boundaries.maxY = (window.innerHeight || 600) * zoomFactor;
    },

    constrainPosition: function() {
        this.currentPos.x = Math.max(this.boundaries.minX, Math.min(this.boundaries.maxX, this.currentPos.x));
        this.currentPos.y = Math.max(this.boundaries.minY, Math.min(this.boundaries.maxY, this.currentPos.y));
    },

    toggleGrid: function(enabled) {
        this.gridConfig.enabled = enabled;
        if (enabled) {
            this.drawGrid();
        } else if (this.gridContext) {
            this.gridContext.clearRect(0, 0, this.gridCanvas.width, this.gridCanvas.height);
        }
    },

    updateGridSpacing: function(type, value) {
        const numValue = parseInt(value);
        if (type === 'major') {
            this.gridConfig.majorSpacing = numValue;
            document.getElementById('major_spacing_value').textContent = numValue;
        } else if (type === 'minor') {
            this.gridConfig.minorSpacing = numValue;
            document.getElementById('minor_spacing_value').textContent = numValue;
        }
      
        if (this.gridConfig.enabled) {
            this.drawGrid();
        }
    },

    setupWheelZoom: function() {
        const canvasContainer = document.querySelector('.ZoneFullMap_canvas_td');
        if (!canvasContainer) return;
      
        canvasContainer.removeEventListener('wheel', this._onWheel);
        canvasContainer.addEventListener('wheel', this._onWheel.bind(this), {
            passive: false,
            capture: true
        });
        //console.log('Wheel zoom event listener attached to canvas container');
    },

    _onWheel: function(e) {
        //console.log('[gDragSystem] _onWheel - START');
        
        ZoneFullMapRenderController.startMoving();
        
        if (this._wheelStopTimeout) {
            clearTimeout(this._wheelStopTimeout);
        }
        this._wheelStopTimeout = setTimeout(() => {
            //console.log('[gDragSystem] _onWheel timeout - stopping');
            ZoneFullMapRenderController.stopMoving();
        }, 100);

        const now = Date.now();
        if (now - this.lastRedrawTime < 32) {
            //console.log('[gDragSystem] _onWheel - SKIP (too soon)');
            return;
        }
        this.lastRedrawTime = now;

        if (e.ctrlKey || e.metaKey) {
            //console.log('[gDragSystem] _onWheel - SKIP (ctrl/meta key)');
            return;
        }

        e.preventDefault();
        e.stopPropagation();

        const zoomSpeed = this.scaleStep;
        const oldScale = this.canvasTransform.scale;
        const newScale = e.deltaY > 0 ? oldScale * (1 - zoomSpeed) : oldScale * (1 + zoomSpeed);
        const clampedScale = Math.max(this.minScale, Math.min(this.maxScale, newScale));
        
        if (Math.abs(clampedScale - oldScale) < 0.001) {
            //console.log('[gDragSystem] _onWheel - SKIP (scale change too small)');
            return;
        }

        //console.log('[gDragSystem] _onWheel - scale change:', oldScale.toFixed(3), '->', clampedScale.toFixed(3));

        const containerRect = e.currentTarget.getBoundingClientRect();
        const mouseX = e.clientX - containerRect.left;
        const mouseY = e.clientY - containerRect.top;
        
        const scaleRatio = clampedScale / oldScale;
        
        this.canvasTransform.scale = clampedScale;
        this.canvasTransform.x = mouseX + (this.canvasTransform.x - mouseX) * scaleRatio;
        this.canvasTransform.y = mouseY + (this.canvasTransform.y - mouseY) * scaleRatio;

        if (this._wheelRedrawFrame) {
            cancelAnimationFrame(this._wheelRedrawFrame);
        }
        
        const currentTransform = {
            x: this.canvasTransform.x,
            y: this.canvasTransform.y,
            scale: this.canvasTransform.scale
        };
        
        this._wheelRedrawFrame = requestAnimationFrame(() => {
            //console.log('[gDragSystem] _onWheel RAF - redrawing');
            this._performGridRedraw();
            
            if (typeof ZoneFullMap_updateDevicePositions === 'function') {
                ZoneFullMap_updateDevicePositions(currentTransform);
            }
            
            if (typeof ZoneFullMap_redrawQuadrilaterals === 'function') {
                ZoneFullMap_redrawQuadrilaterals(currentTransform);
            }
            
            if (typeof gSelectedDevice !== 'undefined' && gSelectedDevice && 
                typeof ZoneFullMap_highlightSelectedDevice === 'function') {
                ZoneFullMap_highlightSelectedDevice(currentTransform);
            }
            
            this._wheelRedrawPending = false;
        });

        if (this._saveStateTimeout) {
            clearTimeout(this._saveStateTimeout);
        }

        this._saveStateTimeout = setTimeout(() => {
            this.saveCurrentState();
        }, 200);
    },

    validateTransform: function() {
        // 驗證並修正座標值
        if (isNaN(this.canvasTransform.x) || !isFinite(this.canvasTransform.x)) {
            this.canvasTransform.x = (this.gridCanvas?.width || window.innerWidth) / 2;
        }
        if (isNaN(this.canvasTransform.y) || !isFinite(this.canvasTransform.y)) {
            this.canvasTransform.y = (this.gridCanvas?.height || window.innerHeight) / 2;
        }
        if (isNaN(this.canvasTransform.scale) || !isFinite(this.canvasTransform.scale) || 
            this.canvasTransform.scale <= 0) {
            this.canvasTransform.scale = 1.5;
        }
        
        return {
            x: this.canvasTransform.x,
            y: this.canvasTransform.y,
            scale: this.canvasTransform.scale
        };
    },

    setupCanvasDrag: function() {
        const canvasContainer = document.querySelector('.ZoneFullMap_canvas_td');
        if (!canvasContainer) return;
      
        canvasContainer.removeEventListener('mousedown', this._onCanvasMouseDown);
        canvasContainer.addEventListener('mousedown', this._onCanvasMouseDown.bind(this));
        //console.log('Canvas drag event listener attached to canvas container');
    },

    _onCanvasMouseDown: function(e) {

        //console.log('[gDragSystem] _onCanvasMouseDown');

        if (typeof gZoneFullMap_DrawingMode !== 'undefined' && gZoneFullMap_DrawingMode) {
            //console.log('Canvas drag blocked - in drawing mode');
            return; // 在繪畫模式下阻止畫布拖拽
        }

        const containerRect = e.currentTarget.getBoundingClientRect();
        const isInCanvas = e.clientX >= containerRect.left && e.clientX <= containerRect.right &&
                          e.clientY >= containerRect.top && e.clientY <= containerRect.bottom;
    
        // console.log('Canvas mouse down:', {
        //     button: e.button,
        //     clientX: e.clientX,
        //     clientY: e.clientY,
        //     isInCanvas,
        //     containerRect: {
        //         left: containerRect.left,
        //         top: containerRect.top,
        //         width: containerRect.width,
        //         height: containerRect.height
        //     }
        // });
    
        if (!isInCanvas) {
            //console.log('Mouse not in canvas area, ignoring event');
            return;
        }
    
        if (e.target.classList.contains('ZoneFullMap_close_popup_btn') ||
            e.target.classList.contains('resize-handle') ||
            e.target.tagName === 'BUTTON' ||
            e.target.tagName === 'INPUT' ||
            e.target.tagName === 'SELECT' ||
            e.target.tagName === 'TEXTAREA' ||
            e.target.closest('.ZoneFullMap_calibration_view') ||
            e.target.closest('.ZoneFullMap_calibration_controls') ||
            e.target.closest('.ZoneFullMap_info_container')) {
            //console.log('Clicked on control element, ignoring drag');
            return;
        }
    
        this.isCanvasDragging = true;
        this.hasStarted = false;
    
        this.mouseStart.x = e.clientX - containerRect.left;
        this.mouseStart.y = e.clientY - containerRect.top;
        this.mouseCurrent.x = this.mouseStart.x;
        this.mouseCurrent.y = this.mouseStart.y;
    
        // console.log('Starting canvas drag:', {
        //     startX: this.mouseStart.x,
        //     startY: this.mouseStart.y,
        //     currentTransform: {
        //         x: this.canvasTransform.x,
        //         y: this.canvasTransform.y,
        //         scale: this.canvasTransform.scale
        //     }
        // });
    
        document.addEventListener('mousemove', this._onCanvasMouseMove.bind(this), { passive: true, capture: true });
        document.addEventListener('mouseup', this._onCanvasMouseUp.bind(this), { passive: true, capture: true });
    
        e.preventDefault();
        e.stopPropagation();
    },    

    _onCanvasMouseMove: function(e) {
        if (!this.isCanvasDragging) return;
    
        const now = Date.now();
        if (now - this.lastRedrawTime < 16) return;
        this.lastRedrawTime = now;
    
        ZoneFullMapRenderController.startMoving();
    
        const containerRect = document.querySelector('.ZoneFullMap_canvas_td').getBoundingClientRect();
        this.mouseCurrent.x = e.clientX - containerRect.left;
        this.mouseCurrent.y = e.clientY - containerRect.top;
    
        if (!this.hasStarted) {
            const dx = this.mouseCurrent.x - this.mouseStart.x;
            const dy = this.mouseCurrent.y - this.mouseStart.y;
            const distance = Math.sqrt(dx * dx + dy * dy);
            
            if (distance > this.dragThreshold) {
                this.hasStarted = true;
            }
            return;
        }
    
        const dx = this.mouseCurrent.x - this.mouseStart.x;
        const dy = this.mouseCurrent.y - this.mouseStart.y;
        
        this.canvasTransform.x += dx;
        this.canvasTransform.y += dy;
    
        this.mouseStart.x = this.mouseCurrent.x;
        this.mouseStart.y = this.mouseCurrent.y;
    
        if (!this._moveRedrawPending) {
            this._moveRedrawPending = true;
            ZoneFullMapRenderController.requestGridRedraw();
            this._moveRedrawPending = false;
        }
    },    

    _onCanvasMouseUp: function() {
        //console.log('[gDragSystem] _onCanvasMouseUp');
        this.isCanvasDragging = false;
        this.hasStarted = false;
        
        document.removeEventListener('mousemove', this._onCanvasMouseMove, { capture: true });
        document.removeEventListener('mouseup', this._onCanvasMouseUp, { capture: true });
        this.saveCurrentState();

        ZoneFullMapRenderController.stopMoving();
    },

    resetToFullScreen: function(useHomePosition = false) {
        if (!this.element) return;
        
        this.currentScale = 1.0;
        this.currentPos.x = 0;
        this.currentPos.y = 0;
      
        const newWidth = window.innerWidth || 800;
        const newHeight = window.innerHeight || 600;
        this.element.style.width = newWidth + 'px';
        this.element.style.height = newHeight + 'px';
        this.elementRect.width = newWidth;
        this.elementRect.height = newHeight;
        this.element.style.transform = 'translate3d(0px, 0px, 0)';

        if (useHomePosition) {
            this.canvasTransform.x = newWidth / 2;
            this.canvasTransform.y = newHeight * 5 / 10;
            this.canvasTransform.scale = 1.5;
        } else {
            this.canvasTransform.x = newWidth / 2;
            this.canvasTransform.y = newHeight / 2;
            this.canvasTransform.scale = 1.5;
        }
      
        this.updateBoundaries();
        this.constrainPosition();
        this.saveCurrentState();
      
        requestAnimationFrame(() => {
            resizeCanvases();
        });
    },

    start: function(e) {
        if (!e.target.closest('.ZoneFullMap_draggable_title_bar')) return;
        if (!this.element) return;
      
        this.isDragging = true;
        this.isResizing = false;
        this.hasStarted = false;
      
        const rect = this.element.getBoundingClientRect();
      
        if (this.needsBoundaryUpdate) {
            this.updateCachedValues();
        }
      
        this.dragOffset.x = (e.clientX - rect.left) | 0;
        this.dragOffset.y = (e.clientY - rect.top) | 0;
      
        this.mouseStart.x = e.clientX | 0;
        this.mouseStart.y = e.clientY | 0;
        this.mouseCurrent.x = this.mouseStart.x;
        this.mouseCurrent.y = this.mouseStart.y;
      
        this.currentPos.x = rect.left | 0;
        this.currentPos.y = rect.top | 0;
      
        document.removeEventListener('mousemove', this._onMove);
        document.removeEventListener('mouseup', this._onStop);
        document.addEventListener('mousemove', this._onMove, { passive: true, capture: true });
        document.addEventListener('mouseup', this._onStop, { passive: true, capture: true });
      
        e.preventDefault();
        e.stopPropagation();
    },

    _onMove: function(e) {
        if (!this.isDragging && !this.isResizing) return;
      
        this.mouseCurrent.x = e.clientX | 0;
        this.mouseCurrent.y = e.clientY | 0;
      
        if (this.isResizing) {
            this._updateResize();
            return;
        }
      
        if (!this.hasStarted) {
            const dx = this.mouseCurrent.x - this.mouseStart.x;
            const dy = this.mouseCurrent.y - this.mouseStart.y;
            if (dx * dx + dy * dy > this.dragThreshold) {
                this._startDragging();
            }
            return;
        }
      
        this._updatePosition();
    },

    _startDragging: function() {
        this.hasStarted = true;
        if (this.element) {
            this.element.classList.add('dragging');
        }
        this._updatePosition();
    },

    _onStop: function() {
        if (!this.isDragging && !this.isResizing) return;
      
        this.isDragging = false;
        this.isResizing = false;
        this.hasStarted = false;
      
        if (this.element) {
            this.element.classList.remove('dragging', 'resizing');
        }
      
        document.body.style.cursor = '';
      
        document.removeEventListener('mousemove', this._onMove, { capture: true });
        document.removeEventListener('mouseup', this._onStop, { capture: true });
      
        this.needsBoundaryUpdate = true;
      
        this.saveCurrentState();
        requestAnimationFrame(() => {
            resizeCanvases();
        });
    },

    _updatePosition: function() {
        if (!this.element) return;
        let x = this.mouseCurrent.x - this.dragOffset.x;
        let y = this.mouseCurrent.y - this.dragOffset.y;
      
        x = Math.max(this.boundaries.minX, Math.min(this.boundaries.maxX, x));
        y = Math.max(this.boundaries.minY, Math.min(this.boundaries.maxY, y));
      
        this.element.style.transform = `translate3d(${x}px, ${y}px, 0)`;
      
        this.currentPos.x = x;
        this.currentPos.y = y;
      
        requestAnimationFrame(() => {
            resizeCanvases();
            this.drawGrid();
        });
    },

    setupGPUAcceleration: function() {
        if (!this.element) return;
      
        const style = this.element.style;
        style.willChange = 'transform';
        style.backfaceVisibility = 'hidden';
        style.perspective = '1000px';
        style.position = 'fixed';
      
        if (style.top !== '0px' || style.left !== '0px') {
            const rect = this.element.getBoundingClientRect();
            const currentX = rect.left;
            const currentY = rect.top;
          
            style.top = '0px';
            style.left = '0px';
            style.transform = `translate3d(${currentX}px, ${currentY}px, 0)`;
        }
    },

    updateCachedValues: function() {
        if (!this.element) return;
      
        this.windowSize.width = window.innerWidth || 800;
        this.windowSize.height = window.innerHeight || 600;
        this.elementRect.width = this.element.offsetWidth;
        this.elementRect.height = this.element.offsetHeight;
      
        this.updateBoundaries();
        this.calculateMaxScale();
        this.needsBoundaryUpdate = false;
      
        if (!this.canvasTransform.x || !this.canvasTransform.y) {
            this.canvasTransform.x = this.elementRect.width / 2;
            this.canvasTransform.y = this.elementRect.height / 2;
        }
    },

    destroy: function() {
        this.isInitialized = false;
        this._onStop();
      
        if (this._gridUpdateTimeout) {
            clearTimeout(this._gridUpdateTimeout);
            this._gridUpdateTimeout = null;
        }
      
        if (this.browserZoomCheckInterval) {
            clearInterval(this.browserZoomCheckInterval);
            this.browserZoomCheckInterval = null;
        }
      
        if (this.element) {
            this.saveCurrentState();
            this.element.removeEventListener('wheel', this._onWheel);
            this.element.removeEventListener('mousedown', startDragging);
        }
      
        if (this.gridCanvas) {
            this.gridCanvas.removeEventListener('mousedown', this._onCanvasMouseDown);
        }
      
        this.gridCanvas = null;
        this.gridContext = null;

        if (this._batchRedrawTimeout) {
            clearTimeout(this._batchRedrawTimeout);
            this._batchRedrawTimeout = null;
        }
        this._gridRedrawPending = false;
        this._moveRedrawPending = false;
        this._wheelRedrawPending = false;

        const labelCanvas = document.getElementById('ZoneFullMap_xy_canvas_labels');
        if (labelCanvas) {
            const ctx = labelCanvas.getContext('2d');
            if (ctx) {
                ctx.clearRect(0, 0, labelCanvas.width, labelCanvas.height);
            }
        }
        
        this.lastLabelTransform = {
            x: null,
            y: null,
            scale: null
        };
    },

    _performGridRedraw: function() {
        //console.log('[gDragSystem] _performGridRedraw - START');
        //console.trace('[gDragSystem] _performGridRedraw - Called from:'); 
        
        if (!this.gridCanvas || !this.gridContext || !this.gridConfig.enabled) {
            //console.log('[gDragSystem] _performGridRedraw - SKIP (canvas/context/config invalid)');
            return;
        }

        const ctx = this.gridContext;
        const canvas = this.gridCanvas;
        const config = this.gridConfig;
      
        const scale = isNaN(this.canvasTransform.scale) ? 1.0 : this.canvasTransform.scale;
        const centerX = isNaN(this.canvasTransform.x) ? canvas.width / 2 : this.canvasTransform.x;
        const centerY = isNaN(this.canvasTransform.y) ? canvas.height / 2 : this.canvasTransform.y;

        // console.log('[gDragSystem] _performGridRedraw - transform:', {
        //     scale: scale.toFixed(3),
        //     centerX: centerX.toFixed(2),
        //     centerY: centerY.toFixed(2)
        // });

        ctx.clearRect(0, 0, canvas.width, canvas.height);
      
        const majorSpacing = config.majorSpacing * scale;
        const minorSpacing = config.minorSpacing * scale;
      
        const visibleBounds = {
            left: this.coordinateTransform.webToMath(0, 0, centerX, centerY, scale).mathX,
            right: this.coordinateTransform.webToMath(canvas.width, 0, centerX, centerY, scale).mathX,
            top: this.coordinateTransform.webToMath(0, 0, centerX, centerY, scale).mathY,
            bottom: this.coordinateTransform.webToMath(0, canvas.height, centerX, centerY, scale).mathY
        };
      
        const gridRangeX = 10000;
        const gridRangeY = 10000;
        const bounds = {
            left: Math.max(visibleBounds.left, -gridRangeX),
            right: Math.min(visibleBounds.right, gridRangeX),
            top: Math.min(visibleBounds.top, gridRangeY),
            bottom: Math.max(visibleBounds.bottom, -gridRangeY)
        };

        if (this.canvasTransform.scale > 1.2) {
            this.drawGridLines(ctx, centerX, centerY, minorSpacing, config.minorColor, config.minorWidth, bounds);
        }
        this.drawGridLines(ctx, centerX, centerY, majorSpacing, config.majorColor, config.majorWidth, bounds);
        this.drawAxes(ctx, centerX, centerY, config.axisColor, config.axisWidth);
        this.drawOriginPoint(ctx, centerX, centerY, scale);

        if (config.showLabels) {
            let labelCanvas = document.getElementById('ZoneFullMap_xy_canvas_labels');
            let labelContext = labelCanvas ? labelCanvas.getContext('2d') : null;
            
            if (labelCanvas && labelContext) {
                const transformChanged = 
                    !this.lastLabelTransform ||
                    Math.abs(this.lastLabelTransform.x - centerX) > 0.5 ||
                    Math.abs(this.lastLabelTransform.y - centerY) > 0.5 ||
                    Math.abs(this.lastLabelTransform.scale - scale) > 0.01;
                
                if (transformChanged) {
                    //console.log('[gDragSystem] _performGridRedraw - redrawing labels');
                    labelContext.clearRect(0, 0, labelCanvas.width, labelCanvas.height);
                    this.drawGridLabelsImproved(labelContext, centerX, centerY, majorSpacing, scale, bounds);
                    
                    this.lastLabelTransform = {
                        x: centerX,
                        y: centerY,
                        scale: scale
                    };
                } else {
                    //console.log('[gDragSystem] _performGridRedraw - skip labels (no transform change)');
                }
            }
        }
        
        //console.log('[gDragSystem] _performGridRedraw - END');
    },

};

class ZoneFullMap_Quadrilateral {
    constructor(points, deviceId = null) {
        // console.log('ZoneFullMap_Quadrilateral constructor called with:', {
        //     points: points,
        //     deviceId: deviceId,
        //     deviceIdType: typeof deviceId
        // });
        
        this.points = points || [
            { x: 0, y: 0 },
            { x: 100, y: 0 },
            { x: 100, y: 100 },
            { x: 0, y: 100 }
        ];
        
        this.deviceId = deviceId;
        //console.log('Set this.deviceId to:', this.deviceId);
        
        this.id = this.assignNextAvailableId();
        this.isSelected = false;
        this.color = deviceId ? this.getDeviceColor(deviceId) : '#FFD700';
        this.minDistance = 20;
        
        this.imgZoneCancel = new Image();
        this.imgZoneCancel.src = '../images/item_cancel.png';
        this.imgZoneResize = new Image();
        this.imgZoneResize.src = '../images/item_resize.png';
        
        this.initializeColorScheme();
        
        if (deviceId) {
            this.updateRelativePosition();
        }
        
        // console.log('ZoneFullMap_Quadrilateral created:', {
        //     id: this.id,
        //     deviceId: this.deviceId,
        //     colorScheme: this.colorScheme ? this.colorScheme.name : 'none'
        // });
    }

    initializeColorScheme() {
        //console.log('initializeColorScheme called with deviceId:', this.deviceId);
        
        if (this.deviceId) {
            const device = ZoneFullMap_devices.find(d => d.id === this.deviceId);
            this.colorScheme = ZoneFullMap_getMicrophoneColorScheme(this.deviceId);
            this.color = this.colorScheme.primary;
        } else {
            this.colorScheme = {
                name: 'Default Gold',
                primary: '#FFD700',
                light: '#FFEB3B',
                dark: '#F57F17',
                zone_fill: 'rgba(255, 215, 0, 0.3)',
                zone_stroke: '#FFA000',
                highlight: '#FFCA28'
            };
            this.color = this.colorScheme.primary;
            //console.log('Set default gold color scheme for undefined deviceId');
        }
    }

    updateRelativePosition() {
        // const device = ZoneFullMap_devices.find(d => d.id === this.deviceId);
        // if (device && this.points.length > 0) {
        //     const center = this.getCenter();
        //     this.relativeToDevice = {
        //         x: center.x - device.x,
        //         y: center.y - device.y
        //     };
        // }
        const device = ZoneFullMap_devices.find(d => d.id === this.deviceId);
        if (device && this.points.length > 0) {
            const center = this.getCenter();
            
            this.relativeToDevice = {
                x: Math.round((center.x - device.x) * 100) / 100, // 保留2位小數
                y: Math.round((center.y - device.y) * 100) / 100
            };
            
            // console.log('Updated relative position:', {
            //     quadId: this.getDisplayZoneId(),
            //     devicePos: { x: device.x, y: device.y },
            //     center: center,
            //     relativeToDevice: this.relativeToDevice
            // });
        }
    }

    getCenter() {
        const sumX = this.points.reduce((sum, p) => sum + p.x, 0);
        const sumY = this.points.reduce((sum, p) => sum + p.y, 0);
        return {
            x: sumX / this.points.length,
            y: sumY / this.points.length
        };
    }

    syncWithDevice() {
        const device = ZoneFullMap_devices.find(d => d.id === this.deviceId);
        if (device && this.relativeToDevice) 
        {
            const currentCenter = this.getCenter();
            const targetCenter = {
                x: device.x + this.relativeToDevice.x,
                y: device.y + this.relativeToDevice.y
            };
            
            const deltaX = Math.round((targetCenter.x - currentCenter.x) * 100) / 100;
            const deltaY = Math.round((targetCenter.y - currentCenter.y) * 100) / 100;
            
            if (Math.abs(deltaX) > 0.1 || Math.abs(deltaY) > 0.1) {
                this.points.forEach(point => {
                    point.x += deltaX;
                    point.y += deltaY;
                });
                
                // console.log('Synced zone with device:', {
                //     quadId: this.getDisplayZoneId(),
                //     delta: { x: deltaX, y: deltaY }
                // });
            }
        }
    }

    // Check if two line segments intersect
    doLinesIntersect(p1, q1, p2, q2) {
        function onSegment(p, q, r) {
            return q.x <= Math.max(p.x, r.x) && q.x >= Math.min(p.x, r.x) &&
                   q.y <= Math.max(p.y, r.y) && q.y >= Math.min(p.y, r.y);
        }

        function orientation(p, q, r) {
            const val = (q.y - p.y) * (r.x - q.x) - (q.x - p.x) * (r.y - q.y);
            if (val === 0) return 0;
            return (val > 0) ? 1 : 2;
        }

        const o1 = orientation(p1, q1, p2);
        const o2 = orientation(p1, q1, q2);
        const o3 = orientation(p2, q2, p1);
        const o4 = orientation(p2, q2, q1);

        if (o1 !== o2 && o3 !== o4) return true;

        if (o1 === 0 && onSegment(p1, p2, q1)) return true;
        if (o2 === 0 && onSegment(p1, q2, q1)) return true;
        if (o3 === 0 && onSegment(p2, p1, q2)) return true;
        if (o4 === 0 && onSegment(p2, q1, q2)) return true;

        return false;
    }

    hasSelfIntersection(testPoints = null) {
        const points = testPoints || this.points;
        if (points.length < 4) return false;

        for (let i = 0; i < points.length; i++) {
            const currentEdge = {
                start: points[i],
                end: points[(i + 1) % points.length]
            };

            for (let j = i + 2; j < points.length; j++) {
                if (j === points.length - 1 && i === 0) continue;

                const testEdge = {
                    start: points[j],
                    end: points[(j + 1) % points.length]
                };

                if (this.doLinesIntersect(currentEdge.start, currentEdge.end, testEdge.start, testEdge.end)) {
                    return true;
                }
            }
        }
        return false;
    }

    constrainCorner(cornerIndex, screenX, screenY) {
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
        
        let mathX = (screenX - ox) * 100 / scale;
        let mathY = (oy - screenY) * 100 / scale;
        
        mathX = Math.max(-1000, Math.min(1000, mathX));
        mathY = Math.max(-1000, Math.min(1000, mathY));
        
        const testPoints = [...this.points];
        testPoints[cornerIndex] = { x: mathX, y: mathY };
        
        if (this.hasSelfIntersection(testPoints)) {
            return this.findClosestValidPosition(cornerIndex, mathX, mathY);
        }
        
        const prevIndex = (cornerIndex - 1 + this.points.length) % this.points.length;
        const nextIndex = (cornerIndex + 1) % this.points.length;
        
        const distToPrev = Math.sqrt(
            Math.pow(mathX - this.points[prevIndex].x, 2) + 
            Math.pow(mathY - this.points[prevIndex].y, 2)
        );
        
        const distToNext = Math.sqrt(
            Math.pow(mathX - this.points[nextIndex].x, 2) + 
            Math.pow(mathY - this.points[nextIndex].y, 2)
        );
        
        if (distToPrev < this.minDistance || distToNext < this.minDistance) {
            return this.points[cornerIndex];
        }
        
        return { x: mathX, y: mathY };
    }

    findClosestValidPosition(cornerIndex, targetX, targetY) {
        const originalPoint = { ...this.points[cornerIndex] };
        const step = 2;
        let bestPoint = originalPoint;
        let minDistance = Infinity;
        
        for (let dx = -20; dx <= 20; dx += step) {
            for (let dy = -20; dy <= 20; dy += step) {
                const testX = targetX + dx;
                const testY = targetY + dy;
                const testPoints = [...this.points];
                testPoints[cornerIndex] = { x: testX, y: testY };
                
                if (!this.hasSelfIntersection(testPoints)) {
                    const distance = Math.sqrt(
                        Math.pow(testX - targetX, 2) + 
                        Math.pow(testY - targetY, 2)
                    );
                    
                    if (distance < minDistance) {
                        minDistance = distance;
                        bestPoint = { x: testX, y: testY };
                    }
                }
            }
        }
        
        return bestPoint;
    }

    canMoveTo(deltaX, deltaY) {
        const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
        if (!canvas) return true;
        
        const testPoints = this.points.map(point => ({
            x: point.x + deltaX,
            y: point.y + deltaY
        }));
        
        if (this.hasSelfIntersection(testPoints)) {
            return false;
        }
        
        for (const point of testPoints) {
            const screenPos = this.mathToScreen(point.x, point.y);
            if (screenPos.x < 10 || screenPos.x > canvas.width - 10 ||
                screenPos.y < 10 || screenPos.y > canvas.height - 10) {
                return false;
            }
        }
        
        return true;
    }

    assignNextAvailableId() {
        // console.log('assignNextAvailableId called with:', {
        //     thisDeviceId: this.deviceId,
        //     hasDeviceId: !!this.deviceId,
        //     deviceIdType: typeof this.deviceId
        // });
        
        if (!this.deviceId) 
        {
            if (gZoneFullMap_DeletedIds.length > 0) {
                const reusedId = gZoneFullMap_DeletedIds.shift();
                //console.log('Global system: reusing deleted ID:', reusedId);
                return reusedId;
            }
            const maxId = gZoneFullMap_ActiveRectangles.length > 0 ? Math.max(...gZoneFullMap_ActiveRectangles) : 0;
            const newId = maxId + 1;
            //console.log('Global system: assigned new ID:', newId);
            return newId;
        }
        
        const micNumber = ZoneFullMap_getMicrophoneNumber(this.deviceId);
        const zoneId = ZoneFullMap_getNextMicrophoneZoneId(this.deviceId);
        

        const compositeId = micNumber * 100 + zoneId;
        
        // console.log('Microphone system: assigned composite ID:', {
        //     deviceId: this.deviceId,
        //     micNumber,
        //     zoneId,
        //     compositeId
        // });
        
        return compositeId;
    }

    getDeviceColor(deviceId) {
        const colors = ['#FFD700', '#FF6B35', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7'];
        const index = deviceId.charCodeAt(deviceId.length - 1) % colors.length;
        return colors[index];
    }

    mathToScreen(mathX, mathY) {
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
        return {
            x: ox + mathX * scale / 100,
            y: oy - mathY * scale / 100
        };
    }

    screenToMath(screenX, screenY) {
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
        return {
            x: (screenX - ox) * 100 / scale,
            y: (oy - screenY) * 100 / scale
        };
    }

    getDistance(p1, p2) {
        const dx = p1.x - p2.x;
        const dy = p1.y - p2.y;
        return Math.sqrt(dx * dx + dy * dy);
    }

    getDisplayZoneId = function() {
        if (!this.deviceId) {
            return `Z${this.id}`;
        }
        
        const micNumber = ZoneFullMap_getMicrophoneNumber(this.deviceId);
        const zoneNumber = this.id % 100;
        
        return `M${micNumber}-Z${zoneNumber}`;
    };

    draw = function(ctx) {
        if (!this.colorScheme) {
            this.initializeColorScheme();
        }
    
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
        const screenPoints = this.points.map(point => {
            return {
                x: ox + point.x * scale / 100,
                y: oy - point.y * scale / 100
            };
        });
    
        if (screenPoints.length < 3) return;
    
        ctx.save();
    

        const dpr = window.devicePixelRatio || 1;
        const { scale: canvasScale } = gDragSystem.canvasTransform;
        const effectiveDpr = Math.max(0.5, Math.min(2, 1 / canvasScale));
    
        ctx.beginPath();
        ctx.moveTo(screenPoints[0].x, screenPoints[0].y);
        for (let i = 1; i < screenPoints.length; i++) {
            ctx.lineTo(screenPoints[i].x, screenPoints[i].y);
        }
        ctx.closePath();
    
        if (this.isSelected) {
            ctx.fillStyle = this.colorScheme.zone_fill.replace('0.3', '0.5');
            ctx.strokeStyle = this.colorScheme.dark;
            ctx.lineWidth = 3;
        } else {
            ctx.fillStyle = this.colorScheme.zone_fill;
            ctx.strokeStyle = this.colorScheme.zone_stroke;
            ctx.lineWidth = 2;
        }
        ctx.fill();
        ctx.stroke();
    
        // 計算中心點
        const center = {
            x: screenPoints.reduce((sum, p) => sum + p.x, 0) / screenPoints.length,
            y: screenPoints.reduce((sum, p) => sum + p.y, 0) / screenPoints.length
        };
    
        // 添加日誌以驗證中心點
        // console.log('Drawing Zone:', {
        //     quadId: this.getDisplayZoneId(),
        //     center: center,
        //     screenPoints: screenPoints,
        //     mathPoints: this.points
        // });
    
        // 繪製 Zone ID 圓圈
        ctx.fillStyle = 'white';
        ctx.font = `bold ${Math.max(13, 14 / effectiveDpr)}px Arial`;
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
    
        const text = this.getDisplayZoneId();
        const textMetrics = ctx.measureText(text);
        const textWidth = textMetrics.width;
        const textHeight = 14;
    
        ctx.beginPath();
        ctx.fillStyle = this.colorScheme.dark;
        ctx.arc(center.x, center.y, Math.max(12, textWidth / 2 + 8), 0, 2 * Math.PI);
        ctx.fill();
    
        ctx.beginPath();
        ctx.strokeStyle = 'white';
        ctx.lineWidth = 2;
        ctx.arc(center.x, center.y, Math.max(12, textWidth / 2 + 8), 0, 2 * Math.PI);
        ctx.stroke();
    
        ctx.fillStyle = 'white';
        ctx.fillText(text, center.x, center.y);
    
        // 繪製控制項（選中時）
        if (this.isSelected) {
            this.drawZoneControls(ctx, screenPoints, effectiveDpr);
        }
    
        ctx.restore();
    };
    
    
    
    drawZoneControls = function(ctx, screenPoints, dpr) {
        const ICON_SIZE = 16;
        const CIRCLE_RADIUS = ICON_SIZE / 1.5;
        const OFFSET = 20;
    
        if (screenPoints.length > 1) {
            const rightTop = screenPoints[1];
            const cancelX = rightTop.x - OFFSET;
            const cancelY = rightTop.y;
    
            ctx.beginPath();
            ctx.fillStyle = '#F44336';
            ctx.arc(cancelX, cancelY, CIRCLE_RADIUS, 0, Math.PI * 2);
            ctx.fill();
    
            ctx.beginPath();
            ctx.strokeStyle = '#C62828';
            ctx.lineWidth = 2;
            ctx.arc(cancelX, cancelY, CIRCLE_RADIUS, 0, Math.PI * 2);
            ctx.stroke();
    
            ctx.fillStyle = 'white';
            ctx.font = 'bold 14px Arial';
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.fillText('×', cancelX, cancelY);
        }
    
        screenPoints.forEach((point, index) => {
            ctx.beginPath();
            ctx.fillStyle = this.colorScheme.primary;
            ctx.arc(point.x, point.y, CIRCLE_RADIUS, 0, Math.PI * 2);
            ctx.fill();
            
            ctx.beginPath();
            ctx.strokeStyle = this.colorScheme.dark;
            ctx.lineWidth = 2;
            ctx.arc(point.x, point.y, CIRCLE_RADIUS, 0, Math.PI * 2);
            ctx.stroke();
            
            ctx.fillStyle = 'white';
            ctx.fillRect(point.x - 3, point.y - 3, 6, 6);
        });
    };
    


    drawOldStyleControls(ctx, screenPoints, dpr) {
        const ICON_SIZE = 16;
        const CIRCLE_RADIUS = ICON_SIZE / 1.5; // 10.67px
        const OFFSET = 20;

        // === 繪製刪除按鈕（右上角） ===
        if (screenPoints.length > 1) {
            const rightTop = screenPoints[1]; // 假設第二個點是右上角
            const cancelX = rightTop.x - OFFSET;
            const cancelY = rightTop.y;

            // 繪製藍色圓圈背景
            ctx.beginPath();
            ctx.fillStyle = 'rgba(0, 0, 200, 0.4)';
            ctx.arc(cancelX, cancelY, CIRCLE_RADIUS, 0, Math.PI * 2);
            ctx.fill();

            // 繪製刪除圖標
            if (this.imgZoneCancel && this.imgZoneCancel.complete) {
                ctx.drawImage(
                    this.imgZoneCancel, 
                    cancelX - ICON_SIZE / 2, 
                    cancelY - ICON_SIZE / 2, 
                    ICON_SIZE, 
                    ICON_SIZE
                );
            } else {
                // 備用文字圖標
                ctx.fillStyle = 'white';
                ctx.font = '14px Arial';
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                ctx.fillText('×', cancelX, cancelY);
            }
        }

        // === 繪製角點 resize 控制項 ===
        screenPoints.forEach((point, index) => {
            // 繪製藍色圓圈背景
            ctx.beginPath();
            ctx.fillStyle = 'rgba(0, 0, 200, 0.4)';
            ctx.arc(point.x, point.y, CIRCLE_RADIUS, 0, Math.PI * 2);
            ctx.fill();
            
            // 繪製 resize 圖標
            if (this.imgZoneResize && this.imgZoneResize.complete) {
                // 根據角點位置決定是否旋轉圖標
                if (index === 1 || index === 3) {
                    // 右上角和左下角需要旋轉 90 度
                    this.rotateImage(ctx, this.imgZoneResize, point.x, point.y, 90, ICON_SIZE);
                } else {
                    // 左上角和右下角不旋轉
                    ctx.drawImage(
                        this.imgZoneResize, 
                        point.x - ICON_SIZE / 2, 
                        point.y - ICON_SIZE / 2, 
                        ICON_SIZE, 
                        ICON_SIZE
                    );
                }
            } else {
                // 備用圖標 - 小方塊
                ctx.fillStyle = 'white';
                ctx.fillRect(
                    point.x - 3, 
                    point.y - 3, 
                    6, 
                    6
                );
            }
        });
    }

    rotateImage(ctx, image, x, y, angle, size) {
        ctx.save();
        ctx.translate(x, y);
        ctx.rotate(angle * Math.PI / 180);
        ctx.drawImage(image, -size / 2, -size / 2, size, size);
        ctx.restore();
    }

    drawCornerHandles(ctx, screenPoints) {
        screenPoints.forEach((point, index) => {
            ctx.fillStyle = '#FF0000';
            ctx.beginPath();
            ctx.arc(point.x, point.y, 6, 0, 2 * Math.PI);
            ctx.fill();
            
            ctx.fillStyle = '#FFFFFF';
            ctx.font = '10px Arial';
            ctx.textAlign = 'center';
            ctx.fillText((index + 1).toString(), point.x, point.y + 3);
        });
    }

    drawDeleteButton(ctx, screenPoints) {
        if (screenPoints.length > 1) {
            const deleteButtonX = screenPoints[1].x + 15;
            const deleteButtonY = screenPoints[1].y - 15;
            
            ctx.fillStyle = '#FF4444';
            ctx.fillRect(deleteButtonX - 10, deleteButtonY - 10, 20, 20);
            ctx.fillStyle = '#FFFFFF';
            ctx.font = '14px Arial';
            ctx.textAlign = 'center';
            ctx.fillText('×', deleteButtonX, deleteButtonY + 4);
        }
    }

    drawShapeInfo(ctx, screenPoints) {
        if (this.deviceId) {
            const device = ZoneFullMap_devices.find(d => d.id === this.deviceId);
            if (device) {
                const centerX = screenPoints.reduce((sum, p) => sum + p.x, 0) / screenPoints.length;
                const centerY = screenPoints.reduce((sum, p) => sum + p.y, 0) / screenPoints.length;
                
                ctx.fillStyle = '#000';
                ctx.font = '12px Arial';
                ctx.textAlign = 'center';
                
                const area = this.calculateArea();
                ctx.fillText(`${device.name} ${area.toFixed(1)}m²`, centerX, centerY);
            }
        }
    }

    calculateArea() {
        let area = 0;
        for (let i = 0; i < this.points.length; i++) {
            const j = (i + 1) % this.points.length;
            area += this.points[i].x * this.points[j].y;
            area -= this.points[j].x * this.points[i].y;
        }
        area = Math.abs(area) / 2;
        
        const { scale } = gDragSystem.canvasTransform;
        const scaleM = scale / 100;
        return area * scaleM * scaleM / 10000;
    }

    contains(screenX, screenY) {
        const mathPos = this.screenToMath(screenX, screenY);
        
        let inside = false;
        for (let i = 0, j = this.points.length - 1; i < this.points.length; j = i++) {
            const xi = this.points[i].x, yi = this.points[i].y;
            const xj = this.points[j].x, yj = this.points[j].y;
            
            if (((yi > mathPos.y) !== (yj > mathPos.y)) &&
                (mathPos.x < (xj - xi) * (mathPos.y - yi) / (yj - yi) + xi)) {
                inside = !inside;
            }
        }
        return inside;
    }

    isOnCorner(screenX, screenY) {
        if (!this.isSelected) return -1;
        
        const CIRCLE_RADIUS = 16 / 1.5;
        
        for (let i = 0; i < this.points.length; i++) {
            const screenPos = this.mathToScreen(this.points[i].x, this.points[i].y);
            const distance = Math.sqrt(
                Math.pow(screenX - screenPos.x, 2) + 
                Math.pow(screenY - screenPos.y, 2)
            );
            
            if (distance <= CIRCLE_RADIUS) {
                return i;
            }
        }
        return -1;
    }

    isOnDeleteButton(screenX, screenY) {
        if (!this.isSelected || this.points.length < 2) return false;
        
        const screenPos = this.mathToScreen(this.points[1].x, this.points[1].y);
        const OFFSET = 20;
        const CIRCLE_RADIUS = 16 / 1.5;
        
        const deleteButtonX = screenPos.x - OFFSET;
        const deleteButtonY = screenPos.y;
        
        const distance = Math.sqrt(
            Math.pow(screenX - deleteButtonX, 2) + 
            Math.pow(screenY - deleteButtonY, 2)
        );
        
        return distance <= CIRCLE_RADIUS;
    }

    // Constrain corner movement to screen boundaries
    constrainCornerToScreen(cornerIndex, screenX, screenY) {
        const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
        if (!canvas) return { x: screenX, y: screenY };
        
        const margin = 10;
        return {
            x: Math.max(margin, Math.min(canvas.width - margin, screenX)),
            y: Math.max(margin, Math.min(canvas.height - margin, screenY))
        };
    }
}

function resizeCanvases() {
    // console.log('[resizeCanvases] START');
    
    const canvasContainer = document.querySelector('.ZoneFullMap_canvas_td');
    if (!canvasContainer) return;

    const containerRect = canvasContainer.getBoundingClientRect();
    const width = containerRect.width;
    const height = containerRect.height;

    const canvases = canvasContainer.querySelectorAll('canvas');
    canvases.forEach(canvas => {
        if (canvas.width !== width || canvas.height !== height) {
            canvas.width = width;
            canvas.height = height;
        }
    });

    
    requestAnimationFrame(() => {
        if (typeof redrawAllCanvases === 'function') {
            redrawAllCanvases();
        }

        // }
    });
}

function ZoneFullMap_autoCollapseControls() {
    const toggleBtn = document.getElementById('ZoneFullMap_toggle_btn');
    const buttonsContainer = document.getElementById('ZoneFullMap_buttons_container');
    
    if (toggleBtn && buttonsContainer) {
        // Change to collapsed state
        buttonsContainer.style.display = 'none';
        toggleBtn.textContent = '<';
        //console.log('Auto-collapsed controls');
    }
}

function ZoneFullMap_toggleControls() {
    const toggleBtn = document.getElementById('ZoneFullMap_toggle_btn');
    const buttonsContainer = document.getElementById('ZoneFullMap_buttons_container');
    if (buttonsContainer.style.display === 'none' || buttonsContainer.style.display === '') {
        buttonsContainer.style.display = 'block';
        toggleBtn.textContent = '>';
    } else {
        buttonsContainer.style.display = 'none';
        toggleBtn.textContent = '<';
    }
    const microphoneCount = ZoneFullMap_devices.filter(device => device.type === 'microphone').length;
    const cameraCount = ZoneFullMap_devices.filter(device => device.type === 'camera').length;
    const visionCount = ZoneFullMap_devices.filter(device => device.type === 'vision').length;
    const anchorCount = ZoneFullMap_devices.filter(device => device.type === 'anchor').length;
   
    if (microphoneCount >= 4) {
        document.getElementById('ZoneFullMap_Add_Microphone_btn').disabled = true;
    }
   
    if (cameraCount >= 4) {
        document.getElementById('ZoneFullMap_Add_Camera_btn').disabled = true;
    }
   
    if (visionCount >= 4) {
        document.getElementById('ZoneFullMap_Add_BC200_Device_btn').disabled = true;
    }
    if (anchorCount >= 1) {
        document.getElementById('ZoneFullMap_Add_Anchor_Point_btn').disabled = true;
    }
}

function updateScaleDisplay() {
    const scaleDisplay = document.getElementById('current_scale_display');
    const maxScaleDisplay = document.getElementById('max_scale_display');
    const browserZoomDisplay = document.getElementById('browser_zoom_display');
  
    if (scaleDisplay && gDragSystem.currentScale) {
        scaleDisplay.textContent = Math.round(gDragSystem.currentScale * 100) + '%';
    }
  
    if (maxScaleDisplay && gDragSystem.maxScale) {
        maxScaleDisplay.textContent = Math.round(gDragSystem.maxScale * 100) + '%';
    }
  
    if (browserZoomDisplay && gDragSystem.browserZoom) {
        browserZoomDisplay.textContent = Math.round(gDragSystem.browserZoom * 100) + '%';
      
        if (Math.abs(gDragSystem.browserZoom - 1.0) > 0.05) {
            browserZoomDisplay.style.color = '#ff9900';
        } else {
            browserZoomDisplay.style.color = '#ccc';
        }
    }
}

function startDragging(e) {
    if (!e.target.closest('.ZoneFullMap_draggable_title_bar')) return;
    const popup = document.getElementById('Zone_Full_Map_popup_Window');
    if (!popup) return;
  
    gDragSystem.init(popup);
    gDragSystem.start(e);
}

function closeFullMapModWindow() {
    //console.log('Closing ZoneFullMap window...');
    
    const popup = document.getElementById('Zone_Full_Map_popup_Window');
    if (popup) {
        popup.style.display = 'none';
        //console.log('Hidden popup window');
    }
    
    if (gZoneFullMapAnchorAnimationFrame) {
        cancelAnimationFrame(gZoneFullMapAnchorAnimationFrame);
        gZoneFullMapAnchorAnimationFrame = null;
    }
    
    if (ZoneFullMap_animationId) {
        cancelAnimationFrame(ZoneFullMap_animationId);
        ZoneFullMap_animationId = null;
    }
    
    if (gCanvasResizeTimeout) {
        clearTimeout(gCanvasResizeTimeout);
        gCanvasResizeTimeout = null;
    }
    
    if (gDragSystem && gDragSystem.browserZoomCheckInterval) {
        clearInterval(gDragSystem.browserZoomCheckInterval);
        gDragSystem.browserZoomCheckInterval = null;
    }
    
    document.removeEventListener('mousemove', ZoneFullMap_onCanvasMouseMove_WithQuad, { capture: true });
    document.removeEventListener('mouseup', ZoneFullMap_onCanvasMouseUp_WithQuad, { capture: true });
    document.removeEventListener('keydown', ZoneFullMap_onKeyDown);
    document.removeEventListener('keyup', ZoneFullMap_onKeyUp);
    
    if (window._resizeHandler) {
        window.removeEventListener('resize', window._resizeHandler);
        window._resizeHandler = null;
    }
    
    gSelectedDevice = null;
    gIsDraggingDevice = false;
    gZoneFullMap_DrawingMode = false;
    gZoneFullMap_SelectedMicrophone = null;
    gZoneFullMap_IsDrawing = false;
    gZoneFullMap_DraggingQuadrilateral = null;
    gZoneFullMap_DraggingCornerIndex = -1;
    
    const infoBox = document.getElementById('ZoneFullMap_device_info');
    if (infoBox && infoBox.parentNode) {
        infoBox.remove();
    }
    
    const drawingHint = document.getElementById('ZoneFullMap_drawing_hint');
    if (drawingHint && drawingHint.parentNode) {
        drawingHint.remove();
    }
    
    const globalHint = document.getElementById('ZoneFullMap_global_drawing_hint');
    if (globalHint && globalHint.parentNode) {
        globalHint.remove();
    }
    
    if (gDragSystem && typeof gDragSystem.destroy === 'function') {
        gDragSystem.destroy();
    }

    const VoiceTracking = document.getElementById('SetVoiceTrackingCheckboxInput');
    if (VoiceTracking) 
    {
        VoiceTracking.checked = gVoiceTrackingStatus;
    }

    clearZoneFullMapVisionPoints();
    
    ZoneFullMap_closeBC200ViewWindow();
    ZoneFullMap_closePTZWindow();

    setTimeout(function(){
        ZoneFullMap_DeviceSelect();
        ZoneFullMap_MicZoneEdit();
        ZoneFullMap_InfoSetup();
        ZoneFullMap_closeDeviceConfigMap();
    },50);

}

function onZoneFullMapMode(index) {
    const Zone_Full_Map_popup_Window = document.getElementById('Zone_Full_Map_popup_Window');
    if (!Zone_Full_Map_popup_Window) {
        console.error('Popup window not found');
        return;
    }

    sendMessage('GetCameraList'); 

    Zone_Full_Map_popup_Window.innerHTML = '';
  
    Zone_Full_Map_popup_Window.removeEventListener('mousedown', startDragging);
  
    const ZoneFullMap_titleBar = document.createElement('table');
    ZoneFullMap_titleBar.className = 'ZoneFullMap_titleBar ZoneFullMap_draggable_title_bar';
    ZoneFullMap_titleBar.style.cursor = 'move';
  
    const ZoneFullMap_titleBarRow = document.createElement('tr');
    ZoneFullMap_titleBarRow.className = 'ZoneFullMap_titleBarRow';

    const ZoneFullMap_cell1 = document.createElement('td');
    ZoneFullMap_cell1.className = 'ZoneFullMap_title_bar_cell ZoneFullMap_logo_cell';
    const ZoneFullMap_logoImage = document.createElement('img');
    ZoneFullMap_logoImage.id = 'ZoneFullMap_logo_image';
    ZoneFullMap_logoImage.className = 'ZoneFullMap_logo_image';
    ZoneFullMap_logoImage.src = '../imagesaibox/CamConnect.png';
    ZoneFullMap_cell1.appendChild(ZoneFullMap_logoImage);

    const ZoneFullMap_cell2 = document.createElement('td');
    ZoneFullMap_cell2.className = 'ZoneFullMap_title_bar_cell ZoneFullMap_title_cell';
    const ZoneFullMap_mic_xy_title = document.createElement('label');
    ZoneFullMap_mic_xy_title.className = 'Font_Arial_18_bold';
    ZoneFullMap_mic_xy_title.id = 'ZoneFullMap_mic_xy_title';
    ZoneFullMap_mic_xy_title.textContent = window.LanguageManager.getTranslatedText('All_Zone_map');
    ZoneFullMap_mic_xy_title.style.color = '#89cfd8';
    ZoneFullMap_mic_xy_title.style.fontSize = '18px';
    ZoneFullMap_mic_xy_title.style.fontWeight = 'bold';
    ZoneFullMap_cell2.appendChild(ZoneFullMap_mic_xy_title);
    const ZoneFullMap_cell3 = document.createElement('td');
    ZoneFullMap_cell3.className = 'ZoneFullMap_title_bar_cell ZoneFullMap_close_button_cell';
    const ZoneFullMap_closeButton = document.createElement('button');
    ZoneFullMap_closeButton.id = 'ZoneFullMap_close_popup_' + index;
    ZoneFullMap_closeButton.className = 'ZoneFullMap_close_popup_btn';
    ZoneFullMap_closeButton.setAttribute('onclick', 'closeFullMapModWindow(this)');
    ZoneFullMap_cell3.appendChild(ZoneFullMap_closeButton);
    ZoneFullMap_titleBarRow.appendChild(ZoneFullMap_cell1);
    ZoneFullMap_titleBarRow.appendChild(ZoneFullMap_cell2);
    ZoneFullMap_titleBarRow.appendChild(ZoneFullMap_cell3);
    
    ZoneFullMap_titleBarRow.addEventListener('mousedown', function(e) {
        e.stopPropagation();
    });

    ZoneFullMap_titleBar.appendChild(ZoneFullMap_titleBarRow);
    const ZoneFullMap_fixedContainer = document.createElement('div');
    ZoneFullMap_fixedContainer.className = 'zonefullmap_container';
    const ZoneFullMap_canvasContainer = document.createElement('div');
    ZoneFullMap_canvasContainer.className = 'ZoneFullMap_InfoSections_canvasContainer_td';
    const ZoneFullMap_canvas_td = document.createElement('div');
    ZoneFullMap_canvas_td.className = 'ZoneFullMap_canvas_td';

    const canvasConfigs = [
        { id: 'ZoneFullMap_xy_canvas_grid', zIndex: '0' },
        { id: 'ZoneFullMap_xy_canvas_labels', zIndex: '1' },
        { id: 'ZoneFullMap_xy_canvas_mic', zIndex: '2' },
        { id: 'ZoneFullMap_xy_canvas_camera', zIndex: '3' },
        { id: 'ZoneFullMap_xy_canvas_bc200', zIndex: '4' },
        { id: 'ZoneFullMap_xy_canvas_highLight', zIndex: '5' },
        { id: 'ZoneFullMap_xy_canvas_rect', zIndex: '6' },
        { id: 'ZoneFullMap_xy_canvas_anchor', zIndex: '7' },
        { id: 'ZoneFullMap_xy_canvas_visionPoint', zIndex: '8' },
        { id: 'ZoneFullMap_xy_canvas_soundPoint', zIndex: '9' },
        { id: 'ZoneFullMap_xy_canvas_mouse', zIndex: '10' },
    ];

    canvasConfigs.forEach(config => {
        const canvas = document.createElement('canvas');
        canvas.id = config.id;
        canvas.className = 'ZoneFullMap_canvas_border';
        canvas.style.position = 'absolute';
        canvas.style.zIndex = config.zIndex;
        ZoneFullMap_canvas_td.appendChild(canvas);
    });
    ZoneFullMap_canvasContainer.appendChild(ZoneFullMap_canvas_td);
    ZoneFullMap_fixedContainer.appendChild(ZoneFullMap_canvasContainer);
    
    const ZoneFullMap_controls1 = document.createElement('div');
    ZoneFullMap_controls1.className = 'ZoneFullMap_calibration_view';
    ZoneFullMap_controls1.innerHTML = `
        <button class="ZoneFullMap_control_btn" onclick="ZoneFullMap_goHome()">Home</button>
        <button class="ZoneFullMap_clear_all_btn" onclick="ZoneFullMap_clearAllDevices()">Clear All</button>
    `;
    //<button id="ZoneFullMap_startCalibration_btn" class="ZoneFullMap_startCalibration_btn" onclick="ZoneFullMap_CameraCalibration()">Camera Calibration</button>

    const ZoneFullMap_controls2 = document.createElement('div');
    ZoneFullMap_controls2.className = 'ZoneFullMap_calibration_controls';
    ZoneFullMap_controls2.innerHTML = `
        <span class="ZoneFullMap_device_control_span" style="display: block; margin-bottom: 10px;">Add Device</span>
        <button id="ZoneFullMap_toggle_btn" class="ZoneFullMap_toggle_btn" onclick="ZoneFullMap_toggleControls()" style="padding: 5px 10px; margin-bottom: 10px;"><</button>
        <div id="ZoneFullMap_buttons_container" style="display: none;">
            <button id="ZoneFullMap_Add_Microphone_btn" 
                    class="ZoneFullMap_Add_Microphone_btn ZoneFullMap_draggable_device_btn" 
                    data-device-type="microphone">MIC.</button>
            <button id="ZoneFullMap_Add_Camera_btn" 
                    class="ZoneFullMap_Add_Camera_btn ZoneFullMap_draggable_device_btn" 
                    data-device-type="camera">CAM.</button>

        </div>
    `;

    // <button id="ZoneFullMap_Add_BC200_Device_btn" 
    // class="ZoneFullMap_BC200_Device_btn ZoneFullMap_draggable_device_btn" 
    // data-device-type="vision">BC-200</button>

    // <button id="ZoneFullMap_Add_Anchor_Point_btn" 
    // class="ZoneFullMap_Add_Anchor_Point_btn ZoneFullMap_draggable_device_btn" 
    // data-device-type="anchor">Anchor</button>

    const ZoneFullMap_controls3 = document.createElement('div');
    ZoneFullMap_controls3.className = 'ZoneFullMap_DrawZone_controls';
    ZoneFullMap_controls3.innerHTML = `
        <button id="ZoneFullMapEditModeBtn"    class="ZoneFullMap_Zone_draw_btn"            onclick="ZoneFullMap_toggleEditMode()" >Map Edit Mode</button>
        <button id="ZoneFullMapDataModeBtn"    class="ZoneFullMap_Zone_show_all_data_btn"   onclick="ZoneFullMap_toggleDataMode()" >Single Mic Data Visible</button>
        <button id="ZoneFullMapApplyBtn"       class="ZoneFullMap_Zone_apply_btn"           onclick="ZoneFullMap_applyZoneChanges()">Zone Change Apply</button>
        <button id="ZoneFullMapZoneDisplayBtn" class="ZoneFullMap_Zone_display_btn active"  onclick="ZoneFullMap_applyZoneDisplayChanges()">Display Zone</button>
    `;

    const ZoneFullMap_controls4 = document.createElement('div');
    ZoneFullMap_controls4.className = 'ZoneFullMap_MicSetting_controls';
    ZoneFullMap_controls4.innerHTML = `
        <button id="ZoneFullMapDeviceSelectBtn"    class="ZoneFullMap_Device_Select_btn"            onclick="ZoneFullMap_DeviceSelect()" >
        <img class="deviceImg" />
        </button>
        <button id="ZoneFullMapMicZoneEditBtn"    class="ZoneFullMapMicZoneEditBtn"   onclick="ZoneFullMap_MicZoneEdit()" >
        <img class="editImg" />
        </button>
        <button id="ZoneFullMapInfoSetupBtn"    class="ZoneFullMapMicZoneDisplayBtn"   onclick="ZoneFullMap_InfoSetup()" >
        <img class="aboutImg" />
        </button>
    `;
    //14mode
    // <button id="ZoneFullMapDeviceCalibrationBtn"    class="ZoneFullMapDeviceCalibrationBtn"   onclick="ZoneFullMap_openDeviceConfigMap()" >
    // <img class="systemImg" />
    // </button>

    const ZoneFullMap_panel = document.createElement('div');
    ZoneFullMap_panel.id = 'ZoneFullMap_trigger_info_panel';
    ZoneFullMap_panel.className = 'ZoneFullMap_info_container';
    ZoneFullMap_panel.innerHTML = `
        <div class="ZoneFullMap_info_wrapper">
            <div class="ZoneFullMap_info_item" style="margin-bottom: 15px;">
                <label class="ZoneFullMap_info_title Font_Arial_20_bold" style="margin-top: 12px; display: block;">${window.LanguageManager.getTranslatedText("Label_SetVoiceTracking")}</label>
                <label class="switch_onoff ZoneFullMap_switch" style="margin-top: 10px;">
                    <input type="checkbox" id="ZoneFullMapVoiceTrackingCheckbox" onchange="ZoneFullMapVoiceTrackingChange(this)">
                    <span class="slider_onoff"></span>
                </label>
            </div>
            <div class="ZoneFullMap_info_item" style="margin-bottom: 10px; display: flex; justify-content: space-between;">
                <span class="ZoneFullMap_info_title">${window.LanguageManager.getTranslatedText("Microphone_Select")}</span>
                <span id="ZoneFullMap_mic_index_value" class="ZoneFullMap_info_value Font_Arial_20_bold">${window.LanguageManager.getTranslatedText("null")}</span>
            </div>

            <div class="ZoneFullMap_info_item" style="display: flex; justify-content: space-between;">
                <span class="ZoneFullMap_info_title">${window.LanguageManager.getTranslatedText("Select_Camera")}</span>
                <span id="ZoneFullMap_camera_value" class="ZoneFullMap_info_value Font_Arial_20_bold">${window.LanguageManager.getTranslatedText("null")}</span>
            </div>

            <div class="ZoneFullMap_info_item" style="display: flex; justify-content: space-between;">
                <span class="ZoneFullMap_info_title">${window.LanguageManager.getTranslatedText("Select_Camera_Preset")}</span>
                <span id="ZoneFullMap_camera_preset_value" class="ZoneFullMap_info_value Font_Arial_20_bold">${window.LanguageManager.getTranslatedText("null")}</span>
            </div>
        </div>
    `;

//     <div class="ZoneFullMap_info_item" style="margin-bottom: 10px; display: flex; justify-content: space-between;">
//     <span class="ZoneFullMap_info_title">${window.LanguageManager.getTranslatedText("Label_BC200CameraView_Title")}</span>
//     <span id="ZoneFullMap_model_value" class="ZoneFullMap_info_value Font_Arial_20_bold">${window.LanguageManager.getTranslatedText("null")}</span>
// </div>

    Zone_Full_Map_popup_Window.appendChild(ZoneFullMap_titleBar);
    ZoneFullMap_fixedContainer.appendChild(ZoneFullMap_controls1);
    ZoneFullMap_fixedContainer.appendChild(ZoneFullMap_controls2);
    // ZoneFullMap_fixedContainer.appendChild(ZoneFullMap_controls3);
    ZoneFullMap_fixedContainer.appendChild(ZoneFullMap_controls4);
    ZoneFullMap_fixedContainer.appendChild(ZoneFullMap_panel);
    Zone_Full_Map_popup_Window.appendChild(ZoneFullMap_fixedContainer);
  
    const initialWidth = window.innerWidth || 800;
    const initialHeight = window.innerHeight || 600;
    const initialX = 0;
    const initialY = 0;
  
    Zone_Full_Map_popup_Window.style.width = initialWidth + 'px';
    Zone_Full_Map_popup_Window.style.height = initialHeight + 'px';
    Zone_Full_Map_popup_Window.style.top = '0px';
    Zone_Full_Map_popup_Window.style.left = '0px';
    Zone_Full_Map_popup_Window.style.transform = `translate3d(${initialX}px, ${initialY}px, 0)`;
    Zone_Full_Map_popup_Window.style.display = 'block';
  
    Zone_Full_Map_popup_Window.addEventListener('mousedown', startDragging);
  
    gDragSystem.init(Zone_Full_Map_popup_Window, true, true); 

    gZoneFullMap_GlobalDrawingMode = false;
    ZoneFullMap_exitDrawingMode();
    gSelectedDevice = null;
    ZoneFullMap_clearSelection();

    ZoneFullMap_initDeviceDragging();
    ZoneFullMap_initialize();
    ZoneFullMap_initializeFromExistingDevices();

    requestAnimationFrame(() => {

        ZoneFullMap_BackgroundManager.initBackgroundCanvas();
        ZoneFullMap_addBackgroundControls();
        

        setTimeout(() => {
            
            if (gVoiceTrackingStatus) {
                document.getElementById("ZoneFullMapVoiceTrackingCheckbox").checked = gVoiceTrackingStatus;
            }
        }, 50);
        
        setTimeout(() => {
            initZoneFullMapVisionPointRenderer();
            ZoneFullMap_setupDeviceButtonDragHandlers();
        }, 150);
    });
    
}

function ZoneFullMapMicZoneDrawMode(checkbox) {
    gZoneFullMap_GlobalDrawingMode = checkbox.checked;
    
    if (gZoneFullMap_GlobalDrawingMode) 
    {
        if (gSelectedDevice && gSelectedDevice.type === 'microphone') {
            ZoneFullMap_enterDrawingMode(gSelectedDevice);
        } 
        else 
        {
            ZoneFullMap_showGlobalDrawingHint();
        }
    } 
    else 
    {
        gZoneFullMap_GlobalDrawingMode = false;
        ZoneFullMap_exitDrawingMode();
        ZoneFullMap_hideGlobalDrawingHint();
    }
    
    const canvas = document.querySelector('.ZoneFullMap_canvas_td');
    if (canvas) 
    {
        canvas.style.cursor = gZoneFullMap_GlobalDrawingMode ? 'crosshair' : 'default';
    }
}

var gZoneFullMapAllDataShow = false;
var gZoneFullMap_EditModeState = false;
var gZoneFullMap_DataModeState = false;
var gZoneFullMap_DisplayZoneState = true;//default display
var gZoneFullMap_LastDisplayZoneState = true;

function ZoneFullMapMicZoneAllDataShow(checkbox)
{
    gZoneFullMapAllDataShow = checkbox.checked;
}

function ZoneFullMap_toggleEditMode() 
{
    const button = document.getElementById('ZoneFullMapEditModeBtn');
    const displayButton = document.getElementById('ZoneFullMapZoneDisplayBtn');
    if (!button || !displayButton) return;
    
    gZoneFullMap_EditModeState = !gZoneFullMap_EditModeState;
    
    if (gZoneFullMap_EditModeState) 
    {
        button.textContent = '區域編輯模式';
        button.classList.add('active');
        const checkbox = { checked: true };
        ZoneFullMapMicZoneDrawMode(checkbox);

        if(!gZoneFullMap_DisplayZoneState)
        {
            gZoneFullMap_DisplayZoneState = true;
            displayButton.textContent = '顯示麥克風區域';
            displayButton.classList.remove('active');
            displayButton.classList.add('disabled');
            displayButton.disabled = true;
        }
        else
        {            
            displayButton.classList.remove('active');
            displayButton.classList.add('disabled');
            displayButton.disabled = true;
        }
    } 
    else 
    {
        button.textContent = '地圖編輯模式';
        const checkbox = { checked: false };
        button.classList.remove('active');
        ZoneFullMapMicZoneDrawMode(checkbox);

        if(!gZoneFullMap_LastDisplayZoneState)
        {
            ZoneFullMap_applyZoneDisplayChanges();  
        }

        if(gZoneFullMap_LastDisplayZoneState)
        {
            displayButton.classList.add('active');
        }

        displayButton.classList.remove('disabled');
        displayButton.disabled = false;
    }
}


//v1
// function ZoneFullMap_applyZoneDisplayChanges() {
//     if (gZoneFullMap_EditModeState) return;
    
//     gZoneFullMap_DisplayZoneState = !gZoneFullMap_DisplayZoneState;
//     gZoneFullMap_LastDisplayZoneState = gZoneFullMap_DisplayZoneState;
    
//     const button = document.getElementById('ZoneFullMapZoneDisplayBtn');
//     if (button) {
//         button.textContent = gZoneFullMap_DisplayZoneState ? 'Display Zone' : 'Hide Zone';
//         button.classList.toggle('active', gZoneFullMap_DisplayZoneState);
//     }
    
//     // 更新 Info Setup 視窗中的按鈕狀態
//     const infoZoneBtn = document.getElementById('zoneDisplayBtn');
//     if (infoZoneBtn) {
//         infoZoneBtn.classList.toggle('active', gZoneFullMap_DisplayZoneState);
//         const btnText = infoZoneBtn.querySelector('div:first-child');
//         if (btnText) {
//             btnText.textContent = gZoneFullMap_DisplayZoneState ? 'Display Zone' : 'Hide Zone';
//         }
//     }
    
//     ZoneFullMap_redrawQuadrilaterals();
// }

//v1
// function ZoneFullMap_toggleDataMode() {
//     const button = document.getElementById('ZoneFullMapDataModeBtn');
//     if (!button) return;
    
//     gZoneFullMap_DataModeState = !gZoneFullMap_DataModeState;
    
//     if (gZoneFullMap_DataModeState) {
//         button.textContent = 'Full Data Visible';
//         button.classList.add('active');
//         const checkbox = { checked: true };
//         ZoneFullMapMicZoneAllDataShow(checkbox);
//     } else {
//         button.textContent = 'Single Mic Data Visible';
//         const checkbox = { checked: false };
//         button.classList.remove('active');
//         ZoneFullMapMicZoneAllDataShow(checkbox);
//     }
// }

function ZoneFullMap_showGlobalDrawingHint() {
    ZoneFullMap_hideGlobalDrawingHint();
    
    const hint = document.createElement('div');
    hint.id = 'ZoneFullMap_global_drawing_hint';
    hint.style.position = 'fixed';
    hint.style.top = '10px';
    hint.style.left = '50%';
    hint.style.transform = 'translateX(-50%)';
    hint.style.backgroundColor = 'rgba(76, 175, 80, 0.9)';
    hint.style.color = 'white';
    hint.style.padding = '8px 16px';
    hint.style.borderRadius = '5px';
    hint.style.zIndex = '1001';
    hint.style.fontSize = '14px';
    hint.style.fontWeight = 'bold';
    hint.style.boxShadow = '0 2px 10px rgba(0,0,0,0.3)';
    hint.innerHTML = `
        區域繪製模式啟動 - 點擊麥克風選擇並繪製區域
        <br><small>關閉開關以返回畫布拖曳模式</small>
    `;
    
    document.body.appendChild(hint);
}

function ZoneFullMap_hideGlobalDrawingHint() {
    const hint = document.getElementById('ZoneFullMap_global_drawing_hint');
    if (hint && hint.parentNode) {
        hint.remove();
    }
}

function ZoneFullMap_handleDeviceClick(hitDevice, mouseX, mouseY, isRightClick = false) {
    if (gZoneFullMap_GlobalDrawingMode && hitDevice.type === 'microphone') {
        gSelectedDevice = hitDevice;
        if (isRightClick) {
            ZoneFullMap_showDeviceInfo(hitDevice);
        }
        ZoneFullMap_highlightSelectedDevice();
        ZoneFullMap_enterDrawingMode(hitDevice);
        console.log('Entered editing mode for microphone:', hitDevice.name);
        return;
    }
    
    console.log('Hit device:', hitDevice.name);
    gSelectedDevice = hitDevice;
    
    if (isRightClick) {
        ZoneFullMap_showDeviceInfo(hitDevice);
    }
    
    if (!isRightClick) {
        gIsDraggingDevice = true;
        gDragSystem.isCanvasDragging = false;
        gDeviceDragOffset.x = 0;
        gDeviceDragOffset.y = 0;
        
        ZoneFullMap_addMouseEventListeners();
    }
    
    ZoneFullMap_highlightSelectedDevice();
    ZoneFullMap_autoCollapseControls(); 
}


var ZoneFullMap_devices = [];

function ZoneFullMap_goHome() {
    if (!gDragSystem.isInitialized) return;
    
    const canvas = gDragSystem.gridCanvas;
    if (!canvas) return;
    
    gDragSystem.canvasTransform.x = canvas.width / 2;
    gDragSystem.canvasTransform.y = canvas.height * 5 / 10;
    gDragSystem.canvasTransform.scale = 1.5;
    
    gDragSystem.saveCurrentState();
    
    requestAnimationFrame(() => {
        if (typeof redrawAllCanvases === 'function') {
            redrawAllCanvases();
        }
    });
}

// Streamlined redraw function - only essential operations
function ZoneFullMap_fastRedraw() {
    // Single requestAnimationFrame for optimal performance
    requestAnimationFrame(() => {
        // Only redraw what's necessary
        if (gDragSystem.drawGrid) {
            gDragSystem.drawGrid();
        }
        
        // Update devices and zones in single pass
        if (typeof ZoneFullMap_updateDevicePositions === 'function') {
            ZoneFullMap_updateDevicePositions();
        }
        
        if (typeof ZoneFullMap_redrawQuadrilaterals === 'function') {
            ZoneFullMap_redrawQuadrilaterals();
        }
        
        // Highlight selected device if exists
        if (gSelectedDevice && typeof ZoneFullMap_highlightSelectedDevice === 'function') {
            ZoneFullMap_highlightSelectedDevice();
        }
    });
}

// Alternative ultra-fast version if above still has issues
function ZoneFullMap_goHomeUltraFast() {
    if (!gDragSystem.isInitialized) return;
    
    const canvas = gDragSystem.gridCanvas;
    if (!canvas) return;
    
    gDragSystem.canvasTransform.x = canvas.width / 2;
    gDragSystem.canvasTransform.y = canvas.height * 5 / 10;
    gDragSystem.canvasTransform.scale = 1.5;
    gDragSystem.drawGrid();
    ZoneFullMap_updateDevicePositions();
    ZoneFullMap_redrawQuadrilaterals();
    
    if (gSelectedDevice) {
        ZoneFullMap_highlightSelectedDevice();
    }
    
    gDragSystem.saveCurrentState();
}

function ZoneFullMap_forceCompleteRedraw() {
    ZoneFullMap_fastRedraw();
}

function ZoneFullMap_verifyCanvasContent() {
    return true;
}

function ZoneFullMap_emergencyCanvasRecovery() {
    ZoneFullMap_fastRedraw();
}

function ZoneFullMap_safeStartSoundAnimation(selectedDevice) {
    if (!selectedDevice || selectedDevice.type !== 'microphone' || !selectedDevice.id) return;
    
    try {
        const micTabIndex = ZoneFullMap_getMicrophoneNumber(selectedDevice.id) - 1;
        if (micTabIndex >= 0 && typeof startSoundAnimation === 'function') {
            startSoundAnimation(micTabIndex, selectedDevice);
        }
    } catch (error) {

    }
}

function ZoneFullMap_getRandomPosition() {
    const canvas = gDragSystem.gridCanvas;
    if (!canvas) return { x: 0, y: 0 };
    
    const visibleBounds = {
        left: gDragSystem.coordinateTransform.webToMath(0, 0, gDragSystem.canvasTransform.x, gDragSystem.canvasTransform.y, gDragSystem.canvasTransform.scale).mathX,
        right: gDragSystem.coordinateTransform.webToMath(canvas.width, 0, gDragSystem.canvasTransform.x, gDragSystem.canvasTransform.y, gDragSystem.canvasTransform.scale).mathX,
        top: gDragSystem.coordinateTransform.webToMath(0, 0, gDragSystem.canvasTransform.x, gDragSystem.canvasTransform.y, gDragSystem.canvasTransform.scale).mathY,
        bottom: gDragSystem.coordinateTransform.webToMath(0, canvas.height, gDragSystem.canvasTransform.x, gDragSystem.canvasTransform.y, gDragSystem.canvasTransform.scale).mathY
    };
    
    const padding = 100;
    const x = Math.random() * (visibleBounds.right - visibleBounds.left - 2 * padding) + visibleBounds.left + padding;
    const y = Math.random() * (visibleBounds.top - visibleBounds.bottom - 2 * padding) + visibleBounds.bottom + padding;
    
    return { x, y };
}





function ZoneFullMap_drawMicrophoneFallback(ctx, size) 
{
    ctx.fillStyle = '#4CAF50';
    ctx.beginPath();
    ctx.arc(0, 0, size / 2, 0, 2 * Math.PI);
    ctx.fill();
    
    ctx.fillStyle = 'white';
    ctx.beginPath();
    ctx.arc(0, -size / 6, size / 8, 0, 2 * Math.PI);
    ctx.fill();
    
    ctx.fillRect(-size / 16, -size / 6, size / 8, size / 3);
    ctx.fillRect(-size / 12, size / 6, size / 6, size / 12);
}

function ZoneFullMap_preloadMicrophoneImage(device, onLoadCallback) {
    if (!device.imgSrc) 
    {
        device.imageLoaded = false;
        return;
    }
    
    const img = new Image();
    
    img.onload = function() {
        device.imageLoaded = true;
        device._preloadedImage = img;
        
        if (typeof onLoadCallback === 'function') {
            onLoadCallback();
        }
    };
    
    img.onerror = function() {
        device.imageLoaded = false;
        device._preloadedImage = null;
        
        if (typeof onLoadCallback === 'function') {
            onLoadCallback();
        }
    };
    
    img.crossOrigin = 'anonymous';
    img.src = device.imgSrc;
}

function ZoneFullMap_createMicrophone(customPosition = null) {
    const microphoneCount = ZoneFullMap_devices.filter(device => device.type === 'microphone').length;
    
    if (microphoneCount >= 6) 
    {
        return;
    }
    
    const deviceNumber = ZoneFullMap_getNextDeviceId('microphone');
    
    let position;
    if (customPosition) {
        position = customPosition;
    } else {
        position = microphoneCount === 0 ? { x: 0, y: 0 } : ZoneFullMap_getRandomPosition();
    }
    
    const device = {
        id: `mic_${deviceNumber}`,
        type: 'microphone',
        name: `Microphone ${deviceNumber}`,
        x: position.x,
        y: position.y,
        angle: 0,
        status: 'Connected',
        imgSrc: './images/Web_RMCG.png',
        baseWidthM: 0.6,
        canDelete: true,
        groupColor: ZoneFullMap_getMicrophoneColorScheme(`mic_${deviceNumber}`).primary,
        imageLoaded: false,
        imageElement: null,
        connectedTabIndex: undefined,
        displayNumber: deviceNumber,
    };
    
    ZoneFullMap_devices.push(device);
    
    ZoneFullMap_preloadMicrophoneImage(device, () => {
        requestAnimationFrame(() => {
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_redrawQuadrilaterals();
            
            if (gSelectedDevice) {
                ZoneFullMap_highlightSelectedDevice();
            }
            
            if (gDragSystem && gDragSystem.drawGrid) {
                gDragSystem.drawGrid();
            }
        });
    });
    
    if (microphoneCount === 0) {
        gSelectedDevice = device;
        setTimeout(() => {
            ZoneFullMap_highlightSelectedDevice();
            //console.log('Auto-selected first microphone:', device.name);
        }, 100);
    }
    
    const imgEl = document.createElement('img');
    imgEl.className = 'ZoneFullMap_device';
    imgEl.src = device.imgSrc;
    imgEl.setAttribute('data-id', device.id);
    imgEl.setAttribute('data-x', device.x);
    imgEl.setAttribute('data-y', device.y);
    imgEl.style.transformOrigin = 'center';
    imgEl.style.position = 'absolute';
    imgEl.style.zIndex = '3';
    imgEl.style.display = 'none';
    imgEl.style.pointerEvents = 'none';

    const container = document.querySelector('.ZoneFullMap_canvas_td');
    if (container) {
        container.appendChild(imgEl);
    }
    
    device.imageElement = imgEl;
    
    ZoneFullMap_updateButtonStates();
    ZoneFullMap_updateDevicePositions();
    
    // console.log('Created microphone:', {
    //     name: device.name,
    //     id: device.id,
    //     displayNumber: device.displayNumber,
    //     position: 'x=' + device.x + 'm, y=' + device.y + 'm',
    //     isFirstMic: microphoneCount === 0
    // });
    
    return device;
}

function ZoneFullMap_addCamera(customPosition = null) {
    const cameraCount = ZoneFullMap_devices.filter(device => device.type === 'camera').length;
    
    if (cameraCount >= 6) {
        console.log('Maximum number of cameras reached');
        return;
    }

    const deviceNumber = ZoneFullMap_getNextDeviceId('camera');
    const position = customPosition || ZoneFullMap_getRandomPosition();
    
    
    const device = {
        id: `camera_${deviceNumber}`,
        type: 'camera',
        name: `Camera ${deviceNumber}`,
        x: position.x,
        y: position.y,
        angle: 0,
        fovAngle: 0,
        offsetAngle: 0,
        status: 'Connected',
        baseWidthM: 0.4,
        isSelectedCamera: false,
        canDelete: true
    };
    
    ZoneFullMap_devices.push(device);
    ZoneFullMap_updateButtonStates();
    ZoneFullMap_updateDevicePositions();
    
    return device;
}

function ZoneFullMap_addVision(customPosition = null) {
    const visionCount = ZoneFullMap_devices.filter(device => device.type === 'vision').length;
    
    if (visionCount >= 6) {
        console.log('Maximum number of BC-200 devices reached');
        return;
    }
    
    const deviceNumber = ZoneFullMap_getNextDeviceId('vision');
    const position = customPosition || ZoneFullMap_getRandomPosition();
    
    const device = {
        id: `vision_${deviceNumber}`,
        type: 'vision',
        name: `BC-200 ${deviceNumber}`,
        x: position.x,
        y: position.y,
        angle: 0,
        fovAngle: 0,
        offsetAngle: 0,
        offsetX: 0,
        offsetY: 0,
        status: 'Null',
        baseWidthM: 0.5,
        canDelete: true
    };
    
    ZoneFullMap_devices.push(device);
    ZoneFullMap_updateButtonStates();
    ZoneFullMap_updateDevicePositions();
    
    return device;
}

function ZoneFullMap_addAnchorPoint(customPosition = null) {
    const anchorCount = ZoneFullMap_devices.filter(device => device.type === 'anchor').length;
    
    if (anchorCount >= 1) {
        console.log('Maximum number of anchor points reached');
        return;
    }
    
    const deviceNumber = ZoneFullMap_getNextDeviceId('anchor');
    const position = customPosition || ZoneFullMap_getRandomPosition();
    
    
    const device = {
        id: `anchor_${deviceNumber}`,
        type: 'anchor',
        name: `Anchor ${deviceNumber}`,
        x: position.x,
        y: position.y,
        status: 'Active',
        baseWidthM: 0.3,
        canDelete: true
    };
    
    ZoneFullMap_devices.push(device);
    ZoneFullMap_updateButtonStates();
    ZoneFullMap_updateDevicePositions();
    
    return device;
}

function ZoneFullMap_updateButtonStates() {
    const microphoneCount = ZoneFullMap_devices.filter(device => device.type === 'microphone').length;
    const cameraCount = ZoneFullMap_devices.filter(device => device.type === 'camera').length;
    const visionCount = ZoneFullMap_devices.filter(device => device.type === 'vision').length;
    const anchorCount = ZoneFullMap_devices.filter(device => device.type === 'anchor').length;
    
    const micBtn = document.getElementById('ZoneFullMap_Add_Microphone_btn');
    const camBtn = document.getElementById('ZoneFullMap_Add_Camera_btn');
    const bc200Btn = document.getElementById('ZoneFullMap_Add_BC200_Device_btn');
    const anchorBtn = document.getElementById('ZoneFullMap_Add_Anchor_Point_btn');
    
    if (micBtn) micBtn.disabled = microphoneCount >= 6;
    if (camBtn) camBtn.disabled = cameraCount >= 6;
    if (bc200Btn) bc200Btn.disabled = visionCount >= 6;
    if (anchorBtn) anchorBtn.disabled = anchorCount >= 1;
}


function ZoneFullMap_clearAllDevices() {    
    const deviceElements = document.querySelectorAll('.ZoneFullMap_device');
    deviceElements.forEach(el => {
        if (el && el.parentNode) {
            el.remove();
        }
    });
    
    ZoneFullMap_devices.forEach(device => {
        if (device._preloadedImage) {
            device._preloadedImage = null;
        }
        if (device.imageElement && device.imageElement.parentNode) {
            device.imageElement.remove();
        }
    });
    ZoneFullMap_devices.length = 0;
    
    gZoneFullMap_DeviceIdManager = {
        microphone: { usedIds: [], deletedIds: [], nextId: 1 },
        camera: { usedIds: [], deletedIds: [], nextId: 1 },
        vision: { usedIds: [], deletedIds: [], nextId: 1 },
        anchor: { usedIds: [], deletedIds: [], nextId: 1 }
    };
    console.log('Device ID manager reset to initial state');
    
    gZoneFullMap_MicIdToTabMapping = {};
    gZoneFullMap_TabToMicIdMapping = {};
    
    ZoneFullMap_clearAllMicrophoneZoneCounters();
    ZoneFullMap_clearAllQuadrilaterals();
    
    gSelectedDevice = null;
    gIsDraggingDevice = false;
    gZoneFullMap_SelectedMicrophone = null;
    gZoneFullMap_LastSelectedQuad = null;
    gZoneFullMap_DrawingMode = false;
    gZoneFullMap_GlobalDrawingMode = false;
    gZoneFullMap_IsDrawing = false;
    gZoneFullMap_DraggingQuadrilateral = null;
    gZoneFullMap_DraggingCornerIndex = -1;
    gZoneFullMap_CornerDragStart = null;
    
    gAnchorMovementTracker.positions = {};
    gAnchorMovementTracker.isMoving = {};
    gAnchorMovementTracker.lastMovingStates = {};
    
    ZoneFullMap_clearAllHighlights();
    ZoneFullMap_hideDeviceInfo();
    ZoneFullMap_hideDrawingHint();
    ZoneFullMap_hideGlobalDrawingHint();
    ZoneFullMap_hideZoneSettings();
    
    if (typeof ZoneFullMap_clearSoundPoints === 'function') {
        ZoneFullMap_clearSoundPoints();
    }
    
    gSimpleSoundPoints = {};
    gSimpleZoneStates = {};
    
    if (gZoneFullMapAnchorAnimationFrame) {
        cancelAnimationFrame(gZoneFullMapAnchorAnimationFrame);
        gZoneFullMapAnchorAnimationFrame = null;
    }
    
    if (ZoneFullMap_animationId) {
        cancelAnimationFrame(ZoneFullMap_animationId);
        ZoneFullMap_animationId = null;
    }
    
    const canvasIds = [
        'ZoneFullMap_xy_canvas_mic',
        'ZoneFullMap_xy_canvas_highLight', 
        'ZoneFullMap_xy_canvas_rect',
        'ZoneFullMap_xy_canvas_visionPoint',
        'ZoneFullMap_xy_canvas_soundPoint',
        'ZoneFullMap_xy_canvas_anchor',
        'ZoneFullMap_xy_canvas_mouse'
    ];
    
    canvasIds.forEach(canvasId => {
        const canvas = document.getElementById(canvasId);
        if (canvas) {
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
        }
    });
    
    const editModeBtn = document.getElementById('ZoneFullMapEditModeBtn');
    if (editModeBtn) {
        editModeBtn.textContent = '地圖編輯模式';
        editModeBtn.classList.remove('active');
    }
    
    const dataModeBtn = document.getElementById('ZoneFullMapDataModeBtn');
    if (dataModeBtn) {
        dataModeBtn.textContent = '單麥克風資料可見';
        dataModeBtn.classList.remove('active');
    }

    const displayZoneBtn = document.getElementById('ZoneFullMapZoneDisplayBtn');
    if (displayZoneBtn) {
        displayZoneBtn.textContent = '顯示麥克風區域';
        displayZoneBtn.checked = true;
    }
    
    const canvas = document.querySelector('.ZoneFullMap_canvas_td');
    if (canvas) {
        canvas.style.cursor = 'default';
    }
    
    ZoneFullMap_updateButtonStates();
    ZoneFullMap_updateDevicePositions();
    ZoneFullMap_autoCollapseControls();
    
    const infoElements = ['ZoneFullMap_mic_index_value', 'ZoneFullMap_model_value', 'ZoneFullMap_camera_value','ZoneFullMap_camera_preset_value'];
    infoElements.forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            const nullText = window.LanguageManager ? 
                window.LanguageManager.getTranslatedText('null') : 'null';
            element.textContent = nullText;
        }
    });
    
    //console.log('Clear all operation completed successfully with device ID manager reset');
}

function ZoneFullMap_hideZoneSettings() {
    const settingsBox = document.getElementById('ZoneFullMap_zone_settings');
    if (settingsBox && settingsBox.parentNode) {
        settingsBox.remove();
        console.log('Zone settings window hidden');
    }
}

function ZoneFullMap_clearDeviceHighlight() {
    const highlightCanvas = document.getElementById('ZoneFullMap_xy_canvas_highLight');
    if (highlightCanvas) {
        const ctx = highlightCanvas.getContext('2d');
        ctx.clearRect(0, 0, highlightCanvas.width, highlightCanvas.height);
        //console.log('Device highlight cleared');
    }
    
    if (gSelectedDevice && gSelectedDevice._highlightArea) {
        delete gSelectedDevice._highlightArea;
    }
}

function ZoneFullMap_setAnchorStationary(deviceId) {
    const now = Date.now();
    const lastUpdate = gAnchorMovementTracker.positions[deviceId]?.timestamp || 0;
    
    // Only mark as not moving if no updates occurred during stability period
    if (now - lastUpdate >= gAnchorMovementTracker.stabilityTime) {
        const wasMoving = gAnchorMovementTracker.isMoving[deviceId];
        gAnchorMovementTracker.isMoving[deviceId] = false;
        
        if (wasMoving) {
            console.log(`Anchor ${deviceId} became stationary - checking animation restart`);
            // Check if we need to restart animation
            ZoneFullMap_manageAnchorBreathingAnimation();
        }
    }
}

function ZoneFullMap_manageAnchorBreathingAnimation() {
    const anchors = ZoneFullMap_devices.filter(device => device.type === 'anchor');
    
    if (anchors.length === 0) {
        // No anchors exist, stop animation if running
        if (gZoneFullMapAnchorAnimationFrame) {
            ZoneFullMap_stopAnchorBreathingAnimation();
        }
        return;
    }
    
    // Check if any anchors are stationary (not moving)
    const hasStationaryAnchors = anchors.some(device => 
        !gAnchorMovementTracker.isMoving || !gAnchorMovementTracker.isMoving[device.id]
    );
    
    if (hasStationaryAnchors && !gZoneFullMapAnchorAnimationFrame) {
        // Start animation if we have stationary anchors and no animation is running
        ZoneFullMap_startAnchorBreathingAnimation();
    } else if (!hasStationaryAnchors && gZoneFullMapAnchorAnimationFrame) {
        // Stop animation if all anchors are moving
        ZoneFullMap_stopAnchorBreathingAnimation();
    }
}

function ZoneFullMap_drawDeviceAngleInfo(ctx, device, x, y) {
    ctx.save();
    ctx.fillStyle = 'rgba(0, 0, 0, 0.8)';
    ctx.font = '12px Arial';
    ctx.textAlign = 'center';
    
    let infoText = `${Math.round(device.angle || 0)}°`;
    
    if (device.type === 'camera') {
        if (device.fovAngle) infoText += ` | FOV:${device.fovAngle}°`;
        if (device.offsetAngle) infoText += ` | Offset:${device.offsetAngle}°`;
    }
    
    // 背景框
    const textWidth = ctx.measureText(infoText).width;
    ctx.fillStyle = 'rgba(255, 255, 255, 0.9)';
    ctx.fillRect(x - textWidth/2 - 4, y - 8, textWidth + 8, 16);
    
    ctx.fillStyle = 'black';
    ctx.fillText(infoText, x, y + 4);
    
    ctx.restore();
}

let gZoneFullMapAnchorAnimationFrame = null;

function ZoneFullMap_startAnchorBreathingAnimation() {
    if (gZoneFullMapAnchorAnimationFrame) return;

    function animate() {
        const anchors = ZoneFullMap_devices.filter(device => device.type === 'anchor');
        
        const stationaryAnchors = anchors.filter(device => 
            !gAnchorMovementTracker.isMoving || !gAnchorMovementTracker.isMoving[device.id]
        );
        
        if (stationaryAnchors.length === 0) {
            gZoneFullMapAnchorAnimationFrame = null;
            return;
        }

        const anchorCanvas = document.getElementById('ZoneFullMap_xy_canvas_anchor');
        if (!anchorCanvas) {
            gZoneFullMapAnchorAnimationFrame = null;
            return;
        }

        const ctx = anchorCanvas.getContext('2d');
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
        
        ctx.clearRect(0, 0, anchorCanvas.width, anchorCanvas.height);

        anchors.forEach(device => {
            const widthPx = device.baseWidthM * scale * 100;
            const cx = ox + device.x * scale / 100;
            const cy = oy - device.y * scale / 100;

            ctx.beginPath();
            ctx.arc(cx, cy, widthPx / 2, 0, 2 * Math.PI);
            ctx.fillStyle = 'red';
            ctx.fill();
            ctx.closePath();
        });

        stationaryAnchors.forEach(device => {
            const widthPx = device.baseWidthM * scale * 100;
            const cx = ox + device.x * scale / 100;
            const cy = oy - device.y * scale / 100;

            const breathFactor = (Math.sin(Date.now() / 250) + 1) / 2;
            const alpha = breathFactor * 0.7;
            
            ctx.beginPath();
            ctx.arc(cx, cy, widthPx + (breathFactor * 12), 0, 2 * Math.PI);
            ctx.strokeStyle = `rgba(255, 0, 0, ${alpha})`;
            ctx.lineWidth = 2;
            ctx.stroke();
            ctx.closePath();
        });

        gZoneFullMapAnchorAnimationFrame = requestAnimationFrame(animate);
    }

    animate();
}

function ZoneFullMap_stopAnchorBreathingAnimation() {
    if (gZoneFullMapAnchorAnimationFrame) {
        cancelAnimationFrame(gZoneFullMapAnchorAnimationFrame);
        gZoneFullMapAnchorAnimationFrame = null;
    }
}

function redrawAllCanvases() 
{
    // console.log('[redrawAllCanvases] START');
    // console.trace('[redrawAllCanvases] Called from:'); 

    ZoneFullMap_clearAllHighlights();
    
    if (gDragSystem && gDragSystem._performGridRedraw) {
        gDragSystem._performGridRedraw();
    }
    
    if (typeof ZoneFullMap_updateDevicePositions === 'function') {
        ZoneFullMap_updateDevicePositions();
    }
    
    if (typeof ZoneFullMap_redrawQuadrilaterals === 'function') {
        ZoneFullMap_redrawQuadrilaterals();
    }
    
    if (typeof gSelectedDevice !== 'undefined' && gSelectedDevice && 
        typeof ZoneFullMap_highlightSelectedDevice === 'function') {
        ZoneFullMap_highlightSelectedDevice();
    }
    
}



function ZoneFullMap_clearAllHighlights() {
    const highlightCanvas = document.getElementById('ZoneFullMap_xy_canvas_highLight');
    if (highlightCanvas) {
        const ctx = highlightCanvas.getContext('2d');
        ctx.clearRect(0, 0, highlightCanvas.width, highlightCanvas.height);
        //console.log('Cleared all highlights during redraw');
    }
}

function ZoneFullMap_hitTestDevice(mouseX, mouseY, isRightClick = false) {
    if (!ZoneFullMap_devices || ZoneFullMap_devices.length === 0) return null;
    
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const hitDevices = [];
    
    for (let i = ZoneFullMap_devices.length - 1; i >= 0; i--) {
        const device = ZoneFullMap_devices[i];

        if (gZoneFullMap_deviceVisibility[device.id] === false) {
            continue;
        }

        const widthPx = device.baseWidthM * scale * 100;
        const cx = ox + device.x * scale / 100;
        const cy = oy - device.y * scale / 100;
        
        const dx = mouseX - cx;
        const dy = mouseY - cy;
        const distance = Math.sqrt(dx * dx + dy * dy);
        
        if (distance <= widthPx / 2) {
            hitDevices.push({
                device: device,
                distance: distance,
                screenX: cx,
                screenY: cy
            });
        }
    }
    
    if (hitDevices.length === 0) {
        return null;
    }
    
    if (hitDevices.length === 1) {
        return hitDevices[0].device;
    }
    
    hitDevices.sort((a, b) => a.distance - b.distance);
    
    if (gSelectedDevice) {
        const selectedInHitList = hitDevices.find(hit => hit.device.id === gSelectedDevice.id);
        
        if (selectedInHitList) 
        {
            if (isRightClick) 
            {
                const lastRightClickDevice = window.ZoneFullMap_lastRightClickDevice;
                
                if (lastRightClickDevice && lastRightClickDevice === gSelectedDevice.id) 
                {
                    return gSelectedDevice;
                } 
                else 
                {
                    window.ZoneFullMap_lastRightClickDevice = null;
                    ZoneFullMap_showDeviceSelector(hitDevices, mouseX, mouseY);
                    return null;
                }
            } 
            else 
            {
                return gSelectedDevice;
            }
        }
    }
    
    ZoneFullMap_showDeviceSelector(hitDevices, mouseX, mouseY, isRightClick); 
    return null;
}

function ZoneFullMap_clearDeviceHighlight() {
    if (!gSelectedDevice || !gSelectedDevice._highlightArea) return;
    
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    const area = gSelectedDevice._highlightArea;
    
    if (area.nameBox) {
        ctx.clearRect(area.nameBox.x, area.nameBox.y, area.nameBox.w, area.nameBox.h);
    }
    
    if (area.coordBox) {
        ctx.clearRect(area.coordBox.x, area.coordBox.y, area.coordBox.w, area.coordBox.h);
    }
    
    delete gSelectedDevice._highlightArea;
}

function ZoneFullMap_highlightSelectedDevice() {
    if (!gSelectedDevice) {
        ZoneFullMap_clearAllHighlights();
        return;
    }
    
    if (gZoneFullMap_deviceVisibility[gSelectedDevice.id] === false) {
        ZoneFullMap_clearAllHighlights();
        gSelectedDevice = null;
        return;
    }
    
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_highLight');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    // 根據設備類型使用不同的基礎尺寸
    let widthPx;
    if (gSelectedDevice.type === 'camera' || gSelectedDevice.type === 'vision') {
        // 攝影機和 BC-200 使用配置中的圓圈大小
        const config = gSelectedDevice.type === 'camera' ? 
            ZoneFullMap_DeviceVisualConfig.camera : 
            ZoneFullMap_DeviceVisualConfig.bc200;
        widthPx = config.baseCircleSize * scale * 100;
    } else {
        // 麥克風和錨點使用原本的 baseWidthM
        widthPx = gSelectedDevice.baseWidthM * scale * 100;
    }
    
    const cx = ox + gSelectedDevice.x * scale / 100;
    const cy = oy - gSelectedDevice.y * scale / 100;
    
    let highlightColor = '#FFD700';
    let deviceDisplayName = gSelectedDevice.name;
    let colorScheme = null;
    
    function drawRoundedRect(ctx, x, y, width, height, radius) {
        ctx.beginPath();
        ctx.moveTo(x + radius, y);
        ctx.lineTo(x + width - radius, y);
        ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
        ctx.lineTo(x + width, y + height - radius);
        ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
        ctx.lineTo(x + radius, y + height);
        ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
        ctx.lineTo(x, y + radius);
        ctx.quadraticCurveTo(x, y, x + radius, y);
        ctx.closePath();
    }
    
    if (gSelectedDevice.type === 'microphone') {
        colorScheme = ZoneFullMap_getMicrophoneColorScheme(gSelectedDevice.id);
        highlightColor = colorScheme.primary;
        deviceDisplayName = ZoneFullMap_getMicrophoneDisplayName(gSelectedDevice);
    } else if (gSelectedDevice.type === 'camera') {
        colorScheme = ZoneFullMap_getCameraColorScheme(gSelectedDevice.id);
        highlightColor = colorScheme.primary;
        const camNumber = gSelectedDevice.id.match(/\d+/);
        deviceDisplayName = `Cam ${camNumber ? camNumber[0] : ''}`;
    } else if (gSelectedDevice.type === 'vision') {
        colorScheme = ZoneFullMap_getVisionColorScheme(gSelectedDevice.id);
        highlightColor = colorScheme.primary;
        const idNumber = gSelectedDevice.id.match(/vision_(\d+)/);
        deviceDisplayName = `BC-200 ${idNumber ? idNumber[1] : '1'}`;
    } else if (gSelectedDevice.type === 'anchor') {
        highlightColor = '#FF5722';
        const anchorNumber = gSelectedDevice.name.match(/\d+/);
        deviceDisplayName = `Anchor ${anchorNumber ? anchorNumber[0] : ''}`;
    }
    
    const baseFontSize = 18;
    const fontSize = Math.max(18, Math.min(20, baseFontSize / Math.sqrt(scale)));
    
    ctx.font = `bold ${fontSize}px Arial`;
    const nameMetrics = ctx.measureText(deviceDisplayName);
    const nameWidth = nameMetrics.width;
    const nameHeight = fontSize;
    
    const namePaddingLeft = 10;
    const namePaddingRight = 10;
    const namePaddingTop = 4;
    const namePaddingBottom = 4;
    
    const nameBoxWidth = nameWidth + namePaddingLeft + namePaddingRight;
    const nameBoxHeight = nameHeight + namePaddingTop + namePaddingBottom;
    
    let  nameBoxX = cx - nameBoxWidth / 2;
    let  nameBoxY = cy - widthPx / 2 - 15 - nameBoxHeight;
    if (gSelectedDevice.type !== 'microphone') {
        nameBoxX = cx - nameBoxWidth / 2;
        nameBoxY = cy - widthPx / 8 - 90 - nameBoxHeight;
    }
    
    gSelectedDevice._highlightArea = {
        nameBox: { x: nameBoxX - 5, y: nameBoxY - 5, w: nameBoxWidth + 10, h: nameBoxHeight + 10 },
        coordBox: null
    };
    
    ctx.clearRect(nameBoxX - 5, nameBoxY - 5, nameBoxWidth + 10, nameBoxHeight + 10);
    
    if (colorScheme && gSelectedDevice.type === 'microphone') {
        ctx.fillStyle = colorScheme.primary;
    } else {
        ctx.fillStyle = highlightColor;
    }
    
    drawRoundedRect(ctx, nameBoxX, nameBoxY, nameBoxWidth, nameBoxHeight, 4);
    ctx.fill();
    
    ctx.fillStyle = '#FFFFFF';
    ctx.font = `bold ${fontSize}px Arial`;
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText(
        deviceDisplayName,
        nameBoxX + nameBoxWidth / 2,
        nameBoxY + nameBoxHeight / 2
    );

    const xInMeters = (gSelectedDevice.x / 10000).toFixed(2);
    const yInMeters = (gSelectedDevice.y / 10000).toFixed(2);
    const coordText = `(${xInMeters}, ${yInMeters})`;
    
    const coordFontSize = Math.max(18, Math.min(20, 20 / Math.sqrt(scale)));
    ctx.font = `bold ${coordFontSize}px Arial`;
    
    const textMetrics = ctx.measureText(coordText);
    const textWidth = textMetrics.width;
    const textHeight = coordFontSize;
    
    const coordPaddingLeft = 12;
    const coordPaddingRight = 12;
    const coordPaddingTop = 4;
    const coordPaddingBottom = 4;
    
    const coordBoxWidth = textWidth + coordPaddingLeft + coordPaddingRight;
    const coordBoxHeight = textHeight + coordPaddingTop + coordPaddingBottom;
    
    const minCoordBoxWidth = 80;
    const finalCoordBoxWidth = Math.max(minCoordBoxWidth, coordBoxWidth);
    
    const coordBoxX = cx - finalCoordBoxWidth / 2;
    let coordOffset = 15;
    let coordBoxY = cy + widthPx / 2 + coordOffset;
    if(gSelectedDevice.type !== 'microphone')
    {
        coordBoxY = cy + widthPx / 8 + 90;
    }
    
    gSelectedDevice._highlightArea.coordBox = {
        x: coordBoxX - 5, y: coordBoxY - 5, 
        w: finalCoordBoxWidth + 10, h: coordBoxHeight + 10
    };
    
    ctx.clearRect(coordBoxX - 5, coordBoxY - 5, finalCoordBoxWidth + 10, coordBoxHeight + 10);
    
    if (colorScheme && gSelectedDevice.type === 'microphone') {
        ctx.fillStyle = colorScheme.primary;
    } else {
        ctx.fillStyle = highlightColor;
    }
    
    drawRoundedRect(ctx, coordBoxX, coordBoxY, finalCoordBoxWidth, coordBoxHeight, 4);
    ctx.fill();
    
    ctx.fillStyle = '#FFFFFF';
    ctx.font = `bold ${coordFontSize}px Arial`;
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    
    const coordTextX = coordBoxX + finalCoordBoxWidth / 2;
    const coordTextY = coordBoxY + coordBoxHeight / 2;
    
    ctx.fillText(coordText, coordTextX, coordTextY);
}

function ZoneFullMap_createQuadrilateralForDevice(screenPoints, deviceId) {
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    const mathPoints = screenPoints.map(screenPoint => ({
        x: (screenPoint.x - ox) * 100 / scale,
        y: (oy - screenPoint.y) * 100 / scale
    }));
    
    console.log('Creating quadrilateral for device:', {
        deviceId,
        screenPoints,
        mathPoints,
        canvasTransform: { x: ox, y: oy, scale }
    });
    
    const newQuad = new ZoneFullMap_Quadrilateral(mathPoints, deviceId);
    
    if (newQuad.hasSelfIntersection()) {
        console.warn('Cannot create self-intersecting quadrilateral for device:', deviceId);
        return null;
    }
    
    gZoneFullMap_Quadrilaterals.push(newQuad);
    gZoneFullMap_ActiveRectangles.push(newQuad.id);
    gZoneFullMap_ActiveRectangles.sort((a, b) => a - b);
    
    const device = ZoneFullMap_devices.find(d => d.id === deviceId);
    const deviceName = device ? device.name : deviceId;
    
    console.log('Zone created successfully:', {
        deviceId,
        deviceName,
        quadId: newQuad.id,
        displayId: newQuad.getDisplayZoneId(),
        colorScheme: newQuad.colorScheme.name,
        totalQuadsForDevice: gZoneFullMap_Quadrilaterals.filter(q => q.deviceId === deviceId).length,
        allActiveIds: gZoneFullMap_ActiveRectangles.slice()
    });
    
    return newQuad;
}


function ZoneFullMap_getMicrophoneZoneStats(microphoneId) {
    ZoneFullMap_initializeMicrophoneZoneCounter(microphoneId);
    const counter = gZoneFullMap_MicrophoneZoneCounters[microphoneId];
    const activeZones = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === microphoneId);
    
    return {
        totalCreated: counter.nextId - 1,
        activeCount: activeZones.length,
        deletedCount: counter.deletedIds.length,
        maxZones: 128 // 每個麥克風最多128個Zone
    };
}


function ZoneFullMap_drawZonePreview(ctx, startX, startY, currentX, currentY, selectedMicrophone) {
    if (!selectedMicrophone) return;
    
    const rectLeft = Math.min(startX, currentX);
    const rectTop = Math.min(startY, currentY);
    const rectRight = Math.max(startX, currentX);
    const rectBottom = Math.max(startY, currentY);
    
    const width = rectRight - rectLeft;
    const height = rectBottom - rectTop;
    
    // 如果區域太小，不顯示預覽
    if (width < 5 || height < 5) return;
    
    // 獲取麥克風的配色方案
    const colorScheme = ZoneFullMap_getMicrophoneColorScheme(selectedMicrophone.id);
    
    ctx.save();
    
    ctx.beginPath();
    ctx.rect(rectLeft, rectTop, width, height);
    
    // 填充
    ctx.fillStyle = colorScheme.zone_fill.replace('0.3', '0.2');
    ctx.fill();
    
    // 邊框
    ctx.strokeStyle = colorScheme.primary;
    ctx.setLineDash([8, 4]);
    ctx.lineWidth = 3;
    ctx.globalAlpha = 0.9;
    ctx.stroke();
    
    // 標記起始點（固定角）
    ctx.setLineDash([]);  // 重置虛線
    ctx.beginPath();
    ctx.arc(startX, startY, 6, 0, 2 * Math.PI);
    ctx.fillStyle = colorScheme.dark;
    ctx.fill();
    ctx.strokeStyle = 'white';
    ctx.lineWidth = 2;
    ctx.stroke();
    
    // 標記當前滑鼠位置（活動角）
    ctx.beginPath();
    ctx.arc(currentX, currentY, 4, 0, 2 * Math.PI);
    ctx.fillStyle = 'white';
    ctx.fill();
    ctx.strokeStyle = colorScheme.primary;
    ctx.lineWidth = 1.5;
    ctx.stroke();
    
    // 顯示尺寸提示（使用正確的中心點）
    if (width > 50 && height > 50) {
        const centerX = rectLeft + width / 2;
        const centerY = rectTop + height / 2;
        
        // 顯示區域大小
        const { scale } = gDragSystem.canvasTransform;
        const widthInCm = (width * 100 / scale).toFixed(0);
        const heightInCm = (height * 100 / scale).toFixed(0);
        
        const text = `${widthInCm} x ${heightInCm} cm`;
        
        ctx.font = 'bold 14px Arial';
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        
        const metrics = ctx.measureText(text);
        const textWidth = metrics.width;
        const padding = 4;
        
        // 背景框
        ctx.fillStyle = 'rgba(255, 255, 255, 0.95)';
        ctx.fillRect(centerX - textWidth/2 - padding, centerY - 10, textWidth + padding*2, 20);
        
        // 文字
        ctx.fillStyle = colorScheme.dark;
        ctx.fillText(text, centerX, centerY);
    }
    
    ctx.restore();
}

function ZoneFullMap_clearSelection() {
    gSelectedDevice = null;
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_mouse');
    if (canvas) {
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
}

function ZoneFullMap_onCanvasDoubleClick(e) {
    const containerRect = e.currentTarget.getBoundingClientRect();
    const mouseX = e.clientX - containerRect.left;
    const mouseY = e.clientY - containerRect.top;
    
    const hitDevice = ZoneFullMap_hitTestDevice(mouseX, mouseY);
    
    if (hitDevice) {
        if (gSelectedDevice === hitDevice) {
            ZoneFullMap_clearSelection();
        } else {
            gSelectedDevice = hitDevice;
            ZoneFullMap_highlightSelectedDevice();
        }
    } else {
        ZoneFullMap_clearSelection();
    }
}

var gMappingOverViewStatus = 0; 

function ZoneFullMap_onKeyDown(e) {

    if (e.key === 'Control') 
    {
        gZoneFullMap_IsCtrlPressed = true;
        gZoneFullMap_GroupSelectionMode = true;
        console.log('Ctrl pressed - Group selection mode enabled');
        return;
    }

    if (e.key === 'd' || e.key === 'D') {
        e.preventDefault();
        
        ZoneFullMap_applyZoneDisplayChanges();
        
        console.log('D key toggled display zone data');
        return;
    }

    if (e.key === 's' || e.key === 'S') {
        e.preventDefault();
        
        ZoneFullMap_applyZoneChanges();

        console.log('S key toggled save zone data');
        return;
    }

    if (e.code === 'Space' || e.key === ' ') {
        e.preventDefault();
        
        const editWindow = document.getElementById('MicZone_Edit_Window');
        
        if (editWindow) 
        {
            const mapEditBtn = document.getElementById('mapEditModeBtn');
            const zoneEditBtn = document.getElementById('zoneEditModeBtn');
            
            if (gZoneFullMap_EditModeState) 
            {
                mapEditBtn?.click();

                gMappingOverViewStatus = 3;
                getMappingOverViewBlockUIforPage();

                setTimeout(
                    function()
                    {
                        getMappingOverViewUnblockUIforPage();
                        gMappingOverViewStatus = 0;
                    }
                ,600);
            } 
            else 
            {
                zoneEditBtn?.click();

                gMappingOverViewStatus = 2;
                getMappingOverViewBlockUIforPage();
                setTimeout(
                    function()
                    {
                        getMappingOverViewUnblockUIforPage();
                        gMappingOverViewStatus = 0;
                    }
                ,600);
            }
        } 
        else 
        {
            const displayButton = document.getElementById('ZoneFullMapZoneDisplayBtn');
            
            if (gZoneFullMap_EditModeState) 
            {
                gZoneFullMap_EditModeState = false;

                gMappingOverViewStatus = 3;
                getMappingOverViewBlockUIforPage();

                setTimeout(
                    function()
                    {
                        getMappingOverViewUnblockUIforPage();
                        gMappingOverViewStatus = 0;
                    }
                ,600);
                
                const checkbox = { checked: false };
                ZoneFullMapMicZoneDrawMode(checkbox);
                
                if (displayButton) {
                    if (!gZoneFullMap_LastDisplayZoneState) {
                        ZoneFullMap_applyZoneDisplayChanges();  
                    }
                    if (gZoneFullMap_LastDisplayZoneState) {
                        displayButton.classList.add('active');
                    }
                    displayButton.classList.remove('disabled');
                    displayButton.disabled = false;
                }
            } 
            else 
            {
                gZoneFullMap_EditModeState = true;

                gMappingOverViewStatus = 2;
                getMappingOverViewBlockUIforPage();
                setTimeout(
                    function()
                    {
                        getMappingOverViewUnblockUIforPage();
                        gMappingOverViewStatus = 0;
                    }
                ,600);
                
                const checkbox = { checked: true };
                ZoneFullMapMicZoneDrawMode(checkbox);
                
                if (displayButton) {
                    if (!gZoneFullMap_DisplayZoneState) {
                        gZoneFullMap_DisplayZoneState = true;
                        displayButton.textContent = '顯示麥克風區域';
                        displayButton.classList.remove('active');
                    } else {            
                        displayButton.classList.remove('active');
                    }
                    displayButton.classList.add('disabled');
                    displayButton.disabled = true;
                }
            }
        }
        
        console.log('Space key toggled edit mode:', gZoneFullMap_EditModeState);
        return;
    }

    if (e.key === 'f' || e.key === 'F') {
        e.preventDefault();
        
        ZoneFullMap_toggleDataMode();
        
        console.log('F key toggled data mode:', gZoneFullMap_DataModeState);
        return;
    }

    // ESC 鍵退出繪圖模式
    if (gZoneFullMap_DrawingMode && e.key === 'Escape') {
        ZoneFullMap_exitDrawingMode();
        e.preventDefault();
        return;
    }

    if (!gSelectedDevice) return;
    
    // 其他按鍵處理保持不變
    switch(e.key) {
        case 'Delete':
        case 'Backspace':
            if (gZoneFullMap_LastSelectedQuad) {
                ZoneFullMap_deleteQuadrilateral(gZoneFullMap_LastSelectedQuad);
            } else if (gSelectedDevice && gSelectedDevice.canDelete) {
                ZoneFullMap_deleteDevice(gSelectedDevice);
            }
            break;
        case 'Escape':
            ZoneFullMap_clearSelection();
            break;
        case 'ArrowUp':
            gSelectedDevice.y += e.shiftKey ? 10 : 1;
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_highlightSelectedDevice();
            e.preventDefault();
            break;
        case 'ArrowDown':
            gSelectedDevice.y -= e.shiftKey ? 10 : 1;
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_highlightSelectedDevice();
            e.preventDefault();
            break;
        case 'ArrowLeft':
            gSelectedDevice.x -= e.shiftKey ? 10 : 1;
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_highlightSelectedDevice();
            e.preventDefault();
            break;
        case 'ArrowRight':
            gSelectedDevice.x += e.shiftKey ? 10 : 1;
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_highlightSelectedDevice();
            e.preventDefault();
            break;
    }
}

function ZoneFullMap_onKeyUp(e) {
    if (e.key === 'Control') 
    {
        gZoneFullMap_IsCtrlPressed = false;
        gZoneFullMap_GroupSelectionMode = false;
        console.log('Ctrl released - Group selection mode disabled');
        return;
    }
}

function ZoneFullMap_deleteDevice(device) {
    if (!device || !device.canDelete) return;
    
    console.log('Deleting device:', device.name, 'Type:', device.type);
    
    const index = ZoneFullMap_devices.indexOf(device);
    if (index > -1) {
        if (device.type === 'microphone') {
            if (device.connectedTabIndex !== undefined) {
                console.log('Clearing tab connection:', device.connectedTabIndex);
                delete gZoneFullMap_MicIdToTabMapping[device.id];
                delete gZoneFullMap_TabToMicIdMapping[device.connectedTabIndex];
            }
            
            console.log('Clearing microphone zones for:', device.id);
            ZoneFullMap_clearMicrophoneZones(device.id);
            
            if (gZoneFullMap_MicrophoneZoneCounters[device.id]) {
                delete gZoneFullMap_MicrophoneZoneCounters[device.id];
            }
        }
        
        if (gSelectedDevice === device) {
            console.log('Clearing selected device and highlights');
            gSelectedDevice = null;
            ZoneFullMap_clearAllHighlights();
        }
        
        if (device.imageElement && device.imageElement.parentNode) {
            device.imageElement.remove();
            console.log('Removed DOM image element');
        }
        
        if (device._preloadedImage) {
            device._preloadedImage = null;
            console.log('Cleared preloaded image');
        }
        
        const infoBox = document.getElementById('ZoneFullMap_device_info');
        if (infoBox) {
            infoBox.remove();
            console.log('Closed device info window');
        }
        
        const match = device.id.match(/_(\d+)$/);
        if (match) {
            const deviceNumber = parseInt(match[1]);
            ZoneFullMap_releaseDeviceId(device.type, deviceNumber);
        }
        ZoneFullMap_devices.splice(index, 1);

        console.log('Removed device from array');
        
        if (device.type === 'microphone') {
            Object.keys(gSimpleSoundPoints).forEach(tabIndex => {
                if (parseInt(tabIndex) === device.connectedTabIndex) {
                    delete gSimpleSoundPoints[tabIndex];
                    console.log('Cleared sound points for tab:', tabIndex);
                }
            });
            
            Object.keys(gSimpleZoneStates).forEach(tabIndex => {
                if (parseInt(tabIndex) === device.connectedTabIndex) {
                    delete gSimpleZoneStates[tabIndex];
                    console.log('Cleared zone states for tab:', tabIndex);
                }
            });
        }
        
        ZoneFullMap_updateButtonStates();
        
        requestAnimationFrame(() => {
            ZoneFullMap_clearAllCanvases();
            
            if (gDragSystem && gDragSystem.drawGrid) {
                gDragSystem.drawGrid();
            }
            
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_redrawQuadrilaterals();
            
            if (!gSelectedDevice) {
                ZoneFullMap_updateInfoPanelToDefault();
            }
            
            console.log('Complete redraw after device deletion finished');
        });
        
        console.log('Device deletion completed:', {
            deletedDevice: device.name,
            remainingDevices: ZoneFullMap_devices.length,
            remainingZones: gZoneFullMap_Quadrilaterals.length
        });
    }
}

function ZoneFullMap_clearAllCanvases() {
    const canvasIds = [
        'ZoneFullMap_xy_canvas_mic',
        'ZoneFullMap_xy_canvas_highLight', 
        'ZoneFullMap_xy_canvas_rect',
        'ZoneFullMap_xy_canvas_visionPoint',
        'ZoneFullMap_xy_canvas_soundPoint',
        'ZoneFullMap_xy_canvas_anchor',
        'ZoneFullMap_xy_canvas_mouse'
    ];
    
    canvasIds.forEach(canvasId => {
        const canvas = document.getElementById(canvasId);
        if (canvas) {
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
        }
    });
    
    console.log('Cleared all canvases');
}

function ZoneFullMap_updateInfoPanelToDefault() {
    const infoElements = [
        { id: 'ZoneFullMap_mic_index_value', defaultText: 'null' },
        { id: 'ZoneFullMap_model_value', defaultText: 'null' },
        { id: 'ZoneFullMap_camera_value', defaultText: 'null' },
        { id: 'ZoneFullMap_camera_preset_value', defaultText: 'null' }
    ];
    
    infoElements.forEach(({ id, defaultText }) => {
        const element = document.getElementById(id);
        if (element) {
            const nullText = window.LanguageManager ? 
                window.LanguageManager.getTranslatedText('null') : defaultText;
            element.textContent = nullText;
        }
    });
    
    console.log('Reset info panel to default state');
}

function ZoneFullMap_initDeviceDragging() 
{
    ZoneFullMap_setupEnhancedMouseEvents();
    ZoneFullMap_initQuadrilateralSystem();
}

function ZoneFullMap_setupEnhancedMouseEvents() {
    const canvasContainer = document.querySelector('.ZoneFullMap_canvas_td');
    if (!canvasContainer) return;
    
    canvasContainer.removeEventListener('mousedown', gDragSystem._onCanvasMouseDown);
    canvasContainer.removeEventListener('mousedown', ZoneFullMap_onCanvasMouseDown_WithQuad);
    canvasContainer.removeEventListener('dblclick', ZoneFullMap_onCanvasDoubleClick);
    canvasContainer.removeEventListener('contextmenu', ZoneFullMap_preventContextMenu);
    
    canvasContainer.addEventListener('mousedown', ZoneFullMap_onCanvasMouseDown_WithQuad);
    canvasContainer.addEventListener('dblclick', ZoneFullMap_onCanvasDoubleClick);
    canvasContainer.addEventListener('contextmenu', ZoneFullMap_preventContextMenu);
    
    document.addEventListener('keydown', ZoneFullMap_onKeyDown);
    document.addEventListener('keyup', ZoneFullMap_onKeyUp);
    
    //console.log('Enhanced mouse events with quadrilateral support initialized');
}

function ZoneFullMap_preventContextMenu(e) {
    e.preventDefault();
    return false;
}

function ZoneFullMap_showDeviceInfo(device) {
    ZoneFullMap_hideDeviceInfo();

    const infoBox = document.createElement('div');
    infoBox.id = 'ZoneFullMap_device_info';
    infoBox.style.position = 'absolute';
    infoBox.className = "ZoneFullMap_device_info"
    infoBox.style.padding = '12px';
    infoBox.style.borderRadius = '8px';
    infoBox.style.zIndex = '100';
    infoBox.style.boxShadow = '0 4px 20px rgba(0,0,0,0.3)';
    infoBox.style.fontFamily = 'Arial, sans-serif';
    infoBox.style.pointerEvents = 'auto';
    infoBox.style.minWidth = '280px';
    infoBox.style.fontSize = '14px';

    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const widthPx = device.baseWidthM * scale * 100;
    const cx = ox + device.x * scale / 100;
    const cy = oy - device.y * scale / 100;

    infoBox.style.left = `${cx + widthPx / 2 + 20}px`;
    infoBox.style.top = `${cy - 120}px`;

    const originalData = {
        x: device.x,
        y: device.y,
        angle: device.angle,
        fovAngle: device.fovAngle,
        offsetAngle: device.offsetAngle,
        offsetX: device.offsetX,
        offsetY: device.offsetY,
        name: device.name,
        ip: device.ip || '',
        microphoneType: device.microphoneType || 'Shure_MXA310'
    };

    let infoHTML = `<div style="margin-bottom: 10px; border-bottom: 1px solid  padding-bottom: 8px;">
        <strong style="font-size: 20px;" class="ZoneFullMap_deviceInfo">${device.name}</strong>
    </div>`;

    if (device.type === 'microphone') {
        const micHTML = generateZoneFullMapMicrophoneHTML(device);
        infoHTML += micHTML;
    }

    if (device.type === 'camera') {
        const camHTML = generateZoneFullMapCameraHTML(device);
        infoHTML += camHTML;
    }

    if (device.type === 'vision') {
        const bc200HTML = generateZoneFullMapVisionHTML(device);
        infoHTML += bc200HTML;
    }
    let positionWidth = 120;
    if (device.type === 'vision') 
    {
        positionWidth = 170;
    }

    infoHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: ${positionWidth}px; display: inline-block; font-size: 18px;" class="ZoneFullMap_deviceInfo">Position:</span>
            <input type="number" id="ZoneFullMap_x_${device.id}"
                value="${(device.x / 10000).toFixed(2)}" step="0.01"
                style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; margin-right: 5px; text-align: center; border: none;"
                onchange="event.stopPropagation();" onclick="event.stopPropagation();">
            <input type="number" id="ZoneFullMap_y_${device.id}" 
                value="${(device.y / 10000).toFixed(2)}" step="0.01"
                style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                onchange="event.stopPropagation();" onclick="event.stopPropagation();">
        </div>
    `;

    switch (device.type) {
        case 'microphone':
            const associatedQuads = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === device.id);
            
            infoHTML += `
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 120px; display: inline-block; font-size: 18px;" class="ZoneFullMap_deviceInfo">Rotation:</span>
                    <input type="number" id="ZoneFullMap_angle_${device.id}" 
                        value="${device.angle || 0}" min="0" max="360" step="1"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                        onchange="event.stopPropagation();" onclick="event.stopPropagation();">
                </div>
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Zones:</span>
                    <span style="font-size: 18px;" class="ZoneFullMap_deviceInfo">${associatedQuads.length} zone(s)</span>
                </div>
            `;
            
            if (gZoneFullMap_GlobalDrawingMode) {
                infoHTML += `
                    <div style="display: flex; align-items: center; margin-bottom: 8px; padding: 6px; background-color: rgba(76, 175, 80, 0.2); border-radius: 4px;">
                        <span style="font-size: 18px;" class="ZoneFullMap_deviceInfo">
                            Zone Drawing Mode: Click this microphone to draw zones
                        </span>
                    </div>
                `;
            }
            break;

        case 'camera':
            const nearestDirection = ZoneFullMap_getNearestDirection(device.angle);
            infoHTML += `
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 120px; display: inline-block; font-size: 18px;" class="ZoneFullMap_deviceInfo">Direction:</span>
                    <select id="ZoneFullMap_direction_${device.id}" 
                        style="width: 120px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px; text-align: center; border: none;"
                        onchange="ZoneFullMap_updateDeviceDirection('${device.id}', this.value)">
                        <option value="0" ${nearestDirection === 0 ? 'selected' : ''}>Down (0°)</option>
                        <option value="90" ${nearestDirection === 90 ? 'selected' : ''}>Left (90°)</option>
                        <option value="180" ${nearestDirection === 180 ? 'selected' : ''}>Up (180°)</option>
                        <option value="270" ${nearestDirection === 270 ? 'selected' : ''}>Right (270°)</option>
                    </select>
                </div>
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 120px; display: inline-block;  font-size: 18px;" class="ZoneFullMap_deviceInfo">FOV Angle:</span>
                    <input type="number" id="ZoneFullMap_fov_${device.id}" 
                        value="${device.fovAngle || 0}" min="0" max="360"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                        onchange="event.stopPropagation();" onclick="event.stopPropagation();">
                </div>
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 120px; display: inline-block;  font-size: 18px;" class="ZoneFullMap_deviceInfo">Offset:</span>
                    <input type="number" id="ZoneFullMap_offset_${device.id}" 
                        value="${device.offsetAngle || 0}" min="-360" max="360"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                        onchange="event.stopPropagation();" onclick="event.stopPropagation();">
                </div>
            `;
            break;

        case 'vision':
            const visionDirection = ZoneFullMap_getNearestDirection(device.angle);
            infoHTML += `
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 170px; display: inline-block; font-size: 18px;" class="ZoneFullMap_deviceInfo">Direction:</span>
                    <select id="ZoneFullMap_direction_${device.id}" 
                        style="width: 120px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px; text-align: center; border: none;"
                        onchange="ZoneFullMap_updateDeviceDirection('${device.id}', this.value)">
                        <option value="0" ${visionDirection === 0 ? 'selected' : ''}>Up (0°)</option>
                        <option value="90" ${visionDirection === 90 ? 'selected' : ''}>Right (90°)</option>
                        <option value="180" ${visionDirection === 180 ? 'selected' : ''}>Down (180°)</option>
                        <option value="270" ${visionDirection === 270 ? 'selected' : ''}>Left (270°)</option>
                    </select>
                </div>
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 170px; display: inline-block;  font-size: 18px;" class="ZoneFullMap_deviceInfo">FOV Angle:</span>
                    <input type="number" id="ZoneFullMap_fov_${device.id}" 
                        value="${device.fovAngle || 0}" min="0" max="360"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                        onchange="event.stopPropagation();" onclick="event.stopPropagation();">
                </div>
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 170px; display: inline-block; font-size: 18px;" class="ZoneFullMap_deviceInfo">Offset Angle:</span>
                    <input type="number" id="ZoneFullMap_offset_${device.id}" 
                        value="${device.offsetAngle || 0}" min="-360" max="360" step="1"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                        onchange="ZoneFullMap_updateDeviceOffset('${device.id}', this.value); event.stopPropagation();" 
                        onclick="event.stopPropagation();">
                </div>
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: ${positionWidth}px; display: inline-block; font-size: 18px;" class="ZoneFullMap_deviceInfo">Offset Position:</span>
                    <input type="number" id="ZoneFullMap_offsetX_${device.id}"
                        value="${(device.offsetX).toFixed(2)}" step="0.01"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; margin-right: 5px; text-align: center; border: none;"
                        onchange="event.stopPropagation();" onclick="event.stopPropagation();">
                    <input type="number" id="ZoneFullMap_offsetY_${device.id}" 
                        value="${(device.offsetY).toFixed(2)}" step="0.01"
                        style="width: 60px; background-color: white; color: black; padding: 3px; border-radius: 3px; text-align: center; border: none;"
                        onchange="event.stopPropagation();" onclick="event.stopPropagation();">
                </div>
            `;
            break;

        case 'anchor':
            infoHTML += `
                <div style="display: flex; align-items: center; margin-bottom: 8px;">
                    <span style="width: 100px; display: inline-block;" class="ZoneFullMap_deviceInfo">Status:</span>
                    <span class="ZoneFullMap_deviceInfo">${device.status}</span>
                </div>
            `;
            break;
    }

    // Action buttons
    infoHTML += `
        <div style="display: flex; justify-content: space-between; margin-top: 10px;">
            <div style="display: flex;">
                <button onclick="ZoneFullMap_applyDeviceChanges('${device.id}')" class="ZoneFullMap_applyDevice_btn">Apply</button>
                <button onclick="ZoneFullMap_cancelDeviceChanges('${device.id}')" class="ZoneFullMap_cancelDevice_btn">Cancel</button>
            </div>
            <div>
                <button onclick="ZoneFullMap_deleteDevice(ZoneFullMap_findDeviceById('${device.id}'))" class="ZoneFullMap_deleteDevice_btn">Delete</button>
            </div>
        </div>
    `;

    if(infoBox)
    {
        infoBox.innerHTML = infoHTML;
        infoBox.setAttribute('data-original', JSON.stringify(originalData));
    }

    const container = document.querySelector('.ZoneFullMap_canvas_td');
    if (container)
    {
        container.appendChild(infoBox);
    }

    const containerRect = container.getBoundingClientRect();
    let infoX = cx + widthPx / 2 + 20;
    let infoY = cy - 120;
    const estimatedWidth = 300;
    const estimatedHeight = 250;
    
    if (infoX + estimatedWidth > containerRect.width) {
        infoX = cx - widthPx / 2 - estimatedWidth - 20;
    }
    
    if (infoY < 0) {
        infoY = cy + widthPx / 2 + 20;
    }
    
    if (infoY + estimatedHeight > containerRect.height) {
        infoY = containerRect.height - estimatedHeight - 10;
    }
    
    if (cx < -widthPx || cx > containerRect.width + widthPx ||
        cy < -widthPx || cy > containerRect.height + widthPx) {
        console.log('Device info not shown - device is outside visible area');
        return;
    }

    infoBox.addEventListener('click', (e) => {
        e.stopPropagation();
    });

    // Add input validation to prevent live updates
    const inputs = infoBox.querySelectorAll('input[type="number"]');
    inputs.forEach(input => {
        input.addEventListener('input', (e) => {
            e.stopPropagation();
        });
        input.addEventListener('change', (e) => {
            e.stopPropagation();
        });
        input.addEventListener('keydown', (e) => {
            e.stopPropagation();
            if (e.key === 'Enter') {
                e.preventDefault();
            }
        });
    });
}


function ZoneFullMap_toggleDrawingModeFromInfo(deviceId) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (device && device.type === 'microphone') {
        ZoneFullMap_toggleDrawingMode(device);
        ZoneFullMap_hideDeviceInfo();
        ZoneFullMap_showDeviceInfo(device);
    }
}

function ZoneFullMap_hideDeviceInfo() {
    const infoBox = document.getElementById('ZoneFullMap_device_info');
    if (infoBox && infoBox.parentNode) {
        infoBox.remove();
    }
}

function ZoneFullMap_applyDeviceChanges(deviceId) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) {
        console.error('Device not found:', deviceId);
        return false;
    }

    const inputs = ZoneFullMap_collectDeviceInputs(deviceId);
    
    const oldValues = ZoneFullMap_captureOldValues(device);
    
    const positionChanged = ZoneFullMap_applyPositionChanges(device, inputs, oldValues);
    
    let deviceSpecificChanged = false;
    switch (device.type) {
        case 'microphone':
            deviceSpecificChanged = ZoneFullMap_applyMicrophoneChanges(device, inputs, oldValues);
            break;
        case 'camera':
            deviceSpecificChanged = ZoneFullMap_applyCameraChanges(device, inputs, oldValues);
            break;
        case 'vision':
            deviceSpecificChanged = ZoneFullMap_applyVisionChanges(device, inputs, oldValues);
            break;
        case 'anchor':
            deviceSpecificChanged = ZoneFullMap_applyAnchorChanges(device, inputs, oldValues);
            break;
    }
    
    // 如果有任何變更，更新顯示
    if (positionChanged || deviceSpecificChanged) {
        ZoneFullMap_updateDisplayAfterChanges(device, oldValues);
        console.log('Applied changes to device:', {
            deviceId: device.id,
            deviceName: device.name,
            positionChanged,
            deviceSpecificChanged,
            newValues: ZoneFullMap_getDeviceCurrentValues(device)
        });
    }
    
    // 關閉設備信息面板
    ZoneFullMap_hideDeviceInfo();
    
    return true;
}

// 收集所有相關的輸入元素
function ZoneFullMap_collectDeviceInputs(deviceId) {
    return {
        x: document.getElementById(`ZoneFullMap_x_${deviceId}`),
        y: document.getElementById(`ZoneFullMap_y_${deviceId}`),
        angle: document.getElementById(`ZoneFullMap_angle_${deviceId}`),
        direction: document.getElementById(`ZoneFullMap_direction_${deviceId}`),
        fov: document.getElementById(`ZoneFullMap_fov_${deviceId}`),
        offset: document.getElementById(`ZoneFullMap_offset_${deviceId}`),
        offsetX: document.getElementById(`ZoneFullMap_offsetX_${deviceId}`),
        offsetY: document.getElementById(`ZoneFullMap_offsetY_${deviceId}`),
        micType: document.getElementById(`ZoneFullMap_micType_${deviceId}`),
        ip: document.getElementById(`ZoneFullMap_ip_${deviceId}`)
    };
}

// 保存設備的舊值
function ZoneFullMap_captureOldValues(device) {
    return {
        x: device.x,
        y: device.y,
        angle: device.angle || 0,
        fovAngle: device.fovAngle || 0,
        offsetAngle: device.offsetAngle || 0,
        microphoneType: device.microphoneType,
        ip: device.ip || ''
    };
}

// 應用位置變更
function ZoneFullMap_applyPositionChanges(device, inputs, oldValues) {
    if (!inputs.x || !inputs.y) return false;
    
    const xValue = parseFloat(inputs.x.value);
    const yValue = parseFloat(inputs.y.value);
    
    // 驗證輸入值
    if (isNaN(xValue) || !isFinite(xValue) || isNaN(yValue) || !isFinite(yValue)) {
        //console.warn('Invalid position values:', { x: xValue, y: yValue });
        return false;
    }
    
    // 轉換為內部單位（0.1mm精度）
    const newX = Math.round(xValue * 10000);
    const newY = Math.round(yValue * 10000);
    
    // 檢查是否有變更
    if (newX === oldValues.x && newY === oldValues.y) {
        return false;
    }
    
    device.x = newX;
    device.y = newY;
    
    console.log('Position updated:', {
        deviceId: device.id,
        oldPos: { x: oldValues.x, y: oldValues.y },
        newPos: { x: newX, y: newY },
        inputValues: { x: xValue, y: yValue }
    });
    
    return true;
}

// 應用麥克風特定變更
function ZoneFullMap_applyMicrophoneChanges(device, inputs, oldValues) {
    let hasChanges = false;
    
    // 麥克風類型變更
    if (inputs.micType && inputs.micType.value !== oldValues.microphoneType) {
        device.microphoneType = inputs.micType.value;
        ZoneFullMap_updateMicrophoneIcon(device, inputs.micType.value);
        hasChanges = true;
        
        // 延遲重繪以確保圖標載入
        setTimeout(() => {
            ZoneFullMap_redrawQuadrilaterals();
            if (gSelectedDevice && gSelectedDevice.id === device.id) {
                ZoneFullMap_highlightSelectedDevice();
            }
        }, 50);
    }
    
    // IP 地址變更
    if (inputs.ip && inputs.ip.value.trim() !== oldValues.ip) {
        device.ip = inputs.ip.value.trim();
        hasChanges = true;
    }
    
    // 角度變更
    if (inputs.angle) {
        const newAngle = parseFloat(inputs.angle.value) || 0;
        const normalizedAngle = ((newAngle % 360) + 360) % 360;
        
        if (Math.abs(normalizedAngle - oldValues.angle) > 0.1) {
            device.angle = normalizedAngle;
            
            // 旋轉相關區域
            ZoneFullMap_rotateMicrophoneZones(
                device.id, 
                oldValues.angle, 
                device.angle, 
                device.x, 
                device.y
            );
            hasChanges = true;
        }
    }
    
    return hasChanges;
}

// 應用攝影機特定變更
function ZoneFullMap_applyCameraChanges(device, inputs, oldValues) {
    let hasChanges = false;
    
    // 方向變更（透過下拉選單）
    if (inputs.direction) {
        const newDirection = parseInt(inputs.direction.value) || 0;
        if (newDirection !== oldValues.angle) {
            device.angle = newDirection;
            hasChanges = true;
        }
    }
    
    // FOV 角度變更
    if (inputs.fov) {
        const newFov = Math.max(0, Math.min(360, parseInt(inputs.fov.value) || 0));
        if (newFov !== oldValues.fovAngle) {
            device.fovAngle = newFov;
            hasChanges = true;
        }
    }
    
    // 偏移角度變更
    if (inputs.offset) {
        const newOffset = Math.max(-360, Math.min(360, parseInt(inputs.offset.value) || 0));
        if (newOffset !== oldValues.offsetAngle) {
            device.offsetAngle = newOffset;
            hasChanges = true;
        }
    }
    
    // 清除渲染快取以強制重繪箭頭
    if (hasChanges) {
        delete device._cachedRenderData;
    }
    
    return hasChanges;
}

// 應用 BC-200 特定變更
function ZoneFullMap_applyVisionChanges(device, inputs, oldValues) {
    let hasChanges = false;
    
    // 方向變更（透過下拉選單）
    if (inputs.direction) {
        const newDirection = parseInt(inputs.direction.value) || 0;
        if (newDirection !== oldValues.angle) {
            device.angle = newDirection;
            hasChanges = true;
        }
    }

    // FOV 角度變更
    if (inputs.fov) {
        const newFov = Math.max(0, Math.min(360, parseInt(inputs.fov.value) || 0));
        if (newFov !== oldValues.fovAngle) {
            device.fovAngle = newFov;
            hasChanges = true;
        }
    }

    // 偏移角度變更
    if (inputs.offset) {
        const newOffset = Math.max(-360, Math.min(360, parseInt(inputs.offset.value) || 0));
        if (newOffset !== oldValues.offsetAngle) {
            device.offsetAngle = newOffset;
            hasChanges = true;
        }
    }

    if (inputs.offsetX && inputs.offsetY) 
    {
        const offsetXValue = parseFloat(inputs.offsetX.value);
        const offsetYValue = parseFloat(inputs.offsetY.value);
    
        if (isNaN(offsetXValue) || !isFinite(offsetXValue) || isNaN(offsetYValue) || !isFinite(offsetYValue)) 
        {
            return false;
        }
    
        const newOffsetX = offsetXValue;
        const newOffsetY = offsetYValue;

        if (newOffsetX !== oldValues.offsetX) 
        {
            device.offsetX = newOffsetX;
            hasChanges = true;
        }

        if (newOffsetY !== oldValues.offsetY) 
            {
            device.offsetY = newOffsetY;
            hasChanges = true;
        }
    }

    if (hasChanges) {
        delete device._cachedRenderData;
    }
    
    return hasChanges;
}

// 應用錨點特定變更
function ZoneFullMap_applyAnchorChanges(device, inputs, oldValues) {
    // 錨點目前沒有特殊屬性需要更新
    return false;
}

// 更新顯示
function ZoneFullMap_updateDisplayAfterChanges(device, oldValues) {
    // 如果位置有變更且是麥克風，同步區域位置
    if (device.type === 'microphone' && 
        (device.x !== oldValues.x || device.y !== oldValues.y)) {
        const deltaX = device.x - oldValues.x;
        const deltaY = device.y - oldValues.y;
        
        if (deltaX !== 0 || deltaY !== 0) {
            ZoneFullMap_syncDeviceZones(device.id, deltaX, deltaY);
        }
    }
    
    // 更新設備位置和重繪
    ZoneFullMap_updateDevicePositions();
    ZoneFullMap_redrawQuadrilaterals();
    
    // 保持設備選中狀態
    if (gSelectedDevice === device) {
        ZoneFullMap_highlightSelectedDevice();
    }
}

// 獲取設備當前值（用於日誌記錄）
function ZoneFullMap_getDeviceCurrentValues(device) {
    return {
        position: { x: device.x, y: device.y },
        angle: device.angle,
        fovAngle: device.fovAngle,
        offsetAngle: device.offsetAngle,
        offsetX: device.offsetX,
        offsetY: device.offsetY,
        microphoneType: device.microphoneType,
        ip: device.ip
    };
}

function ZoneFullMap_cancelDeviceChanges(deviceId) {
    const infoBox = document.getElementById('ZoneFullMap_device_info');
    if (!infoBox) return;

    const originalDataStr = infoBox.getAttribute('data-original');
    if (!originalDataStr) return;

    try {
        const originalData = JSON.parse(originalDataStr);
        
        // Restore input values to original
        const xInput = document.getElementById(`ZoneFullMap_x_${deviceId}`);
        const yInput = document.getElementById(`ZoneFullMap_y_${deviceId}`);
        const angleInput = document.getElementById(`ZoneFullMap_angle_${deviceId}`);
        const directionSelect = document.getElementById(`ZoneFullMap_direction_${deviceId}`);
        const fovInput = document.getElementById(`ZoneFullMap_fov_${deviceId}`);
        const offsetInput = document.getElementById(`ZoneFullMap_offset_${deviceId}`);
        const offsetXInput = document.getElementById(`ZoneFullMap_offsetX_${deviceId}`);
        const offsetYInput = document.getElementById(`ZoneFullMap_offsetY_${deviceId}`);
        const micTypeSelect = document.getElementById(`ZoneFullMap_micType_${deviceId}`);
        const ipInput = document.getElementById(`ZoneFullMap_ip_${deviceId}`);

        if (xInput) xInput.value = (originalData.x / 10000).toFixed(2);
        if (yInput) yInput.value = (originalData.y / 10000).toFixed(2);
        if (angleInput) angleInput.value = originalData.angle || 0;
        if (fovInput) fovInput.value = originalData.fovAngle || 0;
        if (offsetInput) offsetInput.value = originalData.offsetAngle || 0;
        if (offsetXInput) offsetXInput.value = originalData.offsetX || 0;
        if (offsetYInput) offsetYInput.value = originalData.offsetY || 0; 
        if (micTypeSelect) micTypeSelect.value = originalData.microphoneType || 'Shure_MXA310';
        if (ipInput) ipInput.value = originalData.ip || '';
        
        if (directionSelect) {
            const nearestDirection = ZoneFullMap_getNearestDirection(originalData.angle);
            directionSelect.value = nearestDirection;
        }

        console.log('Cancelled changes for device:', deviceId, 'Restored to:', originalData);
        
    } catch (error) {
        console.error('Error restoring original device data:', error);
    }
    
    ZoneFullMap_hideDeviceInfo();
}

function ZoneFullMap_rotateMicrophoneZones(microphoneId, oldAngle, newAngle, centerX, centerY) {
    // Calculate rotation difference in radians
    const angleDiff = -(newAngle - oldAngle) * Math.PI / 180;
    
    console.log('Rotating microphone zones:', {
        microphoneId,
        oldAngle: oldAngle.toFixed(1),
        newAngle: newAngle.toFixed(1),
        angleDiffDegrees: ((newAngle - oldAngle) % 360).toFixed(1),
        centerX: centerX.toFixed(1),
        centerY: centerY.toFixed(1)
    });

    // Find all zones associated with this microphone
    const associatedZones = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === microphoneId);
    
    if (associatedZones.length === 0) {
        console.log('No zones found for microphone:', microphoneId);
        return;
    }

    // Rotate each zone around the microphone center
    associatedZones.forEach(quad => {
        console.log(`Rotating zone ${quad.getDisplayZoneId()} around microphone center`);
        
        // Rotate each point of the quadrilateral
        quad.points.forEach((point, index) => {
            const oldPoint = { x: point.x, y: point.y };
            
            // Translate point to origin (relative to microphone center)
            const relativeX = point.x - centerX;
            const relativeY = point.y - centerY;
            
            // Apply rotation matrix
            const rotatedX = relativeX * Math.cos(angleDiff) - relativeY * Math.sin(angleDiff);
            const rotatedY = relativeX * Math.sin(angleDiff) + relativeY * Math.cos(angleDiff);
            
            // Translate back to world coordinates
            point.x = rotatedX + centerX;
            point.y = rotatedY + centerY;
            
            console.log(`  Point ${index}: (${oldPoint.x.toFixed(1)}, ${oldPoint.y.toFixed(1)}) → (${point.x.toFixed(1)}, ${point.y.toFixed(1)})`);
        });
        
        // Update relative position for future sync operations
        quad.updateRelativePosition();
    });
    
    console.log(`Rotated ${associatedZones.length} zones for microphone ${microphoneId}`);
}

function ZoneFullMap_updateDeviceInfoPosition(device) {
    const infoBox = document.getElementById('ZoneFullMap_device_info');
    if (!infoBox || !device) return;

    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const widthPx = device.baseWidthM * scale * 100;
    const cx = ox + device.x * scale / 100;
    const cy = oy - device.y * scale / 100;
    
    infoBox.style.left = `${cx + widthPx / 2 + 20}px`;
    infoBox.style.top = `${cy - 120}px`;

    // Update position values in the form
    const xInput = document.getElementById(`ZoneFullMap_x_${device.id}`);
    const yInput = document.getElementById(`ZoneFullMap_y_${device.id}`);
    const angleInput = document.getElementById(`ZoneFullMap_angle_${device.id}`);
    
    if (xInput && yInput) {
        xInput.value = (device.x / 10000).toFixed(2);
        yInput.value = (device.y / 10000).toFixed(2);
    }
    
    // Update angle for microphones
    if (angleInput && device.type === 'microphone') {
        angleInput.value = device.angle || 0;
    }
}

function ZoneFullMap_findDeviceById(deviceId) {
    return ZoneFullMap_devices.find(d => d.id === deviceId);
}

function ZoneFullMap_getNearestDirection(angle) {
    const directions = [0, 90, 180, 270];
    let minDiff = 360;
    let nearest = 0;
    
    for (const dir of directions) {
        const diff = Math.min(Math.abs(angle - dir), Math.abs(angle - dir + 360), Math.abs(angle - dir - 360));
        if (diff < minDiff) {
            minDiff = diff;
            nearest = dir;
        }
    }
    
    return nearest;
}


var ZoneFullMap_lastClickTime = 0;

function ZoneFullMap_hitTestZone(mouseX, mouseY) {

    for (let i = gZoneFullMap_Quadrilaterals.length - 1; i >= 0; i--) {
        const quad = gZoneFullMap_Quadrilaterals[i];
        if (quad.contains(mouseX, mouseY)) {
            return quad;
        }
    }
    return null;
}

function ZoneFullMap_showZoneSettings(zone, mouseX, mouseY) {
    
    ZoneFullMap_hideDeviceInfo();
    // ZoneFullMap_hideZoneSettings();
    
    const settingsBox = document.createElement('div');
    settingsBox.id = 'ZoneFullMap_zone_settings';
    settingsBox.style.position = 'absolute';
    settingsBox.style.backgroundColor = 'rgba(0, 14, 26, 1)';
    settingsBox.style.padding = '10px';
    settingsBox.style.borderRadius = '5px';
    settingsBox.style.zIndex = '101';
    settingsBox.style.boxShadow = '0 0 10px rgba(0,0,0,0.3)';
    settingsBox.style.color = 'rgba(137, 207, 216, 1)';
    settingsBox.style.fontFamily = 'Arial, sans-serif';
    settingsBox.style.pointerEvents = 'auto';
    settingsBox.style.minWidth = '280px';
    settingsBox.style.fontSize = '14px';
    settingsBox.style.left = `${mouseX + 20}px`;
    settingsBox.style.top = `${mouseY - 50}px`;
    
    const originalSettings = {
        primaryCamera: zone.primaryCamera || '',
        secondaryCamera: zone.secondaryCamera || '',
        aiSetting: zone.aiSetting || 'none'
    };
    
    let settingsHTML = `
        <div style="margin-bottom: 8px; border-bottom: 1px solid rgba(137, 207, 216, 0.3); padding-bottom: 6px;">
            <strong style="color: rgba(137, 207, 216, 1);">${zone.getDisplayZoneId()} 設定</strong>
        </div>
    `;
    
    settingsHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 100px; display: inline-block;">第一攝影機 :</span>
            <select id="ZoneSettings_primaryCamera_${zone.id}" 
                style="width: 150px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px;">
                <option value="">無</option>
                ${generateCameraOptions(zone.primaryCamera)}
            </select>
        </div>
    `;
    
    settingsHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 100px; display: inline-block;">第二攝影機 :</span>
            <select id="ZoneSettings_secondaryCamera_${zone.id}" 
                style="width: 150px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px;">
                <option value="">無</option>
                ${generateCameraOptions(zone.secondaryCamera)}
            </select>
        </div>
    `;
    
    settingsHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 100px; display: inline-block;">AI設定 :</span>
            <select id="ZoneSettings_aiSetting_${zone.id}" 
                style="width: 150px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px;">
                <option value="none" ${zone.aiSetting === 'none' ? 'selected' : ''}>無</option>
                <option value="center" ${zone.aiSetting === 'center' ? 'selected' : ''}>中心</option>
                <option value="continuous" ${zone.aiSetting === 'continuous' ? 'selected' : ''}>持續追蹤</option>
            </select>
        </div>
    `;
    
    settingsHTML += `
        <div style="display: flex; justify-content: space-between; margin-top: 10px; padding-top: 8px; border-top: 1px solid rgba(137, 207, 216, 0.3);">
            <div>
                <button onclick="ZoneFullMap_applyZoneSettings('${zone.id}')" 
                    style="background-color: #4CAF50; color: white; border: none; padding: 5px 12px; border-radius: 3px; cursor: pointer; margin-right: 8px;">
                    套用
                </button>
                <button onclick="ZoneFullMap_cancelZoneSettings('${zone.id}')" 
                    style="background-color: #f44336; color: white; border: none; padding: 5px 12px; border-radius: 3px; cursor: pointer;">
                    取消
                </button>
            </div>
            <button onclick="ZoneFullMap_hideZoneSettings()" 
                style="background-color: #666; color: white; border: none; padding: 5px 12px; border-radius: 3px; cursor: pointer;">
                關閉
            </button>
        </div>
    `;
    
    settingsBox.innerHTML = settingsHTML;
    settingsBox.setAttribute('data-zone-id', zone.id);
    settingsBox.setAttribute('data-original', JSON.stringify(originalSettings));
    
    const container = document.querySelector('.ZoneFullMap_canvas_td');
    if (container) {
        container.appendChild(settingsBox);
    }
    
    settingsBox.addEventListener('click', (e) => {
        e.stopPropagation();
    });
}

function generateCameraOptions(selectedCamera = '') {
    // 假設有一個全域的攝影機列表，類似原始程式碼中的 gConnectedCameras
    // 如果沒有，可以從 ZoneFullMap_devices 中找到攝影機類型的設備
    const cameras = ZoneFullMap_devices.filter(device => device.type === 'camera');
    
    let options = '';
    cameras.forEach(camera => {
        const selected = camera.id === selectedCamera ? 'selected' : '';
        options += `<option value="${camera.id}" ${selected}>${camera.name}</option>`;
    });
    
    return options;
}

function ZoneFullMap_isClickOnControlElements(e) {
    return e.target.classList.contains('ZoneFullMap_close_popup_btn') ||
           e.target.classList.contains('resize-handle') ||
           e.target.tagName === 'BUTTON' ||
           e.target.tagName === 'INPUT' ||
           e.target.tagName === 'SELECT' ||
           e.target.tagName === 'TEXTAREA' ||
           e.target.closest('.ZoneFullMap_calibration_view') ||
           e.target.closest('.ZoneFullMap_calibration_controls') ||
           e.target.closest('.ZoneFullMap_DrawZone_controls') ||
           e.target.closest('.ZoneFullMap_info_container') ||
           e.target.closest('#ZoneFullMap_device_info');
}

function ZoneFullMap_handleDrawingModeClick(mouseX, mouseY) {
    console.log('Handling drawing mode click at:', { x: mouseX, y: mouseY });
    const hitDevice = ZoneFullMap_hitTestDevice(mouseX, mouseY);
    if (hitDevice && hitDevice.type === 'microphone') {
        // 更新選擇的麥克風
        gZoneFullMap_SelectedMicrophone = hitDevice;
        gSelectedDevice = hitDevice;
        ZoneFullMap_highlightSelectedDevice();
        console.log('Switched to microphone in drawing mode:', hitDevice.name);
        return;
    }

    // 1. 檢查刪除按鈕點擊
    for (let i = 0; i < gZoneFullMap_Quadrilaterals.length; i++) {
        const quad = gZoneFullMap_Quadrilaterals[i];
        if (quad.isSelected && quad.isOnDeleteButton && quad.isOnDeleteButton(mouseX, mouseY)) {
            ZoneFullMap_deleteQuadrilateral(quad);
            return;
        }
    }

    // 2. 檢查角點拖拽
    for (let i = 0; i < gZoneFullMap_Quadrilaterals.length; i++) {
        const quad = gZoneFullMap_Quadrilaterals[i];
        if (quad.isSelected) {
            const cornerIndex = quad.isOnCorner(mouseX, mouseY);
            if (cornerIndex !== -1) {
                console.log('Starting corner drag:', { quadId: quad.id, corner: cornerIndex });
                
                // 記錄拖拽開始資訊
                gZoneFullMap_CornerDragStart = {
                    cornerIndex: cornerIndex,
                    startMouseX: mouseX,
                    startMouseY: mouseY,
                    startCornerMath: { ...quad.points[cornerIndex] } // 複製原始數學座標
                };
                
                gZoneFullMap_DraggingQuadrilateral = quad;
                gZoneFullMap_DraggingCornerIndex = cornerIndex;
                ZoneFullMap_addMouseEventListeners();
                return;
            }
        }
    }

    // 3. 檢查四邊形選擇和整體拖動
    gZoneFullMap_StartPoint = { x: mouseX, y: mouseY };
    gZoneFullMap_DraggingQuadrilateral = null;
    gZoneFullMap_DraggingCornerIndex = -1;
    gZoneFullMap_CornerDragStart = null;

    let selectedQuad = null;
    for (const quad of gZoneFullMap_Quadrilaterals) {
        if (quad.deviceId === gZoneFullMap_SelectedMicrophone.id && quad.contains(mouseX, mouseY)) {
            selectedQuad = quad;
            break;
        }
    }

    if (selectedQuad) {
        console.log('Selecting quad for whole drag:', selectedQuad.id);
        gZoneFullMap_Quadrilaterals.forEach(quad => quad.isSelected = false);
        selectedQuad.isSelected = true;
        gZoneFullMap_DraggingQuadrilateral = selectedQuad;
        gZoneFullMap_LastSelectedQuad = selectedQuad;
    } else {
        console.log('Starting new quad drawing');
        gZoneFullMap_IsDrawing = true;
        ZoneFullMap_clearQuadSelection();
    }

    ZoneFullMap_redrawQuadrilaterals();
    ZoneFullMap_addMouseEventListeners();
}

function ZoneFullMap_handleCanvasDrag(mouseX, mouseY) {
    //console.log('Starting canvas drag');
    // ZoneFullMap_hideDeviceInfo(); // 可以選擇保留設備信息
    ZoneFullMap_clearQuadSelection();
    // gSelectedDevice = null; // ❌ 移除這行
    gIsDraggingDevice = false;
    gDragSystem.isCanvasDragging = true;
    gDragSystem.hasStarted = false;

    gDragSystem.mouseStart.x = mouseX;
    gDragSystem.mouseStart.y = mouseY;
    gDragSystem.mouseCurrent.x = gDragSystem.mouseStart.x;
    gDragSystem.mouseCurrent.y = gDragSystem.mouseStart.y;

    ZoneFullMap_addMouseEventListeners();
    ZoneFullMap_autoCollapseControls();
}

function ZoneFullMap_addMouseEventListeners() {
    document.addEventListener('mousemove', ZoneFullMap_onCanvasMouseMove_WithQuad, { passive: true, capture: true });
    document.addEventListener('mouseup', ZoneFullMap_onCanvasMouseUp_WithQuad, { passive: true, capture: true });
}


function ZoneFullMap_handleDeviceDrag_Optimized(mouseX, mouseY) {
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    const oldX = gSelectedDevice.x;
    const oldY = gSelectedDevice.y;
    
    gSelectedDevice.x = (mouseX - ox) * 100 / scale;
    gSelectedDevice.y = (oy - mouseY) * 100 / scale;
    
    const deltaX = gSelectedDevice.x - oldX;
    const deltaY = gSelectedDevice.y - oldY;
    
    ZoneFullMap_syncDeviceZones(gSelectedDevice.id, deltaX, deltaY);
    
    // 使用節流重繪
    ZoneFullMap_throttledRedraw();
}

// function ZoneFullMap_handleCanvasDragMove_Optimized(mouseX, mouseY) {
//     gDragSystem.mouseCurrent.x = mouseX;
//     gDragSystem.mouseCurrent.y = mouseY;

//     if (!gDragSystem.hasStarted) {
//         const dx = gDragSystem.mouseCurrent.x - gDragSystem.mouseStart.x;
//         const dy = gDragSystem.mouseCurrent.y - gDragSystem.mouseStart.y;
//         const distance = Math.sqrt(dx * dx + dy * dy);
        
//         if (distance > gDragSystem.dragThreshold) {
//             gDragSystem.hasStarted = true;
//         }
//         return;
//     }

//     const dx = gDragSystem.mouseCurrent.x - gDragSystem.mouseStart.x;
//     const dy = gDragSystem.mouseCurrent.y - gDragSystem.mouseStart.y;
    
//     gDragSystem.canvasTransform.x += dx;
//     gDragSystem.canvasTransform.y += dy;

//     gDragSystem.mouseStart.x = gDragSystem.mouseCurrent.x;
//     gDragSystem.mouseStart.y = gDragSystem.mouseCurrent.y;

//     ZoneFullMap_fastCanvasRedraw();
// }

function ZoneFullMap_handleCanvasDragMove_Optimized(mouseX, mouseY) {
    gDragSystem.mouseCurrent.x = mouseX;
    gDragSystem.mouseCurrent.y = mouseY;

    if (!gDragSystem.hasStarted) {
        const dx = gDragSystem.mouseCurrent.x - gDragSystem.mouseStart.x;
        const dy = gDragSystem.mouseCurrent.y - gDragSystem.mouseStart.y;
        const distance = Math.sqrt(dx * dx + dy * dy);
        
        if (distance > gDragSystem.dragThreshold) {
            gDragSystem.hasStarted = true;
        }
        return;
    }

    const dx = gDragSystem.mouseCurrent.x - gDragSystem.mouseStart.x;
    const dy = gDragSystem.mouseCurrent.y - gDragSystem.mouseStart.y;
    
    // 更新座標前驗證
    const prevX = gDragSystem.canvasTransform.x;
    const prevY = gDragSystem.canvasTransform.y;
    
    gDragSystem.canvasTransform.x += dx;
    gDragSystem.canvasTransform.y += dy;
    
    // 驗證新座標
    if (isNaN(gDragSystem.canvasTransform.x) || !isFinite(gDragSystem.canvasTransform.x)) {
        gDragSystem.canvasTransform.x = prevX;
    }
    if (isNaN(gDragSystem.canvasTransform.y) || !isFinite(gDragSystem.canvasTransform.y)) {
        gDragSystem.canvasTransform.y = prevY;
    }

    gDragSystem.mouseStart.x = gDragSystem.mouseCurrent.x;
    gDragSystem.mouseStart.y = gDragSystem.mouseCurrent.y;

    // 使用驗證過的座標進行重繪
    const validTransform = gDragSystem.validateTransform ? 
        gDragSystem.validateTransform() : gDragSystem.canvasTransform;
    
    // 單一重繪呼叫，傳遞座標
    if (!gDragSystem._canvasUpdatePending) {
        gDragSystem._canvasUpdatePending = true;
        requestAnimationFrame(() => {
            gDragSystem.drawGrid();
            ZoneFullMap_updateDevicePositions(validTransform);
            if (gSelectedDevice) {
                ZoneFullMap_highlightSelectedDevice(validTransform);
            }
            ZoneFullMap_redrawQuadrilaterals(validTransform);
            gDragSystem._canvasUpdatePending = false;
        });
    }
}

function ZoneFullMap_fastDeviceRedraw() {
    ZoneFullMap_updateDevicePositions();
    ZoneFullMap_highlightSelectedDevice();
    ZoneFullMap_updateDeviceInfoPosition(gSelectedDevice);
    ZoneFullMap_redrawQuadrilaterals();
}

function ZoneFullMap_fastCanvasRedraw() {
    if (!gDragSystem._fastRedrawPending) {
        gDragSystem._fastRedrawPending = true;
        requestAnimationFrame(() => {
            gDragSystem.drawGrid();
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_redrawQuadrilaterals();
            if (gSelectedDevice) {
                ZoneFullMap_highlightSelectedDevice();
            }
            gDragSystem._fastRedrawPending = false;
        });
    }
}

function ZoneFullMap_updateDevicePositions(providedTransform = null) {
    if (!window.gZoneFullMap_updateBuffer) {
        window.gZoneFullMap_updateBuffer = {
            queue: [],
            maxSize: 3,
            isProcessing: false,
            lastProcessTime: 0,
            minInterval: 16,
            scheduledRaf: false
        };
    }

    const currentTransform = providedTransform || gDragSystem.canvasTransform;
    
    if (!currentTransform || isNaN(currentTransform.x) || isNaN(currentTransform.y) || isNaN(currentTransform.scale)) {
        return;
    }
    
    const { x: ox, y: oy, scale } = currentTransform;
    
    const canvases = {
        mic: document.getElementById('ZoneFullMap_xy_canvas_mic'),
        camera: document.getElementById('ZoneFullMap_xy_canvas_camera'),      
        bc200: document.getElementById('ZoneFullMap_xy_canvas_bc200'),        
        anchor: document.getElementById('ZoneFullMap_xy_canvas_anchor'),
        soundPoint: document.getElementById('ZoneFullMap_xy_canvas_soundPoint')
    };

    if (!canvases.mic || !canvases.camera || !canvases.bc200 || !canvases.anchor || !canvases.soundPoint) {
        return;
    }

    const contexts = {
        mic: canvases.mic.getContext('2d'),
        camera: canvases.camera.getContext('2d'),
        bc200: canvases.bc200.getContext('2d'),
        anchor: canvases.anchor.getContext('2d'),
        soundPoint: canvases.soundPoint.getContext('2d')
    };

    const w = canvases.mic.width;
    const h = canvases.mic.height;

    Object.values(contexts).forEach(ctx => ctx.clearRect(0, 0, w, h));

    const devicesByType = {
        microphone: [],
        camera: [],
        vision: [],
        anchor: []
    };

    ZoneFullMap_devices.forEach(device => {
        if (gZoneFullMap_deviceVisibility && gZoneFullMap_deviceVisibility[device.id] === false) {
            return;
        }
        
        if (devicesByType[device.type]) {
            devicesByType[device.type].push(device);
        }
    });

    devicesByType.camera.forEach(device => {
        const config = ZoneFullMap_DeviceVisualConfig.camera;
        const widthPx = config.baseCircleSize * scale * 100;
        const cx = ox + device.x * scale / 100;
        const cy = oy - device.y * scale / 100;        
        ZoneFullMap_drawCameraArrows(contexts.camera, device, cx, cy, widthPx, config);
        ZoneFullMap_drawCameraCircle(contexts.camera, device, cx, cy, widthPx, config);
    });

    devicesByType.vision.forEach(device => {
        const config = ZoneFullMap_DeviceVisualConfig.bc200;
        const widthPx = config.baseCircleSize * scale * 100;
        const cx = ox + device.x * scale / 100;
        const cy = oy - device.y * scale / 100;
        ZoneFullMap_drawBC200Arrow(contexts.bc200, device, cx, cy, widthPx, config);
        ZoneFullMap_drawBC200Circle(contexts.bc200, device, cx, cy, widthPx, config);
    });

    devicesByType.microphone.forEach(device => {
        const widthPx = device.baseWidthM * scale * 100;
        const cx = ox + device.x * scale / 100;
        const cy = oy - device.y * scale / 100;
        ZoneFullMap_drawMicrophoneOnCanvas(contexts.mic, device, cx, cy, widthPx);
    });

    devicesByType.anchor.forEach(device => {
        const widthPx = device.baseWidthM * scale * 100;
        const cx = ox + device.x * scale / 100;
        const cy = oy - device.y * scale / 100;
        ZoneFullMap_trackAnchorMovement(device, cx, cy);
        ZoneFullMap_drawAnchorBasic(contexts.anchor, device, cx, cy, widthPx);
    });

    ZoneFullMap_manageAnchorBreathingAnimation();
    
}

function ZoneFullMap_drawCameraArrows(ctx, device, cx, cy, widthPx, config) {
    ctx.save();
    
    const colorScheme = ZoneFullMap_getCameraColorScheme(device.id);
    const needsUpdate = ZoneFullMap_RenderCache.angleChanged(
        device.id, 
        device.angle || 0, 
        device.fovAngle || 0, 
        device.offsetAngle || 0
    );
    
    if (needsUpdate || !device._cachedRenderData) {
        const hardwareAngle = ZoneFullMap_normalizeAngle(device.angle || 0);
        
        const fovAngle = ZoneFullMap_normalizeAngle(hardwareAngle + (device.fovAngle || 0) + (device.offsetAngle || 0));
        
        device._cachedRenderData = {
            hardwareRad: ZoneFullMap_angleToRadians(device.fovAngle),
            fovRad: ZoneFullMap_angleToRadians(fovAngle),
            hardwareAngle: hardwareAngle,
            fovAngle: fovAngle,
            offsetAngle: device.offsetAngle || 0
        };
    }
    
    const cache = device._cachedRenderData;
    const arrowLength = Math.max(config.arrow.baseLength, widthPx * config.arrow.lengthMultiplier);
    const arrowHeadSize = Math.max(8, widthPx * config.arrow.headSizeRatio);

    const hardwareArrowEndX = cx + arrowLength * Math.cos(cache.hardwareRad);
    const hardwareArrowEndY = cy + arrowLength * Math.sin(cache.hardwareRad);

    ctx.setLineDash(config.arrow.dashPattern);
    ctx.strokeStyle = 'black';
    ctx.lineWidth = config.arrow.hardwareWidth;
    ctx.beginPath();
    ctx.moveTo(cx, cy);
    ctx.lineTo(hardwareArrowEndX, hardwareArrowEndY);
    ctx.stroke();
    
    ZoneFullMap_drawArrowHead(ctx, hardwareArrowEndX, hardwareArrowEndY, cache.hardwareRad, arrowHeadSize, config.border.color);

    const fovArrowEndX = cx + arrowLength * Math.cos(cache.fovRad);
    const fovArrowEndY = cy + arrowLength * Math.sin(cache.fovRad);

    ctx.setLineDash([]);
    ctx.strokeStyle = colorScheme.primary;
    ctx.lineWidth = config.arrow.fovWidth;
    ctx.beginPath();
    ctx.moveTo(cx, cy);
    ctx.lineTo(fovArrowEndX, fovArrowEndY);
    ctx.stroke();
    
    ZoneFullMap_drawArrowHead(ctx, fovArrowEndX, fovArrowEndY, cache.fovRad, arrowHeadSize + 2, colorScheme.primary);
    
    ctx.restore();
}

function ZoneFullMap_drawCameraCircle(ctx, device, cx, cy, widthPx, config) {
    //console.log('ZoneFullMap_drawCameraCircle device.id : ',device.id);
    const colorScheme = ZoneFullMap_getCameraColorScheme(device.id);
    ctx.save();
    
    ctx.beginPath();
    ctx.arc(cx, cy, widthPx / 2, 0, 2 * Math.PI);
    ctx.fillStyle = colorScheme.primary;
    ctx.fill();
    
    ctx.restore();
}


function ZoneFullMap_drawBC200Arrow(ctx, device, cx, cy, widthPx, config) {
    ctx.save();
    const colorScheme = ZoneFullMap_getVisionColorScheme(device.id);
    const needsUpdate = ZoneFullMap_RenderCache.angleChanged(
        device.id, 
        device.angle || 0, 
        device.fovAngle || 0, 
        device.offsetAngle || 0
    );
    
    if (needsUpdate || !device._cachedAngle) 
    {
        const hardwareAngle = ZoneFullMap_normalizeAngle(device.angle || 0);
        const fovAngle = device.fovAngle || 0;
        const offsetAngle = device.offsetAngle || 0;
        
        const actualAngle = ZoneFullMap_normalizeAngle(hardwareAngle + fovAngle + offsetAngle);
        
        device._cachedAngle = {
            hardwareRad: ZoneFullMap_angleToRadians(fovAngle),//
            actualRad: ZoneFullMap_angleToRadians(actualAngle),//
            hasFov: fovAngle !== 0,
            hasOffset: offsetAngle !== 0
        };
    }
    
    const cache = device._cachedAngle;
    const arrowLength = Math.max(config.arrow.baseLength, widthPx * config.arrow.lengthMultiplier);
    const arrowHeadSize = Math.max(10, widthPx * config.arrow.headSizeRatio);
    const isSelected = gSelectedDevice === device;

    // 硬體方向箭頭（虛線）
    const hardwareArrowEndX = cx + arrowLength * Math.cos(cache.hardwareRad);
    const hardwareArrowEndY = cy + arrowLength * Math.sin(cache.hardwareRad);

    ctx.setLineDash([5, 3]);
    ctx.strokeStyle = 'black';
    ctx.lineWidth = config.arrow.width - 1;
    ctx.beginPath();
    ctx.moveTo(cx, cy);
    ctx.lineTo(hardwareArrowEndX, hardwareArrowEndY);
    ctx.stroke();
    
    ZoneFullMap_drawArrowHead(ctx, hardwareArrowEndX, hardwareArrowEndY, cache.hardwareRad, arrowHeadSize - 2, config.border.color || '#666');

    // 實際方向箭頭（實線）- 根據選中狀態決定顏色
    const actualArrowEndX = cx + arrowLength * Math.cos(cache.actualRad);
    const actualArrowEndY = cy + arrowLength * Math.sin(cache.actualRad);

    ctx.setLineDash([]);
    ctx.strokeStyle = colorScheme.primary;
    ctx.lineWidth = config.arrow.width;
    ctx.beginPath();
    ctx.moveTo(cx, cy);
    ctx.lineTo(actualArrowEndX, actualArrowEndY);
    ctx.stroke();

    ZoneFullMap_drawArrowHead(ctx, actualArrowEndX, actualArrowEndY, cache.actualRad, arrowHeadSize, colorScheme.primary);
    
    ctx.restore();
}

function ZoneFullMap_drawBC200Circle(ctx, device, cx, cy, widthPx, config) {
    const colorScheme = ZoneFullMap_getVisionColorScheme(device.id);
    ctx.save();
    
    const rectWidth = widthPx;
    const rectHeight = widthPx * 1.2;
    const borderRadius = widthPx * 0.2;

    const needsUpdate = ZoneFullMap_RenderCache.angleChanged(
        device.id, 
        device.angle || 0, 
        device.fovAngle || 0, 
        device.offsetAngle || 0
    );

    if (needsUpdate || !device._cachedAngle) 
    {
        const hardwareAngle = ZoneFullMap_normalizeAngle(device.angle || 0);
        const offsetAngle = device.offsetAngle || 0;
        
        const actualAngle = ZoneFullMap_normalizeAngle(hardwareAngle + offsetAngle);
        
        device._cachedAngle = {
            hardwareRad: ZoneFullMap_angleToRadians(hardwareAngle),
            actualRad: ZoneFullMap_angleToRadians(actualAngle),
            hasFov: fovAngle !== 0,
            hasOffset: offsetAngle !== 0,
            adjustedDirection: adjustedDirection,
            totalAngle: actualAngle
        };
    }
    
    const cache = device._cachedAngle;
    
    ctx.translate(cx, cy);
    ctx.rotate(cache.actualRad);
    
    const x = -rectWidth / 2;
    const y = -rectHeight / 2;
    
    ctx.beginPath();
    ctx.moveTo(x + borderRadius, y);
    ctx.lineTo(x + rectWidth - borderRadius, y);
    ctx.quadraticCurveTo(x + rectWidth, y, x + rectWidth, y + borderRadius);
    ctx.lineTo(x + rectWidth, y + rectHeight - borderRadius);
    ctx.quadraticCurveTo(x + rectWidth, y + rectHeight, x + rectWidth - borderRadius, y + rectHeight);
    ctx.lineTo(x + borderRadius, y + rectHeight);
    ctx.quadraticCurveTo(x, y + rectHeight, x, y + rectHeight - borderRadius);
    ctx.lineTo(x, y + borderRadius);
    ctx.quadraticCurveTo(x, y, x + borderRadius, y);
    ctx.closePath();
    
    ctx.fillStyle = colorScheme.primary;
    ctx.fill();
    
    ctx.restore();
}


function ZoneFullMap_drawMicrophoneOnCanvas(ctx, device, cx, cy, widthPx) {
    
    ctx.save();
    
    const colorScheme = ZoneFullMap_getMicrophoneColorScheme(device.id);
    const borderSize = widthPx + 16;
    const borderX = cx - borderSize / 2;
    const borderY = cy - borderSize / 2;
    
    ctx.setLineDash([8, 4]);
    ctx.beginPath();
    ctx.rect(borderX, borderY, borderSize, borderSize);
    ctx.strokeStyle = colorScheme.primary;
    ctx.lineWidth = 3;
    ctx.stroke();
    ctx.setLineDash([]);
    
    let imageDrawn = false;
    
    if (device._preloadedImage && device.imageLoaded) {
        ctx.translate(cx, cy);
        ctx.rotate((device.angle * Math.PI) / 180);
        
        ctx.imageSmoothingEnabled = true;
        ctx.imageSmoothingQuality = 'high';
        
        ctx.drawImage(
            device._preloadedImage, 
            -widthPx / 2, 
            -widthPx / 2, 
            widthPx, 
            widthPx
        );
        
        imageDrawn = true;
        ctx.restore();
    }
    else if (device.imageElement && device.imageElement.complete && device.imageElement.naturalWidth > 0) {
        ctx.translate(cx, cy);
        ctx.rotate((device.angle * Math.PI) / 180);
        
        ctx.imageSmoothingEnabled = true;
        ctx.imageSmoothingQuality = 'high';
        
        ctx.drawImage(
            device.imageElement, 
            -widthPx / 2, 
            -widthPx / 2, 
            widthPx, 
            widthPx
        );
        
        imageDrawn = true;
        ctx.restore();
    }
}

function ZoneFullMap_drawAnchorBasic(ctx, device, cx, cy, widthPx) {
    ctx.beginPath();
    ctx.arc(cx, cy, widthPx / 2, 0, 2 * Math.PI);
    ctx.fillStyle = 'red';
    ctx.fill();
    ctx.closePath();
}

function ZoneFullMap_trackAnchorMovement(device, cx, cy) {
    if (!device || device.type !== 'anchor') return;
    
    const deviceId = device.id;
    const currentTime = Date.now();
    
    if (!gAnchorMovementTracker.positions[deviceId]) {
        gAnchorMovementTracker.positions[deviceId] = { x: cx, y: cy, timestamp: currentTime };
        gAnchorMovementTracker.isMoving[deviceId] = false;
        gAnchorMovementTracker.lastMovingStates[deviceId] = false;
        return;
    }
    
    const lastPos = gAnchorMovementTracker.positions[deviceId];
    const deltaX = cx - lastPos.x;
    const deltaY = cy - lastPos.y;
    const distance = Math.sqrt(deltaX * deltaX + deltaY * deltaY);
    
    const wasMoving = gAnchorMovementTracker.isMoving[deviceId];
    
    if (distance > gAnchorMovementTracker.movementThreshold) {
        gAnchorMovementTracker.isMoving[deviceId] = true;
        gAnchorMovementTracker.positions[deviceId] = { x: cx, y: cy, timestamp: currentTime };
        
        if (!wasMoving) {
            ZoneFullMap_manageAnchorBreathingAnimation();
        }
        
        clearTimeout(gAnchorMovementTracker[`timeout_${deviceId}`]);
        gAnchorMovementTracker[`timeout_${deviceId}`] = setTimeout(() => {
            ZoneFullMap_setAnchorStationary(deviceId);
        }, gAnchorMovementTracker.stabilityTime);
    }
}

function ZoneFullMap_drawArrowHead(ctx, endX, endY, angle, size, color) {
    ctx.save();
    ctx.strokeStyle = color;
    ctx.lineWidth = 3;
    
    const arrowAngle = Math.PI / 6;
    
    const arrowP1 = {
        x: endX - Math.cos(angle + arrowAngle) * size,
        y: endY - Math.sin(angle + arrowAngle) * size
    };
    const arrowP2 = {
        x: endX - Math.cos(angle - arrowAngle) * size,
        y: endY - Math.sin(angle - arrowAngle) * size
    };
    
    ctx.beginPath();
    ctx.moveTo(endX, endY);
    ctx.lineTo(arrowP1.x, arrowP1.y);
    ctx.moveTo(endX, endY);
    ctx.lineTo(arrowP2.x, arrowP2.y);
    ctx.stroke();
    
    ctx.restore();
}

function ZoneFullMap_handleDrawingModeMove(mouseX, mouseY) {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return;

    if (gZoneFullMap_DraggingQuadrilateral) {
        if (gZoneFullMap_DraggingCornerIndex !== -1) {
            //console.log(`Moving corner ${gZoneFullMap_DraggingCornerIndex} to screen (${mouseX}, ${mouseY})`);
            
            if (gZoneFullMap_CornerDragStart) {
                const deltaScreenX = mouseX - gZoneFullMap_CornerDragStart.startMouseX;
                const deltaScreenY = mouseY - gZoneFullMap_CornerDragStart.startMouseY;
                
                const originalScreenPos = gZoneFullMap_DraggingQuadrilateral.mathToScreen(
                    gZoneFullMap_CornerDragStart.startCornerMath.x,
                    gZoneFullMap_CornerDragStart.startCornerMath.y
                );
                
                const newScreenX = originalScreenPos.x + deltaScreenX;
                const newScreenY = originalScreenPos.y + deltaScreenY;
                
                const constrained = gZoneFullMap_DraggingQuadrilateral.constrainCornerToScreen(
                    gZoneFullMap_DraggingCornerIndex, newScreenX, newScreenY
                );
                
                const newMathPos = gZoneFullMap_DraggingQuadrilateral.screenToMath(
                    constrained.x, constrained.y
                );
                
                gZoneFullMap_DraggingQuadrilateral.points[gZoneFullMap_DraggingCornerIndex].x = newMathPos.x;
                gZoneFullMap_DraggingQuadrilateral.points[gZoneFullMap_DraggingCornerIndex].y = newMathPos.y;
            }
            
        } else {
            const deltaScreenX = mouseX - gZoneFullMap_StartPoint.x;
            const deltaScreenY = mouseY - gZoneFullMap_StartPoint.y;
            
            const { scale } = gDragSystem.canvasTransform;
            const deltaMathX = deltaScreenX * 100 / scale;
            const deltaMathY = -deltaScreenY * 100 / scale;
            
            let canMove = true;
            const tempPoints = gZoneFullMap_DraggingQuadrilateral.points.map(point => {
                const newMathPos = { x: point.x + deltaMathX, y: point.y + deltaMathY };
                const newScreenPos = gZoneFullMap_DraggingQuadrilateral.mathToScreen(newMathPos.x, newMathPos.y);
                
                if (newScreenPos.x < 10 || newScreenPos.x > canvas.width - 10 ||
                    newScreenPos.y < 10 || newScreenPos.y > canvas.height - 10) {
                    canMove = false;
                }
                return newMathPos;
            });

            if (canMove) {
                gZoneFullMap_DraggingQuadrilateral.points.forEach((point, i) => {
                    point.x = tempPoints[i].x;
                    point.y = tempPoints[i].y;
                });
                gZoneFullMap_StartPoint.x = mouseX;
                gZoneFullMap_StartPoint.y = mouseY;
            }
        }
        
        ZoneFullMap_redrawQuadrilaterals();
        
    } else if (gZoneFullMap_IsDrawing && gZoneFullMap_SelectedMicrophone) {
        ZoneFullMap_redrawQuadrilaterals();
        
        const rectLeft = Math.min(gZoneFullMap_StartPoint.x, mouseX);
        const rectTop = Math.min(gZoneFullMap_StartPoint.y, mouseY);
        const rectRight = Math.max(gZoneFullMap_StartPoint.x, mouseX);
        const rectBottom = Math.max(gZoneFullMap_StartPoint.y, mouseY);
        
        const previewWidth = rectRight - rectLeft;
        const previewHeight = rectBottom - rectTop;

        if (previewWidth > 5 && previewHeight > 5) {
            ZoneFullMap_drawZonePreview(
                canvas.getContext('2d'), 
                gZoneFullMap_StartPoint.x, gZoneFullMap_StartPoint.y,
                mouseX, mouseY,
                gZoneFullMap_SelectedMicrophone
            );
            
            console.log(`Drawing preview from (${gZoneFullMap_StartPoint.x}, ${gZoneFullMap_StartPoint.y}) to (${mouseX}, ${mouseY})`);
            console.log(`Preview bounds: (${rectLeft}, ${rectTop}) size (${previewWidth}x${previewHeight})`);
        }
    }
}


function ZoneFullMap_constrainToBounds(x, y) {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return { x, y };
    
    return {
        x: Math.max(0, Math.min(canvas.width, x)),
        y: Math.max(0, Math.min(canvas.height, y))
    };
}

function ZoneFullMap_adjustRectangleBounds(startX, startY, width, height) {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return { x: startX, y: startY, width, height };
    
    const finalX = Math.max(0, Math.min(startX, startX + width));
    const finalY = Math.max(0, Math.min(startY, startY + height));
    const finalWidth = Math.min(canvas.width - finalX, Math.abs(width));
    const finalHeight = Math.min(canvas.height - finalY, Math.abs(height));
    
    return { x: finalX, y: finalY, width: finalWidth, height: finalHeight };
}


function ZoneFullMap_handleDeviceDrag(mouseX, mouseY) {
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const newScreenX = mouseX;
    const newScreenY = mouseY;
    
    const oldX = gSelectedDevice.x;
    const oldY = gSelectedDevice.y;
    
    gSelectedDevice.x = (newScreenX - ox) * 100 / scale;
    gSelectedDevice.y = (oy - newScreenY) * 100 / scale;
    
    const deltaX = gSelectedDevice.x - oldX;
    const deltaY = gSelectedDevice.y - oldY;
    
    ZoneFullMap_syncDeviceZones(gSelectedDevice.id, deltaX, deltaY);
    ZoneFullMap_updateDeviceInfoPosition(gSelectedDevice);
}

function ZoneFullMap_syncDeviceZones(deviceId, deltaX, deltaY) {
    gZoneFullMap_Quadrilaterals.forEach(quad => {
        if (quad.deviceId === deviceId) {
            // Move all points of the quadrilateral
            quad.points.forEach(point => {
                point.x += deltaX;
                point.y += deltaY;
            });
            
            // Update relative position
            quad.updateRelativePosition();
            
            //console.log(`Synced zone ${quad.id} with device movement: (${deltaX.toFixed(1)}, ${deltaY.toFixed(1)})`);
        }
    });
}

function ZoneFullMap_handleCanvasDragMove(mouseX, mouseY) 
{
    gDragSystem.mouseCurrent.x = mouseX;
    gDragSystem.mouseCurrent.y = mouseY;

    if (!gDragSystem.hasStarted) {
        const dx = gDragSystem.mouseCurrent.x - gDragSystem.mouseStart.x;
        const dy = gDragSystem.mouseCurrent.y - gDragSystem.mouseStart.y;
        const distance = Math.sqrt(dx * dx + dy * dy);
        
        if (distance > gDragSystem.dragThreshold) {
            gDragSystem.hasStarted = true;
        }
        return;
    }

    const dx = gDragSystem.mouseCurrent.x - gDragSystem.mouseStart.x;
    const dy = gDragSystem.mouseCurrent.y - gDragSystem.mouseStart.y;
    
    gDragSystem.canvasTransform.x += dx;
    gDragSystem.canvasTransform.y += dy;

    gDragSystem.mouseStart.x = gDragSystem.mouseCurrent.x;
    gDragSystem.mouseStart.y = gDragSystem.mouseCurrent.y;

    requestAnimationFrame(() => {
        gDragSystem.drawGrid();
        ZoneFullMap_updateDevicePositions();
        ZoneFullMap_redrawQuadrilaterals();
        if (gSelectedDevice) {
            ZoneFullMap_highlightSelectedDevice();
        }
    });
}

function ZoneFullMap_onCanvasMouseDown_WithQuad(e) {
    const containerRect = e.currentTarget.getBoundingClientRect();
    const isInCanvas = e.clientX >= containerRect.left && e.clientX <= containerRect.right &&
                      e.clientY >= containerRect.top && e.clientY <= containerRect.bottom;
    if (!isInCanvas) return;
    if (ZoneFullMap_isClickOnControlElements(e)) {
        return;
    }
    const mouseX = e.clientX - containerRect.left;
    const mouseY = e.clientY - containerRect.top;
    const isRightClick = e.button === 2;
    if (isRightClick) {
        const hitZone = ZoneFullMap_hitTestZone(mouseX, mouseY);
        // if (hitZone) {
        //     ZoneFullMap_showZoneSettings(hitZone, mouseX, mouseY);
        //     e.preventDefault();
        //     return;
        // }
    
        const hitDevice = ZoneFullMap_hitTestDevice(mouseX, mouseY, true);
        
        if (hitDevice && gZoneFullMap_deviceVisibility[hitDevice.id] !== false) 
        {
            const existingInfoBox = document.getElementById('ZoneFullMap_device_info');
            
            if (existingInfoBox && gSelectedDevice && gSelectedDevice.id === hitDevice.id)
            {
                ZoneFullMap_hideDeviceInfo();
            } 
            else 
            {
                if (existingInfoBox) 
                {
                    ZoneFullMap_hideDeviceInfo();
                }
                gSelectedDevice = hitDevice;
                ZoneFullMap_showDeviceInfo(hitDevice);
                ZoneFullMap_highlightSelectedDevice();
                window.ZoneFullMap_lastRightClickDevice = hitDevice.id;
            }
        } 
        else 
        {
            ZoneFullMap_hideDeviceInfo();
        }
        e.preventDefault();
        return;
    }
    if (gZoneFullMap_DrawingMode && gZoneFullMap_SelectedMicrophone) {
        ZoneFullMap_handleDrawingModeClick(mouseX, mouseY);
        return;
    }
    const hitDevice = ZoneFullMap_hitTestDevice(mouseX, mouseY);
    
    if (hitDevice && gZoneFullMap_deviceVisibility[hitDevice.id] !== false) {
        if (gSelectedDevice && gSelectedDevice.id === hitDevice.id) {
            const now = Date.now();
            const timeSinceLastClick = now - (ZoneFullMap_lastClickTime || 0);
            
            if (timeSinceLastClick < 300) {
                gSelectedDevice = null;
                ZoneFullMap_clearDeviceHighlight();
                ZoneFullMap_hideDeviceInfo();
                console.log('Double-click: Deselected device:', hitDevice.name);
                return;
            }
            
            ZoneFullMap_lastClickTime = now;
        } else {
            gSelectedDevice = hitDevice;
            ZoneFullMap_lastClickTime = Date.now();
        }
        
        ZoneFullMap_handleDeviceClick(hitDevice, mouseX, mouseY);
    } else {
        ZoneFullMap_handleCanvasDrag(mouseX, mouseY);
    }
    e.preventDefault();
    e.stopPropagation();
}

function ZoneFullMap_onCanvasMouseMove_WithQuad(e) 
{
    if (!gIsDraggingDevice && !gDragSystem.isCanvasDragging &&
        !gZoneFullMap_DraggingQuadrilateral && !gZoneFullMap_IsDrawing) return;

    const now = Date.now();
    if (now - gDragSystem.lastRedrawTime < 33) return;
    gDragSystem.lastRedrawTime = now;

    const containerRect = document.querySelector('.ZoneFullMap_canvas_td').getBoundingClientRect();
    const mouseX = e.clientX - containerRect.left;
    const mouseY = e.clientY - containerRect.top;

    if (gZoneFullMap_DrawingMode) {
        ZoneFullMapRenderController.startMoving(); 
        ZoneFullMap_handleDrawingModeMove(mouseX, mouseY);
        return;
    }

    if (gIsDraggingDevice && gSelectedDevice) {
        ZoneFullMapRenderController.startDraggingDevice();
        
        ZoneFullMap_handleDeviceDrag_Optimized(mouseX, mouseY);

        if (!gDragSystem._deviceDragPending) {
            gDragSystem._deviceDragPending = true;
            ZoneFullMapRenderController.requestGridRedraw();
            gDragSystem._deviceDragPending = false;
        }
        return;
    }

    if (gDragSystem.isCanvasDragging) {
        ZoneFullMap_handleCanvasDragMove_Optimized(mouseX, mouseY);
    }
}

function ZoneFullMap_onCanvasMouseUp_WithQuad(e)
{
    const containerRect = document.querySelector('.ZoneFullMap_canvas_td').getBoundingClientRect();
    const mouseX = e.clientX - containerRect.left;
    const mouseY = e.clientY - containerRect.top;

    console.log('Mouse up at:', { 
        x: mouseX, 
        y: mouseY,
        isDrawing: gZoneFullMap_IsDrawing,
        selectedMicrophone: gZoneFullMap_SelectedMicrophone ? gZoneFullMap_SelectedMicrophone.id : 'none'
    });

    if (gZoneFullMap_IsDrawing && gZoneFullMap_SelectedMicrophone) {
        const width = mouseX - gZoneFullMap_StartPoint.x;
        const height = mouseY - gZoneFullMap_StartPoint.y;
        
        if (Math.abs(width) > 10 && Math.abs(height) > 10) {
            const startX = gZoneFullMap_StartPoint.x;
            const startY = gZoneFullMap_StartPoint.y;
            const endX = mouseX;
            const endY = mouseY;
            
            const screenPoints = [
                { x: startX, y: startY },                    // 起始點（固定角）
                { x: endX, y: startY },                      // 右上或左上
                { x: endX, y: endY },                        // 對角點
                { x: startX, y: endY }                       // 左下或右下
            ];
            
            // console.log('Creating zone with fixed start point:', {
            //     startPoint: { x: startX, y: startY },
            //     endPoint: { x: endX, y: endY },
            //     width: Math.abs(width),
            //     height: Math.abs(height),
            //     direction: {
            //         horizontal: width > 0 ? 'right' : 'left',
            //         vertical: height > 0 ? 'down' : 'up'
            //     }
            // });
            
            const newQuad = ZoneFullMap_createQuadrilateralForDevice(screenPoints, gZoneFullMap_SelectedMicrophone.id);
            
            if (newQuad) {
                // console.log('Zone created successfully:', {
                //     displayId: newQuad.getDisplayZoneId(),
                //     actualId: newQuad.id,
                //     deviceId: newQuad.deviceId,
                //     points: screenPoints
                // });
            } else {
                //console.warn('Failed to create zone');
            }
        }
        
        gZoneFullMap_IsDrawing = false;
        //console.log('Set gZoneFullMap_IsDrawing = false');
    }

    if (gIsDraggingDevice) {
        //console.log('Finished dragging device');
        gIsDraggingDevice = false;
        ZoneFullMapRenderController.stopMoving();
    }
    
    if (gDragSystem.isCanvasDragging) {
        gDragSystem.isCanvasDragging = false;
        gDragSystem.hasStarted = false;
        gDragSystem.saveCurrentState();
    }

    gZoneFullMap_DraggingQuadrilateral = null;
    gZoneFullMap_DraggingCornerIndex = -1;
    gZoneFullMap_CornerDragStart = null;
    
    document.removeEventListener('mousemove', ZoneFullMap_onCanvasMouseMove_WithQuad, { capture: true });
    document.removeEventListener('mouseup', ZoneFullMap_onCanvasMouseUp_WithQuad, { capture: true });

    //console.log('All states cleared and event listeners removed');
    
    requestAnimationFrame(() => {
        ZoneFullMap_redrawQuadrilaterals();
        if (gSelectedDevice) {
            ZoneFullMap_highlightSelectedDevice();
            console.log('Maintained highlight for selected device:', gSelectedDevice.name);
        }
    });
}


function ZoneFullMap_debugCoordinateTransform(screenX, screenY) 
{
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    const mathX = (screenX - ox) * 100 / scale;
    const mathY = (oy - screenY) * 100 / scale;
    
    const verifyScreenX = ox + mathX * scale / 100;
    const verifyScreenY = oy - mathY * scale / 100;
    
    // console.log('Coordinate transform debug:', {
    //     input: { screenX, screenY },
    //     canvasTransform: { ox, oy, scale },
    //     mathCoords: { mathX: mathX.toFixed(2), mathY: mathY.toFixed(2) },
    //     verification: { 
    //         screenX: verifyScreenX.toFixed(2), 
    //         screenY: verifyScreenY.toFixed(2) 
    //     },
    //     error: {
    //         x: Math.abs(screenX - verifyScreenX).toFixed(3),
    //         y: Math.abs(screenY - verifyScreenY).toFixed(3)
    //     }
    // });
}

function ZoneFullMap_updateSelectedQuadInfo(quad) 
{
    let area = 0;
    for (let i = 0; i < quad.points.length; i++) {
        const j = (i + 1) % quad.points.length;
        area += quad.points[i].x * quad.points[j].y;
        area -= quad.points[j].x * quad.points[i].y;
    }
    area = Math.abs(area) / 2;
    
    console.log(`Updated quad ${quad.id} info - Area: ${area.toFixed(2)}`);
}

function ZoneFullMap_toggleDrawingMode(microphone) {
    if (gZoneFullMap_DrawingMode && gZoneFullMap_SelectedMicrophone === microphone) {
        ZoneFullMap_exitDrawingMode();
    } else {
        ZoneFullMap_enterDrawingMode(microphone);
    }
}

function ZoneFullMap_enterDrawingMode(microphone) 
{
    if (gZoneFullMap_DrawingMode && gZoneFullMap_SelectedMicrophone !== microphone) {
        ZoneFullMap_exitDrawingMode();
    }
    
    gZoneFullMap_DrawingMode = true;
    gZoneFullMap_SelectedMicrophone = microphone;
    
    const canvas = document.querySelector('.ZoneFullMap_canvas_td');
    if (canvas) {
        canvas.style.cursor = 'crosshair';
    }
    
    ZoneFullMap_showDrawingHint(microphone.name);
    
    console.log('Drawing mode enabled for:', microphone.name);
}

function ZoneFullMap_exitDrawingMode() 
{
    gZoneFullMap_DrawingMode = false;
    gZoneFullMap_SelectedMicrophone = null;
    gZoneFullMap_IsDrawing = false;
    
    const canvas = document.querySelector('.ZoneFullMap_canvas_td');
    if (canvas) {
        canvas.style.cursor = 'default';
    }

    ZoneFullMap_clearQuadSelection();
    ZoneFullMap_hideDrawingHint();
    
    //console.log('Drawing mode disabled');
}

function ZoneFullMap_clearAllMicrophoneZoneCounters() {
    gZoneFullMap_MicrophoneZoneCounters = {};
    //console.log('Cleared all microphone zone counters');
}

function ZoneFullMap_deleteQuadrilateral(quad) {
    const index = gZoneFullMap_Quadrilaterals.indexOf(quad);
    if (index > -1) {
        // 釋放麥克風特定的Zone ID
        if (quad.deviceId) {
            const micNumber = ZoneFullMap_getMicrophoneNumber(quad.deviceId);
            const zoneNumber = quad.id % 100;
            ZoneFullMap_releaseMicrophoneZoneId(quad.deviceId, zoneNumber);
        } else {
            // 舊的全局ID系統
            gZoneFullMap_DeletedIds.push(quad.id);
            gZoneFullMap_DeletedIds.sort((a, b) => a - b);
        }
        
        gZoneFullMap_Quadrilaterals.splice(index, 1);
        
        const activeIndex = gZoneFullMap_ActiveRectangles.indexOf(quad.id);
        if (activeIndex > -1) {
            gZoneFullMap_ActiveRectangles.splice(activeIndex, 1);
        }
        
        console.log('Deleted quadrilateral:', quad.getDisplayZoneId());
        ZoneFullMap_redrawQuadrilaterals();
    }
}

function ZoneFullMap_clearQuadSelection() {
    gZoneFullMap_Quadrilaterals.forEach(quad => quad.isSelected = false);
    gZoneFullMap_LastSelectedQuad = null;
    ZoneFullMap_redrawQuadrilaterals();
}

function ZoneFullMap_redrawQuadrilaterals() {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    if (gZoneFullMap_DisplayZoneState) {
        gZoneFullMap_Quadrilaterals.forEach(quad => {
            if (quad.deviceId && gZoneFullMap_selectedDevices.includes(quad.deviceId)) {
                quad.draw(ctx);
            }
            else if (!quad.deviceId) {
                quad.draw(ctx);
            }
        });
    }

    if (gSelectedDevice) {
        ZoneFullMap_highlightSelectedDevice();
    }
}

function ZoneFullMap_clearAllQuadrilaterals() {
    gZoneFullMap_Quadrilaterals = [];
    gZoneFullMap_ActiveRectangles = [];
    gZoneFullMap_DeletedIds = [];
    gZoneFullMap_LastSelectedQuad = null;
    gZoneFullMap_DrawingMode = false;
    gZoneFullMap_SelectedMicrophone = null;
    ZoneFullMap_redrawQuadrilaterals();
    //console.log('Cleared all quadrilaterals');
}

function ZoneFullMap_initQuadrilateralSystem() {
    //console.log('Initializing quadrilateral drawing system');
    ZoneFullMap_clearAllQuadrilaterals();

}

function ZoneFullMap_showDrawingHint(microphoneName) {
    ZoneFullMap_hideDrawingHint();
    
    const hint = document.createElement('div');
    hint.id = 'ZoneFullMap_drawing_hint';
    hint.style.position = 'fixed';
    hint.style.top = '10px';
    hint.style.left = '50%';
    hint.style.transform = 'translateX(-50%)';
    hint.style.backgroundColor = 'rgba(255, 215, 0, 0.9)';
    hint.style.color = 'black';
    hint.style.padding = '8px 16px';
    hint.style.borderRadius = '5px';
    hint.style.zIndex = '1000';
    hint.style.fontSize = '14px';
    hint.style.fontWeight = 'bold';
    hint.style.boxShadow = '0 2px 10px rgba(0,0,0,0.3)';
    hint.innerHTML = `
        Drawing Zone for ${microphoneName} - Click and drag to create zone
        <br><small>Click microphone again to exit drawing mode</small>
    `;
    
    document.body.appendChild(hint);
}

function ZoneFullMap_hideDrawingHint() {
    const hint = document.getElementById('ZoneFullMap_drawing_hint');
    if (hint && hint.parentNode) {
        hint.remove();
    }
}

function ZoneFullMap_initializeMicrophoneZoneCounter(microphoneId) {
    if (!gZoneFullMap_MicrophoneZoneCounters[microphoneId]) {
        gZoneFullMap_MicrophoneZoneCounters[microphoneId] = {
            nextId: 1,
            usedIds: [],
            deletedIds: []
        };
    }
}

function ZoneFullMap_getNextMicrophoneZoneId(microphoneId) {
    ZoneFullMap_initializeMicrophoneZoneCounter(microphoneId);
    
    const counter = gZoneFullMap_MicrophoneZoneCounters[microphoneId];
    
    // console.log('Getting next zone ID for microphone:', {
    //     microphoneId,
    //     currentCounter: {
    //         nextId: counter.nextId,
    //         usedIds: counter.usedIds.slice(), // 複製陣列以避免修改原始資料
    //         deletedIds: counter.deletedIds.slice()
    //     }
    // });
    
    // 如果有刪除的ID，優先重用
    if (counter.deletedIds.length > 0) {
        const reusedId = counter.deletedIds.shift();
        counter.usedIds.push(reusedId);
        counter.usedIds.sort((a, b) => a - b);
        
        console.log('Reusing deleted zone ID:', {
            microphoneId,
            reusedId,
            updatedUsedIds: counter.usedIds.slice()
        });
        
        return reusedId;
    }
    
    // 否則使用下一個新ID
    const newId = counter.nextId;
    counter.usedIds.push(newId);
    counter.nextId++;
    
    console.log('Assigned new zone ID:', {
        microphoneId,
        newId,
        nextIdWillBe: counter.nextId,
        updatedUsedIds: counter.usedIds.slice()
    });
    
    return newId;
}

function ZoneFullMap_releaseMicrophoneZoneId(microphoneId, zoneId) {
    if (!gZoneFullMap_MicrophoneZoneCounters[microphoneId]) return;
    
    const counter = gZoneFullMap_MicrophoneZoneCounters[microphoneId];
    const usedIndex = counter.usedIds.indexOf(zoneId);
    
    if (usedIndex > -1) {
        counter.usedIds.splice(usedIndex, 1);
        counter.deletedIds.push(zoneId);
        counter.deletedIds.sort((a, b) => a - b);
    }
}

function ZoneFullMap_getMicrophoneNumber(microphoneId) {
    const match = microphoneId.match(/mic_(\d+)/);
    return match ? parseInt(match[1]) : 1;
}

function ZoneFullMap_debugCoordinates(device) {
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const cx = ox + (device.x * scale) / 100;
    const cy = oy - (device.y * scale) / 100;
    
    // console.log('Device coordinate debug:', {
    //     deviceName: device.name,
    //     mathCoords: { x: device.x, y: device.y },
    //     screenCoords: { x: cx, y: cy },
    //     canvasTransform: { ox, oy, scale }
    // });
}

function ZoneFullMap_setZoneStatus(micTabIndex, zoneIndex, status) {
    //console.log(`Setting Mic${micTabIndex + 1} Zone${zoneIndex + 1} status to: ${status}`);
    
    // 初始化狀態存儲
    if (!gSimpleZoneStates[micTabIndex]) {
        gSimpleZoneStates[micTabIndex] = {};
    }
    
    gSimpleZoneStates[micTabIndex][zoneIndex] = status;
    
    // 找到對應的麥克風設備
    const micDevice = ZoneFullMap_devices.find(device => 
        device.type === 'microphone' && ZoneFullMap_getMicrophoneNumber(device.id) === (micTabIndex + 1)
    );
    
    if (!micDevice) {
        console.warn(`Microphone ${micTabIndex + 1} not found`);
        return;
    }
    
    // 在麥克風設備上標記狀態，用於繪製時參考
    if (!micDevice.zoneStates) {
        micDevice.zoneStates = {};
    }
    micDevice.zoneStates[zoneIndex] = status;
    
    // 重新繪製以顯示狀態變化
    ZoneFullMap_redrawWithStatus();
    
    // 自動關閉狀態（如果不是off）
    if (status !== 'off') {
        setTimeout(() => {
            ZoneFullMap_setZoneStatus(micTabIndex, zoneIndex, 'off');
        }, 3000); // 3秒後自動關閉
    }
}

function convertMicRelativeToWorld(relativeX, relativeY, micDevice) {
    const micAngleRad = -(micDevice.angle || 0) * Math.PI / 180;
    
    // console.log('Converting mic-relative coordinates:', {
    //     input: { x: relativeX, y: relativeY },
    //     micAngle: micDevice.angle || 0,
    //     micPosition: { x: micDevice.x, y: micDevice.y }
    // });
    
    const rotatedX = relativeX * Math.cos(micAngleRad) - relativeY * Math.sin(micAngleRad);
    const rotatedY = relativeX * Math.sin(micAngleRad) + relativeY * Math.cos(micAngleRad);
    
    const worldCoords = {
        x: micDevice.x + rotatedX * 100,
        y: micDevice.y + rotatedY * 100
    };
    
    //console.log('Converted to world coordinates:', worldCoords);
    
    return worldCoords;
}

function ZoneFullMap_redrawWithStatus() {
    // 重繪麥克風（顯示外環狀態）
    ZoneFullMap_updateDevicePositions();
    
    // 重繪Zone（如果有對應的Zone）
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    
    // 清除並重繪所有Zone
    ZoneFullMap_redrawQuadrilaterals();
    
    // 為有狀態的Zone添加額外的視覺效果
    gZoneFullMap_Quadrilaterals.forEach(quad => {
        if (quad.currentStatus && quad.currentStatus !== 'off') {
            ZoneFullMap_drawZoneStatusEffect(ctx, quad);
        }
    });
}

function ZoneFullMap_drawZoneStatusEffect(ctx, quad) {
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    // 計算Zone中心點
    const centerMath = {
        x: quad.points.reduce((sum, p) => sum + p.x, 0) / quad.points.length,
        y: quad.points.reduce((sum, p) => sum + p.y, 0) / quad.points.length
    };
    
    const centerScreen = {
        x: ox + centerMath.x * scale / 100,
        y: oy - centerMath.y * scale / 100
    };

    ctx.save();
    
    // 根據狀態選擇顏色
    let effectColor, effectAlpha;
    switch (quad.currentStatus) {
        case 'active':
            effectColor = '#4CAF50'; // 綠色
            effectAlpha = 0.8;
            break;
        case 'preset':
            effectColor = '#FF9800'; // 橘色
            effectAlpha = 0.8;
            break;
        default:
            return; // 不繪製效果
    }

    // 繪製狀態指示圓圈（在Zone ID圓圈外圍）
    const time = Date.now();
    const pulseEffect = (Math.sin(time / 200) + 1) / 2; // 脈衝效果
    const radius = 15 + pulseEffect * 5;

    ctx.beginPath();
    ctx.arc(centerScreen.x, centerScreen.y, radius, 0, 2 * Math.PI);
    ctx.strokeStyle = effectColor;
    ctx.lineWidth = 3;
    ctx.globalAlpha = effectAlpha * (0.5 + pulseEffect * 0.5);
    ctx.stroke();

    ctx.restore();
}


function drawZoneWithMicStatus(quad, ctx) {
    // 初始化配色（如果未定義）
    if (!quad.colorScheme) {
        quad.initializeColorScheme();
    }

    // 將世界座標轉換為螢幕座標
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const screenPoints = quad.points.map(point => ({
        x: ox + point.x * scale / 100,
        y: oy - point.y * scale / 100
    }));

    // 如果點數不足，退出
    if (screenPoints.length < 3) return;

    ctx.save();

    // === 1. 繪製 Zone ===
    ctx.beginPath();
    ctx.moveTo(screenPoints[0].x, screenPoints[0].y);
    for (let i = 1; i < screenPoints.length; i++) {
        ctx.lineTo(screenPoints[i].x, screenPoints[i].y);
    }
    ctx.closePath();

    // 填充和外框
    ctx.fillStyle = quad.colorScheme.group || '#888888'; // 使用 group 顏色，預設灰色
    ctx.strokeStyle = '#000000'; // 固定黑色外框
    ctx.lineWidth = quad.isSelected ? 3 : 2; // 選中時加粗
    ctx.fill();
    ctx.stroke();

    // === 2. 繪製 Zone ID ===
    const center = {
        x: screenPoints.reduce((sum, p) => sum + p.x, 0) / screenPoints.length,
        y: screenPoints.reduce((sum, p) => sum + p.y, 0) / screenPoints.length
    };

    const text = quad.getDisplayZoneId();
    const textMetrics = ctx.measureText(text);
    const circleRadius = Math.max(12, textMetrics.width / 2 + 4);

    // 繪製簡單背景圓
    ctx.beginPath();
    ctx.fillStyle = quad.colorScheme.group || '#888888'; // 與 Zone 填充一致
    ctx.arc(center.x, center.y, circleRadius, 0, 2 * Math.PI);
    ctx.fill();

    // 黑色邊框
    ctx.strokeStyle = '#000000';
    ctx.lineWidth = 1;
    ctx.stroke();

    // 繪製白色文字
    ctx.fillStyle = '#FFFFFF';
    ctx.font = 'bold 14px Arial'; // 縮小字體以簡化
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText(text, center.x, center.y);

    // === 3. 簡化狀態顯示 ===
    const micDevice = ZoneFullMap_devices.find(d => d.id === quad.deviceId);
    if (micDevice && micDevice.zoneStates) {
        const zoneIndex = gZoneFullMap_Quadrilaterals.filter(q => q.deviceId === quad.deviceId).indexOf(quad);
        const status = micDevice.zoneStates[zoneIndex];

        if (status && status !== 'off') {
            // 簡單狀態標記：小圓點
            ctx.beginPath();
            ctx.arc(center.x + circleRadius + 4, center.y - circleRadius - 4, 3, 0, 2 * Math.PI);
            ctx.fillStyle = status === 'preset' ? '#FF9800' : '#4CAF50'; // 橘色（preset）或綠色（active）
            ctx.fill();
        }
    }

    // === 4. 選中時的控制項 ===
    if (quad.isSelected) {
        quad.drawZoneControls(ctx, screenPoints, 1);
    }

    ctx.restore();
}

function startSoundAnimation(micTabIndex, micDevice) {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_soundPoint');
    if (!canvas) return;
    

    if (!micDevice || typeof micDevice !== 'object' || !micDevice.id) {
        micDevice = ZoneFullMap_getMicrophoneByTabIndex(micTabIndex);
        if (!micDevice) {
            //console.warn(`No microphone found for Tab${micTabIndex + 1}`);
            return;
        }
    }
    
    const ctx = canvas.getContext('2d');
    
    let targetColorScheme = null;
    if (micDevice && micDevice.id) {
        try {
            targetColorScheme = ZoneFullMap_getMicrophoneColorScheme(micDevice.id);
        } catch (e) {
            
        }
    }
    
    function animatePoints() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        const currentTime = Date.now();
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
        
        // 繪製所有麥克風的聲音點
        Object.keys(gSimpleSoundPoints).forEach(micIdx => {
            const points = gSimpleSoundPoints[micIdx];
            const micDev = ZoneFullMap_devices.find(device => 
                device.type === 'microphone' && ZoneFullMap_getMicrophoneNumber(device.id) === (parseInt(micIdx) + 1)
            );
            
            if (!micDev) return;
            
            let micColorScheme;
            try {
                micColorScheme = ZoneFullMap_getMicrophoneColorScheme(micDev.id);
            } catch (e) {
                console.warn('Failed to get color scheme for mic:', micDev.id, e);
                // 使用預設顏色方案
                micColorScheme = {
                    primary: '#888888',
                    name: 'Default Gray'
                };
            }
            
            points.forEach((point) => {
                const screenX = ox + point.x * scale / 100;
                const screenY = oy - point.y * scale / 100;
                
                const age = currentTime - point.timestamp;
                const fadeDelay = 2000; // 2秒後開始淡出
                const fadeDuration = 1500; // 1.5秒淡出時間
                
                let alpha = point.opacity;
                if (age > fadeDelay) {
                    alpha = Math.max(0, 1 - (age - fadeDelay) / fadeDuration);
                }
                
                if (alpha > 0) {
                    drawSoundPoint(ctx, screenX, screenY, alpha, micColorScheme);
                }
                
                point.opacity = alpha;
            });
            
            gSimpleSoundPoints[micIdx] = points.filter(p => p.opacity > 0);
        });
        
        const hasActivePoints = Object.values(gSimpleSoundPoints).some(points => points.length > 0);
        if (hasActivePoints) {
            requestAnimationFrame(animatePoints);
        }
    }
    
    requestAnimationFrame(animatePoints);
}

function drawSoundPoint(ctx, screenX, screenY, alpha, colorScheme) {
    const soundPointRadius = 10; // 20px直徑 = 10px半徑
    
    ctx.save();
    
    // === Step 1: 繪製黑色外框 ===
    ctx.beginPath();
    ctx.arc(screenX, screenY, soundPointRadius, 0, 2 * Math.PI);
    ctx.fillStyle = `rgba(0, 0, 0, ${alpha})`;
    ctx.fill();
    
    // === Step 2: 繪製群組顏色內部 ===
    const innerRadius = soundPointRadius - 2; // 內圈比外圈小2px（黑框厚度）
    const r = parseInt(colorScheme.primary.slice(1, 3), 16);
    const g = parseInt(colorScheme.primary.slice(3, 5), 16);
    const b = parseInt(colorScheme.primary.slice(5, 7), 16);
    
    ctx.beginPath();
    ctx.arc(screenX, screenY, innerRadius, 0, 2 * Math.PI);
    ctx.fillStyle = `rgba(${r}, ${g}, ${b}, ${alpha})`;
    ctx.fill();
    
    // === Step 3: 添加內部高光效果（可選）===
    const highlightRadius = innerRadius - 3;
    if (highlightRadius > 0) {
        ctx.beginPath();
        // 高光位置稍微偏左上角
        ctx.arc(screenX - 2, screenY - 2, highlightRadius, 0, 2 * Math.PI);
        ctx.fillStyle = `rgba(255, 255, 255, ${alpha * 0.6})`;
        ctx.fill();
    }
    
    ctx.restore();
}

function ZoneFullMap_clearSoundPoints() {
    gSimpleSoundPoints = {};
    
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_soundPoint');
    if (canvas) {
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
    
    console.log('All sound points cleared');
}

function ZoneFullMap_getSoundPointCount(micTabIndex) {
    if (gSimpleSoundPoints[micTabIndex]) {
        return gSimpleSoundPoints[micTabIndex].length;
    }
    return 0;
}

function ZoneFullMap_getAllActiveSoundPoints() {
    let totalPoints = 0;
    Object.values(gSimpleSoundPoints).forEach(points => {
        totalPoints += points.length;
    });
    return totalPoints;
}

function ZoneFullMap_testSoundPoints() {
    console.log('=== Testing Sound Points ===');
    
    // 測試不同位置的聲音點
    const testPositions = [
        { x: 0, y: 150, desc: '正前方50cm' },
        { x: 130, y: 0, desc: '右側30cm' },
        { x: -130, y: 0, desc: '左側30cm' },
        { x: 0, y: -140, desc: '後方40cm' },
        { x: 120, y: 130, desc: '右前方' },
        { x: -120, y: -130, desc: '左後方' }
    ];
    
    testPositions.forEach((pos, index) => {
        setTimeout(() => {
            console.log(`Testing position: ${pos.desc}`);
            ZoneFullMap_showSoundXY(0, pos.x, pos.y); // 在Mic1測試
        }, index * 2000); // 每0.8秒一個點
    });
}

function ZoneFullMap_clearAllTestPoints() {
    // 清除陣列中的所有點
    if (typeof ZoneFullMap_points !== 'undefined') {
        ZoneFullMap_points = [];
    }
    
    // 清除 gSimpleSoundPoints（如果存在）
    if (typeof gSimpleSoundPoints !== 'undefined') {
        gSimpleSoundPoints = {};
    }
    
    // 清除畫布
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_soundPoint');
    if (canvas) {
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
    
    console.log('All test points cleared');
}

function ZoneFullMap_addPointsBatch(points, micTabIndex) {
    let currentBatch = 0;
    const totalBatches = Math.ceil(points.length / ZoneFullMap_TestConfig.batchSize);
    
    //console.log(`Adding ${points.length} points in ${totalBatches} batches...`);
    
    function addNextBatch() {
        const startIndex = currentBatch * ZoneFullMap_TestConfig.batchSize;
        const endIndex = Math.min(startIndex + ZoneFullMap_TestConfig.batchSize, points.length);
        
        for (let i = startIndex; i < endIndex; i++) {
            const point = points[i];
            
            setTimeout(() => {
                ZoneFullMap_showSoundXY(micTabIndex, point.x, point.y, {
                    intensity: point.intensity,
                    type: 'sound'
                });
            }, point.delay || 0);
        }
        
        currentBatch++;
        
        const progress = ((currentBatch / totalBatches) * 100).toFixed(1);
        //console.log(`Batch ${currentBatch}/${totalBatches} added (${progress}%)`);
        
        if (currentBatch < totalBatches) {
            setTimeout(addNextBatch, ZoneFullMap_TestConfig.batchInterval);
        } else {
            //console.log('All test points added successfully!');
        }
    }
    
    addNextBatch();
}

function ZoneFullMap_testMassRandomPoints(pointCount = 400, micTabIndex = 0) {
    console.log(`=== Testing ${pointCount} Random Sound Points ===`);
    
    // 檢查麥克風是否存在
    const micDevice = ZoneFullMap_devices.find(device => 
        device.type === 'microphone' && ZoneFullMap_getMicrophoneNumber(device.id) === (micTabIndex + 1)
    );
    
    if (!micDevice) {
        console.error(`Microphone ${micTabIndex + 1} not found`);
        return;
    }
    
    console.log(`Using microphone: ${micDevice.name}`);
    
    // 清除現有點
    ZoneFullMap_clearAllTestPoints();
    
    // 生成隨機點數據
    const points = [];
    const area = ZoneFullMap_TestConfig.testArea;
    
    for (let i = 0; i < pointCount; i++) {
        points.push({
            x: (Math.random() - 0.5) * area,      // -150cm 到 +150cm
            y: (Math.random() - 0.5) * area,      // -150cm 到 +150cm
            intensity: 0.3 + Math.random() * 0.7, // 0.3-1.0 強度
            delay: Math.random() * 2000            // 0-2秒延遲
        });
    }
    
    // 分批添加點
    ZoneFullMap_addPointsBatch(points, micTabIndex);
    
    return points.length;
}

function ZoneFullMap_CameraCalibration()
{
    //ZoneFullMap_testSoundPoints();
    //ZoneFullMap_testMassRandomPoints(800, 0);

    // ZoneFullMap_GetZoneModeConfig();

    //testMeasurementDevice();

    // setTimeout(() => {
    //     ZoneFullMap_runZoneStatusTests();
    // }, 500);

    //ZoneFullMap_testSingleZone(0, 1, 'active'); 


    //ZoneFullMap_exampleUsage();
    convertAndDisplayDevices();
}

function ZoneFullMap_testSingleZone(tabIndex, zoneIndex, status) {
    console.log(`Testing zone: Tab${tabIndex + 1} Zone${zoneIndex + 1} = ${status}`);
    
    // 找到對應的麥克風設備
    const micDevice = ZoneFullMap_getMicrophoneByTabIndex(tabIndex);
    if (!micDevice) {
        console.error(`Microphone for Tab${tabIndex + 1} not found`);
        return false;
    }
    
    // 使用局部更新函數
    return ZoneFullMap_updateSingleZoneStatus(micDevice.id, zoneIndex, status);
}

function ZoneFullMap_GetZoneModeConfig()
{    
    const allMicrophones = ZoneFullMap_devices.filter(device => device.type === 'microphone');
    
    if (allMicrophones.length === 0) {
        console.warn('No microphones found on canvas');
        alert('No microphones found on canvas. Please add microphones first.');
        return;
    }
    
    const connectedMicrophones = allMicrophones.filter(mic => mic.connectedTabIndex !== undefined);
    
    if (connectedMicrophones.length === 0) {
        console.warn('No microphones are connected to tabs');
        alert('No microphones are connected to tabs. Please configure tab connections first.');
        
        const unconnectedList = allMicrophones.map(mic => mic.name).join(', ');
        console.log('Unconnected microphones:', unconnectedList);
        return;
    }
    
    console.log('Found microphones on canvas:', {
        total: allMicrophones.length,
        connected: connectedMicrophones.length,
        unconnected: allMicrophones.length - connectedMicrophones.length
    });
    
    const connectionsList = connectedMicrophones.map(mic => ({
        deviceId: mic.id,
        deviceName: mic.name,
        tabIndex: mic.connectedTabIndex,
        displayName: ZoneFullMap_getMicrophoneDisplayName(mic)
    }));
    
    // console.log('Connected microphones that will be queried:', connectionsList);

    let requestsSent = 0;
    
    connectedMicrophones.forEach((mic, index) => {
        const jsonmsg = {
            Command: "GetZoneModeConfig",
            SoundTabIndex: mic.connectedTabIndex
        };
        
        setTimeout(() => {
                sendMessage("GetZoneModeConfig", jsonmsg);
                requestsSent++;
                
                console.log(`GetZoneModeConfig sent for Tab${mic.connectedTabIndex + 1}: ${mic.name}`);
                
                if (requestsSent === connectedMicrophones.length) {
                    console.log(`All ${requestsSent} GetZoneModeConfig requests sent successfully`);
                    
                    const calibrationBtn = document.getElementById('ZoneFullMap_startCalibration_btn');
                    if (calibrationBtn) {
                        calibrationBtn.disabled = true;
                        calibrationBtn.textContent = 'Loading Zones...';
                        
                        setTimeout(() => {
                            calibrationBtn.disabled = false;
                            calibrationBtn.textContent = 'Camera Calibration';
                        }, 2000);
                    }
                }
            }, index * 50);
        });
    
    return summary;
}

const ZoneConfig = {
    scaleFactor: 10000,      // 原始數據縮放因子
    deviceSize: 10,          // 設備空間10m x 10m
    deviceCenter: 5,         // 設備中心點5m
    offset: -1,              // 中心偏移-1m
    errorCompensation: 60,   // 60cm設計誤差補償
    pixelToMeter: 0.01       // 1000像素 = 10米
};

function ZoneFullMap_GetZoneConfig(msg) {
    // console.log('ZoneFullMap_GetZoneConfig received:', msg);
    
    const ZoneFullMapXYCanvasRect = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!ZoneFullMapXYCanvasRect) return;

    if (!msg?.ZoneModeConfig?.MicZoneItemArray) {
        console.warn('Invalid zone config data');
        return;
    }
    
    const zoneConfig = msg.ZoneModeConfig;
    const receivedTabIndex = zoneConfig.TabIndex || 0;
    
    console.log('Received zone config for tab:', receivedTabIndex);
    
    try {
        // 根據Tab索引找到對應的麥克風設備
        const micDevice = ZoneFullMap_getMicrophoneByTabIndex(receivedTabIndex);
        
        if (!micDevice) {
            // console.warn(`No microphone found for tab ${receivedTabIndex} in ZoneFullMap`);
            // console.log('Current tab mappings:', {
            //     tabToMicId: gZoneFullMap_TabToMicIdMapping,
            //     micIdToTab: gZoneFullMap_MicIdToTabMapping
            // });
            
            // 檢查是否有未配置Tab的麥克風
            const unconnectedMics = ZoneFullMap_devices.filter(device => 
                device.type === 'microphone' && device.connectedTabIndex === undefined
            );
            
            if (unconnectedMics.length > 0) {
                //console.log('Found unconnected microphones:', unconnectedMics.map(m => m.name));
                alert(`Received zone data for Tab${receivedTabIndex + 1}, but no microphone is connected to this tab. Please configure tab connections first.`);
            }
            return;
        }
        
        console.log(`Processing ${zoneConfig.MicZoneItemArray.length} zones for ${micDevice.name} (Tab${receivedTabIndex + 1})`);
        
        // 清除現有Zone
        ZoneFullMap_clearMicrophoneZones(micDevice.id);
        
        // 創建新的Zone
        const newQuads = [];
        zoneConfig.MicZoneItemArray.forEach((zoneItem, index) => {
            if (zoneItem.ZoneType !== 1 || zoneItem.ZoneID === 0) return;
            
            try {
                const quad = ZoneFullMap_convertAndCreateZone(zoneItem, micDevice);
                if (quad) {
                    newQuads.push(quad);
                }
            } catch (error) {
                console.error(`Error converting zone ${zoneItem.ZoneID}:`, error);
            }
        });
        
        // 更新顯示
        gZoneFullMap_ActiveRectangles.sort((a, b) => a - b);
        ZoneFullMap_redrawQuadrilaterals();
        
        // 如果接收的Tab對應的麥克風是當前選中的設備，保持選中狀態
        if (gSelectedDevice?.id === micDevice.id) {
            ZoneFullMap_highlightSelectedDevice();
        }
        
        // 更新UI信息，使用實際的Tab索引
        ZoneFullMap_updateZoneInfo(receivedTabIndex, newQuads.length, micDevice.name);
        
        // console.log(`Successfully processed ${newQuads.length} zones for ${micDevice.name} (Tab${receivedTabIndex + 1})`);
        
    } catch (error) {
        //console.error('Error in ZoneFullMap_GetZoneConfig:', error);
        ZoneFullMap_redrawQuadrilaterals();
    }
}

function ZoneFullMap_updateInfoPanel(micDevice, tabIndex) {
    console.log('Updated info panel for:', {
        deviceName: micDevice.name,
        tabIndex: tabIndex
    });
}

function ZoneFullMap_getConnectedMicrophones() {
    return ZoneFullMap_devices.filter(device => 
        device.type === 'microphone' && 
        device.connectedTabIndex !== undefined
    ).map(device => ({
        device: device,
        tabIndex: device.connectedTabIndex,
        displayName: ZoneFullMap_getMicrophoneDisplayName(device)
    }));
}

function ZoneFullMap_getAllConnectedMicZones() {
    const connectedMics = ZoneFullMap_getConnectedMicrophones();
    
    if (connectedMics.length === 0) {
        alert('No microphones are connected to tabs. Please configure tab connections first.');
        return;
    }
    
    console.log('Getting zone configurations for all connected microphones:', 
        connectedMics.map(m => `${m.displayName} (Tab${m.tabIndex + 1})`));
    
    connectedMics.forEach(micInfo => {
        const jsonmsg = {
            Command: "GetZoneModeConfig",
            SoundTabIndex: micInfo.tabIndex
        };
        
        console.log(`Sending GetZoneModeConfig for ${micInfo.displayName}`);
        
        if (typeof sendMessage === 'function') {
            // 添加延遲以避免同時發送多個請求
            setTimeout(() => {
                sendMessage("GetZoneModeConfig", jsonmsg);
            }, micInfo.tabIndex * 100);
        }
    });
}

function ZoneFullMap_validateTabConnections() {
    const allMics = ZoneFullMap_devices.filter(d => d.type === 'microphone');
    const connectedMics = ZoneFullMap_getConnectedMicrophones();
    const unconnectedMics = allMics.filter(d => d.connectedTabIndex === undefined);
    
    const report = {
        totalMicrophones: allMics.length,
        connectedMicrophones: connectedMics.length,
        unconnectedMicrophones: unconnectedMics.length,
        connections: connectedMics.map(m => ({
            micName: m.device.name,
            micId: m.device.id,
            tabIndex: m.tabIndex,
            displayName: m.displayName
        })),
        unconnected: unconnectedMics.map(m => ({
            micName: m.name,
            micId: m.id
        }))
    };
    
    console.log('Tab connection validation report:', report);
    return report;
}

function ZoneFullMap_convertAndCreateZone(zoneItem, micDevice) {
    console.log('=== Zone Conversion with Fixed Rotation Matrix ===');
    console.log('Input zoneItem:', zoneItem);
    console.log('Input micDevice:', {
        id: micDevice.id,
        name: micDevice.name,
        x: micDevice.x,
        y: micDevice.y,
        angle: micDevice.angle || 0
    });
    
    const rawPoints = ZoneFullMap_parseZoneCoordinates(zoneItem);
    if (!rawPoints) {
        console.log('Failed to parse coordinates');
        return null;
    }
    
    console.log('Raw device points:', rawPoints);
    
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    console.log('Canvas transform:', { ox, oy, scale });
    
    const majorGridSpacing = gDragSystem.gridConfig.majorSpacing;
    const baseCmToPixel = majorGridSpacing / 60;
    const cmToScreenPixel = baseCmToPixel * scale;
    
    console.log('Scale calculation:', {
        majorGridSpacing,
        baseCmToPixel: baseCmToPixel.toFixed(4),
        canvasScale: scale.toFixed(4),
        cmToScreenPixel: cmToScreenPixel.toFixed(4)
    });
    
    const micScreenX = ox + micDevice.x * scale / 100;
    const micScreenY = oy - micDevice.y * scale / 100;
    console.log('Microphone screen position:', { 
        micScreenX: micScreenX.toFixed(2), 
        micScreenY: micScreenY.toFixed(2) 
    });
    
    const screenPoints = rawPoints.map((point, index) => {
        let relativeCmX = point.x;
        let relativeCmY = -point.y;
        
        const relativeScreenX = relativeCmX * cmToScreenPixel;
        const relativeScreenY = relativeCmY * cmToScreenPixel;
        
        const angleRad = (micDevice.angle || 0) * Math.PI / 180;
        const cosAngle = Math.cos(angleRad);
        const sinAngle = Math.sin(angleRad);
        
        const rotatedX = relativeScreenX * cosAngle - relativeScreenY * sinAngle;
        const rotatedY = relativeScreenX * sinAngle + relativeScreenY * cosAngle;
        
        const screenPoint = {
            x: Math.round((micScreenX + rotatedX) * 100) / 100,
            y: Math.round((micScreenY + rotatedY) * 100) / 100
        };
        
        console.log(`Point ${index} (Fixed Rotation):`, {
            original: point,
            afterYFlip: { x: relativeCmX, y: relativeCmY },
            relativeScreen: { x: relativeScreenX.toFixed(2), y: relativeScreenY.toFixed(2) },
            afterRotation: { x: rotatedX.toFixed(2), y: rotatedY.toFixed(2) },
            finalScreen: screenPoint,
            micAngle: micDevice.angle || 0,
            rotationMatrix: {
                cosAngle: cosAngle.toFixed(4),
                sinAngle: sinAngle.toFixed(4)
            }
        });
        
        return screenPoint;
    });
    
    console.log('Fixed rotation screen points:', screenPoints);
    
    const bounds = {
        minX: Math.min(...screenPoints.map(p => p.x)),
        maxX: Math.max(...screenPoints.map(p => p.x)),
        minY: Math.min(...screenPoints.map(p => p.y)),
        maxY: Math.max(...screenPoints.map(p => p.y))
    };
    
    const zoneCenterX = (bounds.minX + bounds.maxX) / 2;
    const zoneCenterY = (bounds.minY + bounds.maxY) / 2;
    const deltaX = zoneCenterX - micScreenX;
    const deltaY = zoneCenterY - micScreenY;
    
    console.log('Rotation verification:', {
        micAngle: micDevice.angle || 0,
        zoneRelativeToMic: { 
            deltaX: deltaX.toFixed(1), 
            deltaY: deltaY.toFixed(1),
            distance: Math.sqrt(deltaX * deltaX + deltaY * deltaY).toFixed(1),
            angle: (Math.atan2(deltaY, deltaX) * 180 / Math.PI).toFixed(1)
        }
    });
    
    const quad = ZoneFullMap_createQuadrilateralForDevice(screenPoints, micDevice.id);
    
    if (quad) {
        quad.originalZoneID = zoneItem.ZoneID;
        quad.originalZoneName = zoneItem.ZoneName || `Zone_${zoneItem.ZoneID}`;
        
        setTimeout(() => {
            quad.updateRelativePosition();
            // console.log('Updated relative position with fixed rotation:', {
            //     displayId: quad.getDisplayZoneId(),
            //     relativeToDevice: quad.relativeToDevice
            // });
        }, 10);
        
        // console.log(`Zone created with fixed rotation:`, {
        //     id: quad.id,
        //     displayID: quad.getDisplayZoneId(),
        //     deviceId: quad.deviceId,
        //     micAngle: micDevice.angle || 0
        // });
    } else {
        //console.error('Failed to create quadrilateral with fixed rotation');
    }
    
    // console.log('=== END Fixed Rotation Conversion ===');
    return quad;
}

function ZoneFullMap_parseZoneCoordinates(zoneItem) {
    let points = [];
    
    if (zoneItem.AccordingToMicCoordinateTopLeftX !== undefined) {
        // console.log('Using AccordingToMicCoordinate system');
        points = [
            { x: zoneItem.AccordingToMicCoordinateTopLeftX, y: zoneItem.AccordingToMicCoordinateTopLeftY },
            { x: zoneItem.AccordingToMicCoordinateTopRightX, y: zoneItem.AccordingToMicCoordinateTopRightY },
            { x: zoneItem.AccordingToMicCoordinateBottomRightX, y: zoneItem.AccordingToMicCoordinateBottomRightY },
            { x: zoneItem.AccordingToMicCoordinateBottomLeftX, y: zoneItem.AccordingToMicCoordinateBottomLeftY }
        ];
    }
    else if (zoneItem.PolygonTopLeftX !== undefined) {
        // console.log('Using Polygon coordinate system');
        points = [
            { x: zoneItem.PolygonTopLeftX, y: zoneItem.PolygonTopLeftY },
            { x: zoneItem.PolygonTopRightX, y: zoneItem.PolygonTopRightY },
            { x: zoneItem.PolygonBottomRightX, y: zoneItem.PolygonBottomRightY },
            { x: zoneItem.PolygonBottomLeftX, y: zoneItem.PolygonBottomLeftY }
        ];
    } 
    else if (zoneItem.RectTopLeftX !== undefined) {
        // console.log('Using Rect coordinate system');
        points = [
            { x: zoneItem.RectTopLeftX, y: zoneItem.RectTopLeftY },
            { x: zoneItem.RectBottomRightX, y: zoneItem.RectTopLeftY },
            { x: zoneItem.RectBottomRightX, y: zoneItem.RectBottomRightY },
            { x: zoneItem.RectTopLeftX, y: zoneItem.RectBottomRightY }
        ];
    }
    
    if (points.length !== 4 || points.some(p => typeof p.x !== 'number' || typeof p.y !== 'number')) {
        // console.warn('Invalid zone coordinates');
        return null;
    }
    
    return points;
}

function ZoneFullMap_deviceToWorldCoordinate(devicePoint, micDevice) {
    const config = ZoneConfig;
    
    const meterX = (devicePoint.x / config.scaleFactor) * config.pixelToMeter;
    const meterY = (devicePoint.y / config.scaleFactor) * config.pixelToMeter;
    
    const offsetX = meterX - config.deviceCenter - config.offset;
    const offsetY = (config.deviceSize - meterY) - config.deviceCenter - config.offset;
    
    let compensatedX = offsetX;
    let compensatedY = offsetY;
    
    if (offsetX > 0) compensatedX += config.errorCompensation / 100;
    else if (offsetX < 0) compensatedX -= config.errorCompensation / 100;
    
    if (offsetY > 0) compensatedY += config.errorCompensation / 100;
    else if (offsetY < 0) compensatedY -= config.errorCompensation / 100;
    
    return {
        x: micDevice.x + compensatedX * 100,
        y: micDevice.y + compensatedY * 100
    };
}

function ZoneFullMap_worldToDeviceCoordinate(worldPoint, micDevice) {
    const config = ZoneConfig;
    
    const offsetCm_X = worldPoint.x - micDevice.x;
    const offsetCm_Y = worldPoint.y - micDevice.y;
    
    let offsetX = offsetCm_X / 100;
    let offsetY = offsetCm_Y / 100;
    
    if (offsetX > 0) offsetX -= config.errorCompensation / 100;
    else if (offsetX < 0) offsetX += config.errorCompensation / 100;
    
    if (offsetY > 0) offsetY -= config.errorCompensation / 100;
    else if (offsetY < 0) offsetY += config.errorCompensation / 100;
    
    const deviceX = offsetX + config.deviceCenter + config.offset;
    const deviceY = config.deviceSize - (offsetY + config.deviceCenter + config.offset);
    
    return {
        x: Math.round(deviceX / config.pixelToMeter * config.scaleFactor),
        y: Math.round(deviceY / config.pixelToMeter * config.scaleFactor)
    };
}

function ZoneFullMap_clearMicrophoneZones(micDeviceId) {
    if (!micDeviceId) return;
    
    const zonesToRemove = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === micDeviceId);
    
    zonesToRemove.forEach(quad => {
        const activeIndex = gZoneFullMap_ActiveRectangles.indexOf(quad.id);
        if (activeIndex > -1) {
            gZoneFullMap_ActiveRectangles.splice(activeIndex, 1);
        }
        
        if (quad.deviceId) {
            const zoneNumber = quad.id % 100;
            ZoneFullMap_releaseMicrophoneZoneId(quad.deviceId, zoneNumber);
        }
    });
    
    gZoneFullMap_Quadrilaterals = gZoneFullMap_Quadrilaterals.filter(
        quad => quad.deviceId !== micDeviceId
    );
    
    //console.log(`Cleared ${zonesToRemove.length} zones for microphone: ${micDeviceId}`);
}

function ZoneFullMap_updateZoneInfo(tabIndex, zoneCount, micName) {
    console.log(`Updated UI for Tab${tabIndex + 1}: ${micName} with ${zoneCount} zones`);
}

function ZoneFullMap_sendToDevice(micTabIndex = 0) {
    //console.warn('ZoneFullMap_sendToDevice is deprecated. Use ZoneFullMap_applyZoneChanges instead.');
    
    const micDevice = ZoneFullMap_getMicrophoneByTabIndex(micTabIndex);
    
    if (!micDevice) {
        //console.error(`No microphone found for tab ${micTabIndex}`);
        return;
    }
    
    const originalSelectedDevice = gSelectedDevice;
    gSelectedDevice = micDevice;
    
    try {
        ZoneFullMap_applyZoneChanges();
    } finally {
        gSelectedDevice = originalSelectedDevice;
    }
}

function ZoneFullMap_debugTabConnections() {
    console.group('ZoneFullMap Tab Connections Debug');
    
    console.log('=== Mapping Tables ===');
    console.log('micIdToTab:', gZoneFullMap_MicIdToTabMapping);
    console.log('tabToMicId:', gZoneFullMap_TabToMicIdMapping);
    
    console.log('=== All Microphone Devices ===');
    const allMics = ZoneFullMap_devices.filter(d => d.type === 'microphone');
    allMics.forEach(mic => {
        console.log(`${mic.id} (${mic.name}):`, {
            connectedTabIndex: mic.connectedTabIndex,
            displayName: ZoneFullMap_getMicrophoneDisplayName(mic),
            zoneCount: gZoneFullMap_Quadrilaterals.filter(q => q.deviceId === mic.id).length
        });
    });
    
    console.log('=== Available Mic Data ===');
    Object.entries(gZoneFullMap_MicConnectionData).forEach(([tabIndex, micData]) => {
        console.log(`Tab${parseInt(tabIndex) + 1}:`, {
            micTypeName: micData.micTypeName,
            ipAddress: micData.ipAddress,
            isConnect: micData.isConnect,
            assignedToDevice: gZoneFullMap_TabToMicIdMapping[tabIndex] || 'none'
        });
    });
    
    console.log('=== Validation Report ===');
    const report = ZoneFullMap_validateTabConnections();
    console.log(report);
    
    console.groupEnd();
    
    return report;
}

function ZoneFullMap_resetMicZones(micTabIndex) {
    const micDevice = ZoneFullMap_devices.find(device => 
        device.type === 'microphone' && 
        ZoneFullMap_getMicrophoneNumber(device.id) === (micTabIndex + 1)
    );
    
    if (micDevice) {
        ZoneFullMap_clearMicrophoneZones(micDevice.id);
        ZoneFullMap_redrawQuadrilaterals();
        console.log(`Reset all zones for ${micDevice.name}`);
    }
}

function ZoneFullMap_getZoneStats(micTabIndex) {
    const micDevice = ZoneFullMap_devices.find(device => 
        device.type === 'microphone' && 
        ZoneFullMap_getMicrophoneNumber(device.id) === (micTabIndex + 1)
    );
    
    if (!micDevice) return { totalZones: 0, micName: 'Not Found' };
    
    const micZones = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === micDevice.id);
    
    return {
        totalZones: micZones.length,
        micName: micDevice.name,
        micId: micDevice.id
    };
}

function ZoneFullMap_testCoordinateTransform() {
    console.log('=== Testing Coordinate Transform ===');
    
    const micDevice = ZoneFullMap_devices.find(d => d.type === 'microphone');
    if (!micDevice) {
        console.log('No microphone found');
        return;
    }
    
    const testZoneItem = {
        PolygonTopLeftX: 600,
        PolygonTopLeftY: 300,
        PolygonTopRightX: 701,
        PolygonTopRightY: 300,
        PolygonBottomRightX: 701,
        PolygonBottomRightY: 400,
        PolygonBottomLeftX: 600,
        PolygonBottomLeftY: 400,
        ZoneID: 99,
        ZoneName: "TestZone"
    };
    
    console.log('Creating test zone with fixed logic...');
    const testQuad = ZoneFullMap_convertAndCreateZone(testZoneItem, micDevice);
    
    if (testQuad) {
        gZoneFullMap_Quadrilaterals.push(testQuad);
        gZoneFullMap_ActiveRectangles.push(testQuad.id);
        ZoneFullMap_redrawQuadrilaterals();
        console.log('Test zone created and displayed');
    } else {
        console.log('Failed to create test zone');
    }
}


let ZoneFullMap_canvas = null;
let ZoneFullMap_ctx = null;
let ZoneFullMap_points = [];
let ZoneFullMap_animationId = null;
let ZoneFullMap_config = {
    centerX: 400,
    centerY: 300,
    pixelPerMeter: 50,
    maxPoints: 100,     // 降低最大點數避免卡頓
    pointLifetime: 2000 // 縮短生命週期避免累積
};

const ZoneFullMap_TestConfig = {
    batchSize: 20,          // 每批添加的點數（避免卡頓）
    batchInterval: 5,      // 批次間隔（毫秒）
    maxTestPoints: 1000,    // 最大測試點數
    testArea: 1200,          // 測試區域大小（cm）
    micIndex: 0             // 默認使用第一個麥克風
};

function ZoneFullMap_initialize(options = {}) {
    // 找到Canvas
    const canvasId = 'ZoneFullMap_xy_canvas_soundPoint';
    ZoneFullMap_canvas = document.getElementById(canvasId);
    
    if (!ZoneFullMap_canvas) {
        return false;
    }
    
    // 獲取Context
    ZoneFullMap_ctx = ZoneFullMap_canvas.getContext('2d');
    
    // 設置Canvas尺寸和配置
    const rect = ZoneFullMap_canvas.getBoundingClientRect();
    const dpr = window.devicePixelRatio || 1;
    
    ZoneFullMap_canvas.width = rect.width * dpr;
    ZoneFullMap_canvas.height = rect.height * dpr;
    ZoneFullMap_canvas.style.width = rect.width + 'px';
    ZoneFullMap_canvas.style.height = rect.height + 'px';
    
    ZoneFullMap_ctx.scale(dpr, dpr);
    
    // 更新配置
    ZoneFullMap_config.centerX = rect.width / 2;
    ZoneFullMap_config.centerY = rect.height / 2;
    ZoneFullMap_config.pixelPerMeter = Math.min(rect.width / 12, rect.height / 10);
    
    // 合併用戶配置
    if (options.maxPoints) ZoneFullMap_config.maxPoints = options.maxPoints;
    if (options.pointLifetime) ZoneFullMap_config.pointLifetime = options.pointLifetime;
    
    // 啟動渲染循環
    ZoneFullMap_startRender();
    
    // console.log('ZoneFullMap initialized:', {
    //     canvas: rect.width + 'x' + rect.height,
    //     center: ZoneFullMap_config.centerX + ',' + ZoneFullMap_config.centerY,
    //     scale: ZoneFullMap_config.pixelPerMeter.toFixed(1) + 'px/m'
    // });
    
    return true;
}

function ZoneFullMap_showSoundXY(micTabIndex, soundLocationX, soundLocationY, options = {}) {

    if (!ZoneFullMap_ctx) {
        //console.warn('ZoneFullMap not initialized');
        return false;
    }
    
    // const micDevice = ZoneFullMap_devices.find(device => 
    //     device.type === 'microphone' && ZoneFullMap_getMicrophoneNumber(device.id) === (micTabIndex + 1)
    // );
    const micDevice = ZoneFullMap_getMicrophoneByTabIndex(micTabIndex);
    
    if (!micDevice) {
        //console.warn(`Microphone ${micTabIndex + 1} not found`);
        return false;
    }

    if(!gZoneFullMapAllDataShow)
    {
        if (!gSelectedDevice || gSelectedDevice.id !== micDevice.id) 
        {
            //console.log(`Sound point ignored - ${micDevice.name} is not the selected device. Selected: ${gSelectedDevice ? gSelectedDevice.name : 'none'}`);
            return false;
        }
    }

    const worldPos = convertMicRelativeToWorld(soundLocationX, soundLocationY, micDevice);
    
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const screenX = ox + worldPos.x * scale / 100;
    const screenY = oy - worldPos.y * scale / 100;
    
    // console.log('Sound point coordinate conversion:', {
    //     micTabIndex: micTabIndex + 1,
    //     inputRelative: { x: soundLocationX, y: soundLocationY },
    //     micDevice: {
    //         id: micDevice.id,
    //         position: { x: micDevice.x, y: micDevice.y },
    //         angle: micDevice.angle || 0
    //     },
    //     worldCoords: { x: worldPos.x.toFixed(1), y: worldPos.y.toFixed(1) },
    //     canvasTransform: { ox: ox.toFixed(1), oy: oy.toFixed(1), scale: scale.toFixed(3) },
    //     screenCoords: { x: screenX.toFixed(1), y: screenY.toFixed(1) }
    // });
    
    const point = {
        x: screenX,
        y: screenY,
        timestamp: Date.now(),
        opacity: 1.0,
        tabIndex: micTabIndex,
        intensity: options.intensity || 0.8,
        type: options.type || 'sound',
        micId: micDevice.id
    };
    
    ZoneFullMap_points.push(point);
    
    if (ZoneFullMap_points.length > ZoneFullMap_config.maxPoints) {
        ZoneFullMap_points = ZoneFullMap_points.slice(-ZoneFullMap_config.maxPoints);
    }
    
    return true;
}

function ZoneFullMap_cleanup() {
    if (ZoneFullMap_animationId) {
        cancelAnimationFrame(ZoneFullMap_animationId);
        ZoneFullMap_animationId = null;
    }
    
    ZoneFullMap_points = [];
    
    if (ZoneFullMap_ctx && ZoneFullMap_canvas) {
        ZoneFullMap_ctx.clearRect(0, 0, ZoneFullMap_canvas.width, ZoneFullMap_canvas.height);
    }
    
    console.log('ZoneFullMap cleaned up');
}

function ZoneFullMap_startRender() {
    function render() {
        ZoneFullMap_updatePoints();
        
        ZoneFullMap_ctx.clearRect(0, 0, ZoneFullMap_canvas.width, ZoneFullMap_canvas.height);
        
        ZoneFullMap_renderPoints();
        
        ZoneFullMap_animationId = requestAnimationFrame(render);
    }
    
    render();
}

function ZoneFullMap_updatePoints() {
    if (ZoneFullMap_points.length === 0) return;
    
    const currentTime = Date.now();
    const activePoints = [];
    
    for (const point of ZoneFullMap_points) {
        const age = currentTime - point.timestamp;
        
        if (age < ZoneFullMap_config.pointLifetime) {
            point.opacity = Math.max(0, 1 - (age / ZoneFullMap_config.pointLifetime));
            activePoints.push(point);
        }
    }
    
    ZoneFullMap_points = activePoints;
}

function ZoneFullMap_renderPoints() {
    for (const point of ZoneFullMap_points) {
        ZoneFullMap_drawPoint(point);
    }
}

function ZoneFullMap_drawPoint(point) {
    const { x, y, opacity, intensity, type, micId } = point;
    
    ZoneFullMap_ctx.save();
    ZoneFullMap_ctx.globalAlpha = opacity;
    
    if (type === 'sound' || !type) {
        // === 獲取麥克風顏色方案 ===
        let colorScheme = {
            primary: '#FF6464', // 默認紅色
            name: 'Default Red'
        };
        
        if (micId) {
            try {
                colorScheme = ZoneFullMap_getMicrophoneColorScheme(micId);
            } catch (e) {
                console.warn('Failed to get mic color scheme:', e);
            }
        }
        
        // 解析顏色
        const primaryColor = colorScheme.primary;
        let r = 255, g = 100, b = 100; // 默認紅色
        
        if (primaryColor && primaryColor.startsWith('#')) {
            r = parseInt(primaryColor.slice(1, 3), 16);
            g = parseInt(primaryColor.slice(3, 5), 16);
            b = parseInt(primaryColor.slice(5, 7), 16);
        }
        
        const radius = 10 + (1 - opacity) * 10; // 動態大小
        
        // // 外圈光暈效果
        // ZoneFullMap_ctx.beginPath();
        // ZoneFullMap_ctx.arc(x, y, radius, 0, Math.PI * 2);
        // ZoneFullMap_ctx.fillStyle = `rgba(${55}, ${55}, ${55}, ${opacity * 0.3})`;
        // ZoneFullMap_ctx.fill();
        
        // 主體圓圈
        ZoneFullMap_ctx.beginPath();
        ZoneFullMap_ctx.arc(x, y, 6, 0, Math.PI * 2);
        ZoneFullMap_ctx.fillStyle = `rgba(${r}, ${g}, ${b}, ${opacity * 0.8})`;
        ZoneFullMap_ctx.fill();
        
    } else if (type === 'movement') {
        const size = 6;
        ZoneFullMap_ctx.beginPath();
        ZoneFullMap_ctx.rect(x - size/2, y - size/2, size, size);
        ZoneFullMap_ctx.fillStyle = `rgba(100, 150, 255, ${opacity})`;
        ZoneFullMap_ctx.fill();
    }
    
    ZoneFullMap_ctx.restore();
}

function ZoneFullMap_getStatus() {
    return {
        initialized: !!ZoneFullMap_ctx,
        pointCount: ZoneFullMap_points.length,
        maxPoints: ZoneFullMap_config.maxPoints,
        canvasSize: ZoneFullMap_canvas ? 
            `${ZoneFullMap_canvas.width}x${ZoneFullMap_canvas.height}` : 'N/A',
        config: ZoneFullMap_config
    };
}

let ZoneFullMap_pointBuffer = [];
const ZoneFullMap_BatchConfig = {
    maxBufferSize: 20,
    bufferTimeout: 100
};

function ZoneFullMap_queueSoundPoint(msg) {
    if (!msg || !Number.isFinite(msg.SoundTabIndex) || 
        !Number.isFinite(msg.SoundLocationX) || !Number.isFinite(msg.SoundLocationY)) {
        console.warn('Invalid sound point data:', msg);
        return false;
    }

    const point = {
        x: msg.SoundLocationX,
        y: msg.SoundLocationY,
        intensity: msg.intensity || 0.8,
        type: msg.type || 'sound',
        delay: msg.delay || 0,
        micTabIndex: msg.SoundTabIndex
    };

    ZoneFullMap_pointBuffer.push(point);

    if (ZoneFullMap_pointBuffer.length >= ZoneFullMap_BatchConfig.maxBufferSize) {
        ZoneFullMap_processBuffer();
    } else if (!ZoneFullMap_pointBuffer.timeoutId) {
        ZoneFullMap_pointBuffer.timeoutId = setTimeout(ZoneFullMap_processBuffer, 
            ZoneFullMap_BatchConfig.bufferTimeout);
    }

    return true;
}

function ZoneFullMap_processBuffer() {
    if (ZoneFullMap_pointBuffer.length === 0) return;

    const pointsToProcess = [...ZoneFullMap_pointBuffer];
    ZoneFullMap_pointBuffer = [];
    clearTimeout(ZoneFullMap_pointBuffer.timeoutId);
    ZoneFullMap_pointBuffer.timeoutId = null;

    const pointsByMic = pointsToProcess.reduce((acc, point) => {
        const micIndex = point.micTabIndex;
        if (!acc[micIndex]) acc[micIndex] = [];
        acc[micIndex].push({
            x: point.x,
            y: point.y,
            intensity: point.intensity,
            type: point.type,
            delay: point.delay
        });
        return acc;
    }, {});

    for (const micIndex in pointsByMic) {
        ZoneFullMap_addPointsBatch(pointsByMic[micIndex], parseInt(micIndex));
    }

    // console.log(`Processed ${pointsToProcess.length} points from buffer`);
}

const ZoneFullMap_DevicePresets = {
    defaultLayout: {
        microphones: [
            { x: 0, y: 0, angle: 0, ip: '192.168.1.101', microphoneType: 'Shure_MXA310' },
            { x: 300, y: 0, angle: 90, ip: '192.168.1.102', microphoneType: 'Shure_MXA710' }
        ],
        cameras: [
            { x: -200, y: 200, angle: 0, fovAngle: 60, offsetAngle: 0 },
            { x: 200, y: 200, angle: 180, fovAngle: 75, offsetAngle: 0 }
        ],
        visions: [
            { x: 0, y: 300, angle: 270 }
        ],
        anchors: [
            { x: 0, y: -300 }
        ]
    }
};

function ZoneFullMap_batchAddDevices(deviceConfig) {
    console.log('\n=== ZoneFullMap_batchAddDevices Started ===');
    console.log('Received device configuration:', deviceConfig);
    
    if (deviceConfig.clearExisting) {
        console.log('Clearing existing devices...');
        ZoneFullMap_clearAllDevices();
    }
    
    // Add microphones
    if (deviceConfig.microphones) {
        console.log('\n--- Adding Microphones ---');
        deviceConfig.microphones.forEach((micConfig, index) => {
            const existingCount = ZoneFullMap_devices.filter(d => d.type === 'microphone').length;
            console.log(`Adding microphone ${index + 1}, existing count: ${existingCount}`);
            
            if (existingCount >= 6) {
                console.warn('Maximum microphone limit reached');
                return;
            }
            
            const device = {
                id: `mic_${existingCount + 1}`,
                type: 'microphone',
                name: `Microphone ${existingCount + 1}`,
                x: micConfig.x || 0,
                y: micConfig.y || 0,
                angle: micConfig.angle || 0,
                status: 'Connected',
                imgSrc: './images/Web_RMCG.png',
                baseWidthM: 0.6,
                canDelete: true,
                groupColor: ZoneFullMap_getMicrophoneColorScheme(`mic_${existingCount + 1}`).primary,
                ip: micConfig.ip || '',
                microphoneType: micConfig.microphoneType || 'Shure_MXA310',
                imageLoaded: false,
                imageElement: null
            };
            
            console.log('Created microphone device:', {
                id: device.id,
                position: `(${device.x}, ${device.y})`,
                angle: device.angle + '¢X'
            });
            
            ZoneFullMap_devices.push(device);
            ZoneFullMap_preloadMicrophoneImage(device);
        });
    }
    
    // Add cameras
    if (deviceConfig.cameras) {
        console.log('\n--- Adding Cameras ---');
        deviceConfig.cameras.forEach((camConfig, index) => {
            const existingCount = ZoneFullMap_devices.filter(d => d.type === 'camera').length;
            console.log(`Adding camera ${index + 1}, existing count: ${existingCount}`);
            
            if (existingCount >= 6) {
                console.warn('Maximum camera limit reached');
                return;
            }
            
            const device = {
                id: `camera_${existingCount + 1}`,
                type: 'camera',
                name: `Camera ${existingCount + 1}`,
                x: camConfig.x || 0,
                y: camConfig.y || 0,
                angle: camConfig.angle || 0,
                fovAngle: camConfig.fovAngle || 60,
                offsetAngle: camConfig.offsetAngle || 0,
                status: 'Connected',
                baseWidthM: 0.4,
                isSelectedCamera: false,
                canDelete: true
            };
            
            console.log('Created camera device:', {
                id: device.id,
                position: `(${device.x}, ${device.y})`,
                angle: device.angle + '¢X',
                fov: device.fovAngle + '¢X'
            });
            
            ZoneFullMap_devices.push(device);
        });
    }
    
    // Add BC-200 devices
    if (deviceConfig.visions) {
        console.log('\n--- Adding BC-200 Devices ---');
        deviceConfig.visions.forEach((visionConfig, index) => {
            const existingCount = ZoneFullMap_devices.filter(d => d.type === 'vision').length;
            console.log(`Adding BC-200 ${index + 1}, existing count: ${existingCount}`);
            
            if (existingCount >= 6) {
                console.warn('Maximum BC-200 limit reached');
                return;
            }
            
            const device = {
                id: `vision_${existingCount + 1}`,
                type: 'vision',
                name: `BC-200 ${existingCount + 1}`,
                x: visionConfig.x || 0,
                y: visionConfig.y || 0,
                angle: visionConfig.angle || 0,
                status: 'Connected',
                baseWidthM: 0.5,
                canDelete: true
            };
            
            console.log('Created BC-200 device:', {
                id: device.id,
                position: `(${device.x}, ${device.y})`,
                angle: device.angle + '¢X'
            });
            
            ZoneFullMap_devices.push(device);
        });
    }
    
    // Add anchors
    if (deviceConfig.anchors) {
        console.log('\n--- Adding Anchors ---');
        deviceConfig.anchors.forEach((anchorConfig, index) => {
            const existingCount = ZoneFullMap_devices.filter(d => d.type === 'anchor').length;
            console.log(`Adding anchor ${index + 1}, existing count: ${existingCount}`);
            
            if (existingCount >= 1) {
                console.warn('Maximum anchor limit reached');
                return;
            }
            
            const device = {
                id: `anchor_${existingCount + 1}`,
                type: 'anchor',
                name: `Anchor ${existingCount + 1}`,
                x: anchorConfig.x || 0,
                y: anchorConfig.y || 0,
                status: 'Active',
                baseWidthM: 0.3,
                canDelete: true
            };
            
            console.log('Created anchor device:', {
                id: device.id,
                position: `(${device.x}, ${device.y})`
            });
            
            ZoneFullMap_devices.push(device);
        });
    }
    
    // Update display
    console.log('\nUpdating display...');
    ZoneFullMap_updateButtonStates();
    ZoneFullMap_updateDevicePositions();
    
    // Show final canvas coordinates
    console.log('\n--- Final Canvas Coordinates ---');
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    console.log('Canvas transform:', { origin: `(${ox}, ${oy})`, scale });
    
    ZoneFullMap_devices.forEach(device => {
        const screenX = ox + device.x * scale / 100;
        const screenY = oy - device.y * scale / 100;
        
        console.log(`${device.name}:`, {
            worldCoordinates: `(${device.x}, ${device.y})`,
            canvasCoordinates: `(${screenX.toFixed(1)}, ${screenY.toFixed(1)})`
        });
    });
    
    console.log('\n=== ZoneFullMap_batchAddDevices Complete ===\n');
}

function ZoneFullMap_loadPresetLayout(presetName = 'defaultLayout') {
    const preset = ZoneFullMap_DevicePresets[presetName];
    if (!preset) {
        console.error('Preset not found:', presetName);
        return;
    }
    
    ZoneFullMap_batchAddDevices({
        ...preset,
        clearExisting: true
    });
}

function ZoneFullMap_updateDeviceInfo(deviceId, updates) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) {
        console.error('Device not found:', deviceId);
        return false;
    }
    
    console.log('Updating device:', deviceId, updates);
    
    const oldValues = {
        x: device.x,
        y: device.y,
        angle: device.angle
    };
    
    if (updates.x !== undefined) device.x = updates.x;
    if (updates.y !== undefined) device.y = updates.y;
    if (updates.angle !== undefined) {
        device.angle = ((updates.angle % 360) + 360) % 360;
    }
    
    switch (device.type) {
        case 'microphone':
            if (updates.ip !== undefined) device.ip = updates.ip;
            if (updates.microphoneType !== undefined) device.microphoneType = updates.microphoneType;
            
            if (oldValues.angle !== device.angle) {
                ZoneFullMap_rotateMicrophoneZones(deviceId, oldValues.angle, device.angle, device.x, device.y);
            }
            break;
            
        case 'camera':
            if (updates.fovAngle !== undefined) {
                device.fovAngle = Math.max(0, Math.min(360, updates.fovAngle));
            }
            if (updates.offsetAngle !== undefined) {
                device.offsetAngle = Math.max(-360, Math.min(360, updates.offsetAngle));
            }
            if (updates.isSelectedCamera !== undefined) {
                device.isSelectedCamera = updates.isSelectedCamera;
            }
            break;
            
        case 'vision':
            if (updates.fovAngle !== undefined) {
                device.fovAngle = Math.max(0, Math.min(360, updates.fovAngle));
            }
            if (updates.offsetAngle !== undefined) {
                device.offsetAngle = Math.max(-360, Math.min(360, updates.offsetAngle));
            }
            if (updates.isSelectedCamera !== undefined) {
                device.isSelectedCamera = updates.isSelectedCamera;
            }
            break;
            
        case 'anchor':
            if (updates.status !== undefined) device.status = updates.status;
            break;
    }
    
    if (oldValues.x !== device.x || oldValues.y !== device.y) {
        const deltaX = device.x - oldValues.x;
        const deltaY = device.y - oldValues.y;
        ZoneFullMap_syncDeviceZones(deviceId, deltaX, deltaY);
    }
    
    ZoneFullMap_updateDevicePositions();
    ZoneFullMap_redrawQuadrilaterals();
    
    if (gSelectedDevice && gSelectedDevice.id === deviceId) {
        ZoneFullMap_highlightSelectedDevice();
        ZoneFullMap_updateDeviceInfoPosition(device);
    }
    
    return true;
}

function ZoneFullMap_autoHideDeviceInfoOnEdge() {
    const infoBox = document.getElementById('ZoneFullMap_device_info');
    if (!infoBox || !gSelectedDevice) return;
    
    const container = document.querySelector('.ZoneFullMap_canvas_td');
    const containerRect = container.getBoundingClientRect();
    
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    const widthPx = gSelectedDevice.baseWidthM * scale * 100;
    const cx = ox + gSelectedDevice.x * scale / 100;
    const cy = oy - gSelectedDevice.y * scale / 100;
    
    if (cx < -widthPx || cx > containerRect.width + widthPx ||
        cy < -widthPx || cy > containerRect.height + widthPx) {
        ZoneFullMap_hideDeviceInfo();
    }
}

function ZoneFullMap_drawCameraWithPersistentArrows(ctxSound, ctxRect, device, cx, cy, widthPx) {

    ctxSound.beginPath();
    ctxSound.arc(cx, cy, widthPx / 2, 0, 2 * Math.PI);
    ctxSound.fillStyle = device.isSelectedCamera ? 'green' : 'gray';
    ctxSound.fill();
    ctxSound.closePath();

    ctxRect.save();
    
    const arrowLength = Math.max(20, widthPx * 1.5);
    const arrowHeadSize = Math.max(8, widthPx / 6);

    const hardwareAngleRad = (device.angle * Math.PI) / 180;
    const hardwareArrowEndX = cx + arrowLength * Math.sin(hardwareAngleRad);
    const hardwareArrowEndY = cy - arrowLength * Math.cos(hardwareAngleRad);

    ctxRect.setLineDash([5, 5]);
    ctxRect.strokeStyle = 'black';
    ctxRect.lineWidth = 2;
    ctxRect.beginPath();
    ctxRect.moveTo(cx, cy);
    ctxRect.lineTo(hardwareArrowEndX, hardwareArrowEndY);
    ctxRect.stroke();
    ZoneFullMap_drawArrowHead(ctxRect, hardwareArrowEndX, hardwareArrowEndY, hardwareAngleRad, arrowHeadSize, 'black');

    const totalAngle = device.angle + (device.fovAngle || 0) + (device.offsetAngle || 0);
    const fovAngleRad = (totalAngle * Math.PI) / 180;
    const fovArrowEndX = cx + arrowLength * Math.sin(fovAngleRad);
    const fovArrowEndY = cy - arrowLength * Math.cos(fovAngleRad);

    ctxRect.setLineDash([]);
    ctxRect.strokeStyle = device.isSelectedCamera ? 'green' : 'gray';
    ctxRect.lineWidth = 3;
    ctxRect.beginPath();
    ctxRect.moveTo(cx, cy);
    ctxRect.lineTo(fovArrowEndX, fovArrowEndY);
    ctxRect.stroke();
    ZoneFullMap_drawArrowHead(ctxRect, fovArrowEndX, fovArrowEndY, fovAngleRad, arrowHeadSize, device.isSelectedCamera ? 'green' : 'gray');
    
    ctxRect.restore();
    
    if (device.showAngleInfo || gSelectedDevice === device) {
        ZoneFullMap_drawDeviceAngleInfo(ctxRect, device, cx, cy + widthPx + 20);
    }
}

const MeasurementDevice = {
    deviceHeights: {
        microphone: 300,
        camera: 250,
        vision: 250,
        measurement: 100
    },
    
    // 角度轉換工具
    angleConverter: {
        // 將 -180~180 系統轉換為 0~360 系統
        toStandard: function(angle) {
            let normalized = angle;
            if (angle < 0) {
                normalized = angle + 360;
            }
            while (normalized >= 360) normalized -= 360;
            while (normalized < 0) normalized += 360;
            return normalized;
        },
        
        // 將 0~360 系統轉換為 -180~180 系統
        toMeasurement: function(angle) {
            let normalized = angle;
            while (normalized > 180) normalized -= 360;
            while (normalized < -180) normalized += 360;
            return normalized;
        }
    },
    
    // 極座標轉2D座標（自動處理 -180~180 系統）
    polarTo2D: function(distance, pan, tilt) {
        // 自動轉換 -180~180 到 0~360
        const standardPan = this.angleConverter.toStandard(pan);
        const panRad = (standardPan * Math.PI) / 180;
        
        // 忽略 tilt 以獲得精確結果
        const horizontalDistance = distance;
        
        // Pan = 0° 時指向 +Y
        // Pan = 90° 時指向 +X（順時針）
        const x_cm = horizontalDistance * Math.sin(panRad);
        const y_cm = horizontalDistance * Math.cos(panRad);
        
        const x = Math.round(x_cm * 100);
        const y = Math.round(y_cm * 100);
        
        return { x, y, x_cm, y_cm };
    },
    
    // 生成測量數據的函數（可選擇輸出角度系統）
    generateMeasurementData: function(micX, micY, micAngle, outputSystem = 'standard') {
        const angleRad = micAngle * Math.PI / 180;
        const cos_a = Math.cos(angleRad);
        const sin_a = Math.sin(angleRad);
        
        // 使用正確的順時針旋轉矩陣
        const rf_offset = {
            x: 30 * cos_a + 30 * sin_a,   // 右前角
            y: -30 * sin_a + 30 * cos_a
        };
        
        const rb_offset = {
            x: 30 * cos_a - 30 * sin_a,   // 右後角
            y: -30 * sin_a - 30 * cos_a
        };
        
        // 計算絕對位置
        const rf_pos = {
            x: micX + rf_offset.x,
            y: micY + rf_offset.y
        };
        
        const rb_pos = {
            x: micX + rb_offset.x,
            y: micY + rb_offset.y
        };
        
        // 轉換為極座標
        const centerDistance = Math.sqrt(micX * micX + micY * micY) || 300;
        let centerPan = Math.atan2(micX, micY) * 180 / Math.PI;
        
        const rf_distance = Math.sqrt(rf_pos.x * rf_pos.x + rf_pos.y * rf_pos.y);
        let rf_pan = Math.atan2(rf_pos.x, rf_pos.y) * 180 / Math.PI;
        
        const rb_distance = Math.sqrt(rb_pos.x * rb_pos.x + rb_pos.y * rb_pos.y);
        let rb_pan = Math.atan2(rb_pos.x, rb_pos.y) * 180 / Math.PI;
        
        // 根據輸出系統調整角度
        if (outputSystem === 'measurement') {
            // 輸出 -180~180 系統
            centerPan = this.angleConverter.toMeasurement(centerPan);
            rf_pan = this.angleConverter.toMeasurement(rf_pan);
            rb_pan = this.angleConverter.toMeasurement(rb_pan);
        } else {
            // 輸出 0~360 系統（標準）
            centerPan = this.angleConverter.toStandard(centerPan);
            rf_pan = this.angleConverter.toStandard(rf_pan);
            rb_pan = this.angleConverter.toStandard(rb_pan);
        }
        
        return {
            rightFront: {
                distance: rf_distance,
                pan: rf_pan,
                tilt: 0
            },
            center: {
                distance: centerDistance,
                pan: centerPan,
                tilt: 0
            },
            rightBack: {
                distance: rb_distance,
                pan: rb_pan,
                tilt: 0
            }
        };
    },
    
    // 處理麥克風測量數據（自動處理 -180~180 輸入）
    processMicrophoneMeasurement: function(measurements) {
        console.log('\n=== 處理麥克風測量數據 ===');
        
        // 顯示原始輸入
        console.log('原始測量數據:');
        console.log(`  中心: distance=${measurements.center.distance.toFixed(1)}, pan=${measurements.center.pan.toFixed(1)}°`);
        console.log(`  右前: distance=${measurements.rightFront.distance.toFixed(1)}, pan=${measurements.rightFront.pan.toFixed(1)}°`);
        console.log(`  右後: distance=${measurements.rightBack.distance.toFixed(1)}, pan=${measurements.rightBack.pan.toFixed(1)}°`);
        
        // 轉換測量點到2D座標（polarTo2D 會自動處理角度轉換）
        const points = {
            rightFront: this.polarTo2D(
                measurements.rightFront.distance,
                measurements.rightFront.pan,
                measurements.rightFront.tilt
            ),
            center: this.polarTo2D(
                measurements.center.distance,
                measurements.center.pan,
                measurements.center.tilt
            ),
            rightBack: this.polarTo2D(
                measurements.rightBack.distance,
                measurements.rightBack.pan,
                measurements.rightBack.tilt
            )
        };
        
        const micCenter = points.center;
        
        console.log('測量點座標(cm):');
        console.log(`  中心: (${points.center.x_cm.toFixed(1)}, ${points.center.y_cm.toFixed(1)})`);
        console.log(`  右前: (${points.rightFront.x_cm.toFixed(1)}, ${points.rightFront.y_cm.toFixed(1)})`);
        console.log(`  右後: (${points.rightBack.x_cm.toFixed(1)}, ${points.rightBack.y_cm.toFixed(1)})`);
        
        // 計算實際的角落偏移（相對於中心點）
        const actualOffsets = {
            rightFront: {
                x: points.rightFront.x_cm - points.center.x_cm,
                y: points.rightFront.y_cm - points.center.y_cm
            },
            rightBack: {
                x: points.rightBack.x_cm - points.center.x_cm,
                y: points.rightBack.y_cm - points.center.y_cm
            }
        };
        
        console.log('實際偏移量:');
        console.log(`  右前偏移: (${actualOffsets.rightFront.x.toFixed(1)}, ${actualOffsets.rightFront.y.toFixed(1)})`);
        console.log(`  右後偏移: (${actualOffsets.rightBack.x.toFixed(1)}, ${actualOffsets.rightBack.y.toFixed(1)})`);
        
        // 計算右側向量（兩個角落的中點）
        const rightVector = {
            x: (actualOffsets.rightFront.x + actualOffsets.rightBack.x) / 2,
            y: (actualOffsets.rightFront.y + actualOffsets.rightBack.y) / 2
        };
        
        console.log(`  右側向量: (${rightVector.x.toFixed(1)}, ${rightVector.y.toFixed(1)})`);
        
        // 計算麥克風角度
        let rightAngle = Math.atan2(rightVector.x, rightVector.y) * 180 / Math.PI;
        let micAngle = rightAngle - 90;
        
        // 正規化到 0-360 範圍
        while (micAngle < 0) micAngle += 360;
        while (micAngle >= 360) micAngle -= 360;
        
        console.log(`計算角度: ${micAngle.toFixed(1)}°`);
        
        // 驗證計算
        const angleRad = micAngle * Math.PI / 180;
        const cos_a = Math.cos(angleRad);
        const sin_a = Math.sin(angleRad);
        
        const theoreticalOffsets = {
            rightFront: {
                x: 30 * cos_a + 30 * sin_a,
                y: -30 * sin_a + 30 * cos_a
            },
            rightBack: {
                x: 30 * cos_a - 30 * sin_a,
                y: -30 * sin_a - 30 * cos_a
            }
        };
        
        const errors = {
            rightFront: Math.sqrt(
                Math.pow(theoreticalOffsets.rightFront.x - actualOffsets.rightFront.x, 2) +
                Math.pow(theoreticalOffsets.rightFront.y - actualOffsets.rightFront.y, 2)
            ),
            rightBack: Math.sqrt(
                Math.pow(theoreticalOffsets.rightBack.x - actualOffsets.rightBack.x, 2) +
                Math.pow(theoreticalOffsets.rightBack.y - actualOffsets.rightBack.y, 2)
            )
        };
        
        console.log('驗證結果:');
        console.log(`  理論右前: (${theoreticalOffsets.rightFront.x.toFixed(1)}, ${theoreticalOffsets.rightFront.y.toFixed(1)})`);
        console.log(`  理論右後: (${theoreticalOffsets.rightBack.x.toFixed(1)}, ${theoreticalOffsets.rightBack.y.toFixed(1)})`);
        console.log(`  誤差: RF=${errors.rightFront.toFixed(1)}cm, RB=${errors.rightBack.toFixed(1)}cm`);
        console.log(`  平均誤差: ${((errors.rightFront + errors.rightBack) / 2).toFixed(1)}cm`);
        
        return {
            position: micCenter,
            angle: Math.round(micAngle),
            measurementPoints: points,
            analysis: {
                rotationAngle: micAngle,
                positionError: (errors.rightFront + errors.rightBack) / 2,
                actualOffsets: actualOffsets,
                theoreticalOffsets: theoreticalOffsets
            }
        };
    },
    
    // 匯入測量數據
    importMeasurementData: function(measurementData) {
        console.log('\n========== 開始處理測量數據 ==========');
        console.log('角度系統: 自動偵測並轉換 -180~180 到 0~360');
        
        const deviceConfig = {
            clearExisting: measurementData.clearExisting || false,
            microphones: [],
            cameras: [],
            visions: []
        };
        
        if (measurementData.microphones) {
            console.log('\n--- 處理麥克風 ---');
            measurementData.microphones.forEach((micData, index) => {
                console.log(`\n麥克風 ${index + 1}: ${micData.ip}`);
                const processed = this.processMicrophoneMeasurement(micData.measurements);
                
                const micConfig = {
                    x: processed.position.x,
                    y: processed.position.y,
                    angle: processed.angle,
                    ip: micData.ip || '',
                    microphoneType: micData.type || 'Shure_MXA310'
                };
                
                console.log('最終麥克風配置:', {
                    position: `(${(micConfig.x/100).toFixed(1)}, ${(micConfig.y/100).toFixed(1)}) cm`,
                    angle: micConfig.angle + '°',
                    type: micConfig.microphoneType,
                    ip: micConfig.ip
                });
                
                deviceConfig.microphones.push(micConfig);
            });
        }
        
        if (measurementData.cameras) {
            console.log('\n--- 處理攝影機 ---');
            measurementData.cameras.forEach((camData, index) => {
                console.log(`\n攝影機 ${index + 1}:`);
                const pos = this.polarTo2D(
                    camData.distance,
                    camData.pan,
                    camData.tilt
                );
                
                const camConfig = {
                    x: pos.x,
                    y: pos.y,
                    angle: camData.deviceAngle || 0,
                    fovAngle: camData.fovAngle || 60,
                    offsetAngle: camData.offsetAngle || 0
                };
                
                deviceConfig.cameras.push(camConfig);
            });
        }
        
        if (measurementData.visions) {
            console.log('\n--- 處理 BC-200 ---');
            measurementData.visions.forEach((visionData, index) => {
                console.log(`\nBC-200 ${index + 1}:`);
                const pos = this.polarTo2D(
                    visionData.distance,
                    visionData.pan,
                    visionData.tilt
                );
                
                const visionConfig = {
                    x: pos.x,
                    y: pos.y,
                    angle: visionData.deviceAngle || 0
                };
                
                deviceConfig.visions.push(visionConfig);
            });
        }
        
        console.log('\n最終設備配置總覽:', deviceConfig);
        console.log('\n========== 測量數據處理完成 ==========\n');
        
        // 如果存在批次添加函數，呼叫它
        if (typeof ZoneFullMap_batchAddDevices === 'function') {
            ZoneFullMap_batchAddDevices(deviceConfig);
        }
        
        return deviceConfig;
    }
};

function testMeasurementDevice() {
    console.log('=== 測試 MeasurementDevice 系統 ===\n');

    const testCase1 = {
        rightFront: { distance: 157, pan: -69, tilt: 61 },
        center: { distance: 165.9, pan: -88, tilt: 55 },
        rightBack: { distance: 157.8, pan: -109, tilt: 60 }
    };

    const testCase2 = {
        rightFront: { distance: 320.3, pan: 48.8, tilt: 0 },
        center: { distance: 282.8, pan: 45.0, tilt: 0 },
        rightBack: { distance: 264.2, pan: 53.0, tilt: 0 }
    };

    const testCase3 = {
        rightFront: { distance: 250.6, pan: 331.4, tilt: 0 },
        center: { distance: 291.5, pan: 329.0, tilt: 0 },
        rightBack: { distance: 284.3, pan: 320.7, tilt: 0 }
    };
    
    const testCase4 = MeasurementDevice.generateMeasurementData(120, 120, 45);
    const testCase5 = MeasurementDevice.generateMeasurementData(0, -300, 180);
    const testCase6 = MeasurementDevice.generateMeasurementData(-240, -240, 225);
    
    const testData = {
        clearExisting: true,
        measurementDeviceHeight: 100,
        microphones: [
            {
                ip: '192.168.1.101',
                type: 'Shure_MXA310',
                name: '測試1 (硬編碼): 位置(0,300) 角度0°',
                measurements: testCase1
            },
            {
                ip: '192.168.1.102',
                type: 'Shure_MXA710',
                name: '測試2 (硬編碼): 位置(200,200) 角度30°',
                measurements: testCase2
            },
            {
                ip: '192.168.1.103',
                type: 'Shure_MXA310',
                name: '測試3 (硬編碼): 位置(-150,250) 角度90°',
                measurements: testCase3
            },
            {
                ip: '192.168.1.104',
                type: 'Shure_MXA710',
                name: '測試4 (生成): 位置(212,212) 角度45°',
                measurements: testCase4
            },
            {
                ip: '192.168.1.105',
                type: 'Shure_MXA310',
                name: '測試5 (生成): 位置(0,-300) 角度180°',
                measurements: testCase5
            },
            {
                ip: '192.168.1.106',
                type: 'Shure_MXA710',
                name: '測試6 (生成): 位置(-250,-150) 角度225°',
                measurements: testCase6
            }
        ]
    };
    
    console.log('=== 方式1：硬編碼測量數據（3組）===\n');
    
    console.log('testCase1 = {');
    console.log('  rightFront: { distance:', testCase1.rightFront.distance, ', pan:', testCase1.rightFront.pan, ', tilt:', testCase1.rightFront.tilt, '},');
    console.log('  center: { distance:', testCase1.center.distance, ', pan:', testCase1.center.pan, ', tilt:', testCase1.center.tilt, '},');
    console.log('  rightBack: { distance:', testCase1.rightBack.distance, ', pan:', testCase1.rightBack.pan, ', tilt:', testCase1.rightBack.tilt, '}');
    console.log('};\n');
    
    console.log('testCase2 = {');
    console.log('  rightFront: { distance:', testCase2.rightFront.distance, ', pan:', testCase2.rightFront.pan, ', tilt:', testCase2.rightFront.tilt, '},');
    console.log('  center: { distance:', testCase2.center.distance, ', pan:', testCase2.center.pan, ', tilt:', testCase2.center.tilt, '},');
    console.log('  rightBack: { distance:', testCase2.rightBack.distance, ', pan:', testCase2.rightBack.pan, ', tilt:', testCase2.rightBack.tilt, '}');
    console.log('};\n');
    
    console.log('testCase3 = {');
    console.log('  rightFront: { distance:', testCase3.rightFront.distance, ', pan:', testCase3.rightFront.pan, ', tilt:', testCase3.rightFront.tilt, '},');
    console.log('  center: { distance:', testCase3.center.distance, ', pan:', testCase3.center.pan, ', tilt:', testCase3.center.tilt, '},');
    console.log('  rightBack: { distance:', testCase3.rightBack.distance, ', pan:', testCase3.rightBack.pan, ', tilt:', testCase3.rightBack.tilt, '}');
    console.log('};\n');
    
    console.log('=== 方式2：使用 generateMeasurementData 函數（3組）===\n');
    
    console.log('testCase4 = MeasurementDevice.generateMeasurementData(212, 212, 45);');
    console.log('// 結果:', testCase4, '\n');
    
    console.log('testCase5 = MeasurementDevice.generateMeasurementData(0, -300, 180);');
    console.log('// 結果:', testCase5, '\n');
    
    console.log('testCase6 = MeasurementDevice.generateMeasurementData(-250, -150, 225);');
    console.log('// 結果:', testCase6, '\n');
    
    // 執行測量數據處理
    console.log('=== 執行所有6組測試 ===\n');
    
    testData.microphones.forEach((mic, index) => {
        console.log(`\n--- ${mic.name} ---`);
        console.log(`IP: ${mic.ip}, 型號: ${mic.type}`);
        
        // 處理單個麥克風
        const result = MeasurementDevice.processMicrophoneMeasurement(mic.measurements);
        
        console.log(`處理結果:`);
        console.log(`  計算位置: (${(result.position.x/100).toFixed(1)}, ${(result.position.y/100).toFixed(1)}) cm`);
        console.log(`  計算角度: ${result.angle}°`);
        console.log(`  位置誤差: ${result.analysis.positionError.toFixed(2)} cm`);
        
        // 驗證結果
        const expectedPositions = [
            {x: 0, y: 300},      // 測試1
            {x: 200, y: 200},    // 測試2
            {x: -150, y: 250},   // 測試3
            {x: 212, y: 212},    // 測試4
            {x: 0, y: -300},     // 測試5
            {x: -250, y: -150}   // 測試6
        ];
        
        const expectedAngles = [0, 30, 90, 45, 180, 225];
        
        const posError = Math.sqrt(
            Math.pow((result.position.x/100) - expectedPositions[index].x, 2) +
            Math.pow((result.position.y/100) - expectedPositions[index].y, 2)
        );
        
        const angleError = Math.abs(result.angle - expectedAngles[index]);
        
    });
    
    // 批次處理所有數據
    console.log('\n=== 批次處理結果 ===');
    const finalResult = MeasurementDevice.importMeasurementData(testData);
    
    return finalResult;
}

// 將函數加入全域範圍
if (typeof window !== 'undefined') {
    // window.MeasurementDevice = MeasurementDevice;
    window.testMeasurementDevice = testMeasurementDevice;
    window.ZoneFullMap_testMeasurementImport = testMeasurementDevice; // 相容性別名
}

const ZoneFullMap_BackgroundManager = {
    backgroundImage: null,
    imageScale: 1.0,
    imageOffset: { x: 0, y: 0 },
    opacity: 0.5,
    
    // ªì©l€Æ©³¹Ïµe¥¬
    initBackgroundCanvas: function() {
        const canvasContainer = document.querySelector('.ZoneFullMap_canvas_td');
        if (!canvasContainer) return;
        
        let bgCanvas = document.getElementById('ZoneFullMap_xy_canvas_background');
        if (!bgCanvas) {
            bgCanvas = document.createElement('canvas');
            bgCanvas.id = 'ZoneFullMap_xy_canvas_background';
            bgCanvas.className = 'ZoneFullMap_canvas_border';
            bgCanvas.style.position = 'absolute';
            bgCanvas.style.zIndex = '-1'; // ©ñŠb³Ì©³Œh
            canvasContainer.appendChild(bgCanvas);
        }
        
        this.bgCanvas = bgCanvas;
        this.bgContext = bgCanvas.getContext('2d');
        
        const containerRect = canvasContainer.getBoundingClientRect();
        bgCanvas.width = containerRect.width;
        bgCanvas.height = containerRect.height;
    },
    
    // žü€J©³¹Ï
    loadBackgroundImage: function(file) {
        const reader = new FileReader();
        reader.onload = (e) => {
            const img = new Image();
            img.onload = () => {
                this.backgroundImage = img;
                this.drawBackground();
                console.log('Background image loaded:', img.width, 'x', img.height);
            };
            img.src = e.target.result;
        };
        reader.readAsDataURL(file);
    },
    
    drawBackground: function() {
        if (!this.bgCanvas || !this.bgContext || !this.backgroundImage) return;
        
        const ctx = this.bgContext;
        const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
        
        ctx.clearRect(0, 0, this.bgCanvas.width, this.bgCanvas.height);
        
        ctx.save();
        ctx.globalAlpha = this.opacity;
        
        const imgWidth = this.backgroundImage.width * this.imageScale * scale;
        const imgHeight = this.backgroundImage.height * this.imageScale * scale;
        const imgX = ox + this.imageOffset.x * scale;
        const imgY = oy + this.imageOffset.y * scale;
        
        ctx.drawImage(this.backgroundImage, imgX, imgY, imgWidth, imgHeight);
        
        ctx.restore();
    },
    
    updateSettings: function(settings) {
        if (settings.scale !== undefined) this.imageScale = settings.scale;
        if (settings.opacity !== undefined) this.opacity = settings.opacity;
        if (settings.offsetX !== undefined) this.imageOffset.x = settings.offsetX;
        if (settings.offsetY !== undefined) this.imageOffset.y = settings.offsetY;
        
        this.drawBackground();
    },
    
    clearBackground: function() {
        this.backgroundImage = null;
        if (this.bgContext) {
            this.bgContext.clearRect(0, 0, this.bgCanvas.width, this.bgCanvas.height);
        }
    }
};


function ZoneFullMap_addBackgroundControls() {
    const controls = document.createElement('div');
    controls.id = 'ZoneFullMap_background_controls';
    controls.className = 'ZoneFullMap_background_controls';
    controls.style.position = 'absolute';
    controls.style.bottom = '10px';
    controls.style.left = '10px';
    controls.style.backgroundColor = 'rgba(0, 14, 26, 0.9)';
    controls.style.padding = '10px';
    controls.style.borderRadius = '5px';
    controls.style.color = 'white';
    controls.style.display = 'none';
    
    controls.innerHTML = `
        <div style="margin-bottom: 10px;">
            <button onclick="document.getElementById('ZoneFullMap_bg_file').click()" 
                style="background-color: #4CAF50; color: white; border: none; padding: 5px 10px; border-radius: 3px; cursor: pointer;">
                Load Floor Plan
            </button>
            <input type="file" id="ZoneFullMap_bg_file" accept="image/*" style="display: none;" 
                onchange="ZoneFullMap_handleBackgroundFile(this)">
        </div>
        <div style="margin-bottom: 5px;">
            <label>Scale: </label>
            <input type="range" min="0.1" max="3" step="0.1" value="1" 
                onchange="ZoneFullMap_BackgroundManager.updateSettings({scale: parseFloat(this.value)})"
                style="width: 100px;">
        </div>
        <div style="margin-bottom: 5px;">
            <label>Opacity: </label>
            <input type="range" min="0" max="1" step="0.1" value="0.5" 
                onchange="ZoneFullMap_BackgroundManager.updateSettings({opacity: parseFloat(this.value)})"
                style="width: 100px;">
        </div>
        <button onclick="ZoneFullMap_BackgroundManager.clearBackground()" 
            style="background-color: #f44336; color: white; border: none; padding: 5px 10px; border-radius: 3px; cursor: pointer;">
            Clear
        </button>
    `;
    
    const container = document.querySelector('.ZoneFullMap_canvas_td');
    if (container) {
        container.appendChild(controls);
    }
}


function ZoneFullMap_handleBackgroundFile(input) {
    if (input.files && input.files[0]) {
        ZoneFullMap_BackgroundManager.loadBackgroundImage(input.files[0]);
    }
}


function ZoneFullMap_addFloorPlanButton() {
    const calibrationView = document.querySelector('.ZoneFullMap_calibration_view');
    if (!calibrationView) return;
    
    const floorPlanBtn = document.createElement('button');
    floorPlanBtn.className = 'ZoneFullMap_control_btn';
    floorPlanBtn.textContent = 'Floor Plan';
    floorPlanBtn.onclick = function() {
        const controls = document.getElementById('ZoneFullMap_background_controls');
        if (controls) {
            controls.style.display = controls.style.display === 'none' ? 'block' : 'none';
        }
    };
    
    const calibrationBtn = document.getElementById('ZoneFullMap_startCalibration_btn');
    if (calibrationBtn) {
        calibrationBtn.parentNode.insertBefore(floorPlanBtn, calibrationBtn.nextSibling);
    }
}

function ZoneFullMap_applyZoneChanges() {
    try {
        console.log('ZoneFullMap_applyZoneChanges called');
        
        if (!gSelectedDevice || gSelectedDevice.type !== 'microphone') {
            alert('Please select a microphone first');
            return;
        }
        
        if (gSelectedDevice.connectedTabIndex === undefined) {
            alert('Selected microphone is not connected to any tab. Please configure a tab connection first.');
            return;
        }
        
        const tabIndex = gSelectedDevice.connectedTabIndex;
        const micDevice = gSelectedDevice;
        
        const micZones = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === micDevice.id);
        
        if (micZones.length === 0) {
            alert('No zones found for selected microphone');
            return;
        }

        gMappingOverViewStatus = 1;
        getMappingOverViewBlockUIforPage();

        setTimeout(
            function()
            {
                getMappingOverViewUnblockUIforPage();
                gMappingOverViewStatus = 0;
            }
        ,600);
        
        // console.log('Applying zones:', {
        //     deviceId: micDevice.id,
        //     deviceName: micDevice.name,
        //     tabIndex: tabIndex,
        //     zoneCount: micZones.length,
        //     displayName: ZoneFullMap_getMicrophoneDisplayName(micDevice)
        // });
        
        const convertedZones = ZoneFullMap_convertZonesFor12mSpace(micZones, micDevice);
        
        const jsonmsg = {
            Command: "SetZoneModeConfig",
            ZoneModeConfig: convertedZones,
            SoundTabIndex: tabIndex
        };
        
        console.log('Sending zone configuration:', {
            tabIndex: tabIndex,
            micName: micDevice.name,
            zoneCount: convertedZones.length,
            zones: convertedZones.map(z => ({
                id: z.id,
                position: `(${(z.x/10000).toFixed(1)}, ${(z.y/10000).toFixed(1)})`,
                size: `${(z.width/10000).toFixed(1)} x ${(z.height/10000).toFixed(1)}`
            }))
        });
        
        if (typeof sendMessage === 'function') {
            sendMessage("SetZoneModeConfig", jsonmsg);
            
            const applyBtn = document.getElementById('ZoneFullMapApplyBtn');
            if (applyBtn) {
                applyBtn.disabled = true;
                applyBtn.textContent = 'Applying...';
                
                setTimeout(() => {
                    applyBtn.disabled = false;
                    applyBtn.textContent = 'Zone Change Apply';
                    console.log('Zone changes applied successfully for tab:', tabIndex);
                }, 1000);
            }
            
            ZoneFullMap_updateInfoPanel(micDevice, tabIndex);
            
        } else {
            console.error('sendMessage function not available');
        }
        
    } catch (error) {
        console.error('Error applying zone changes:', error);
        alert('Failed to apply zone changes');
    }
}

function ZoneFullMap_convertZonesFor12mSpace(zones, micDevice) {
    const convertedZones = [];
    
    const scaleFactor = 10000;
    const targetSpaceWidth = 10;  // 10m寬
    const targetSpaceHeight = 10; // 10m高
    const targetCanvasWidth = 1000;
    const targetCanvasHeight = 1000;
    const targetPixelPerMeter = targetCanvasWidth / targetSpaceWidth;
    
    // 設備空間中心偏移
    const deviceCenterOffset = 5; // 5m (設備空間中心)
    const coordinateOffset = 0;  // -1m (座標系統偏移)
    
    zones.forEach((quad, index) => {
        try {
            // 獲取四個角點的世界座標 (以cm為單位)
            const worldPoints = quad.points.map(point => ({
                x_cm: point.x / 100,  // 內部單位轉換為cm
                y_cm: point.y / 100
            }));
            
            // 轉換到相對於麥克風的座標
            const relativePoints = worldPoints.map(point => ({
                x_cm: point.x_cm - micDevice.x / 100,
                y_cm: point.y_cm - micDevice.y / 100
            }));
            
            // 考慮麥克風旋轉 - 反向旋轉回到原始座標系
            const micAngleRad = -(micDevice.angle || 0) * Math.PI / 180;
            const cos_a = Math.cos(micAngleRad);
            const sin_a = Math.sin(micAngleRad);
            
            const rotatedPoints = relativePoints.map(point => {
                const rotatedX = point.x_cm * cos_a - point.y_cm * sin_a;
                const rotatedY = point.x_cm * sin_a + point.y_cm * cos_a;
                return {
                    x_cm: rotatedX,
                    y_cm: rotatedY
                };
            });
            
            // 轉換到12m×12m設備座標系統
            // 重要修正：正確處理Y軸方向
            const devicePoints = rotatedPoints.map(point => {
                // 從cm轉換為m
                const x_m = point.x_cm / 100;
                const y_m = point.y_cm / 100;
                
                // 座標系統轉換
                // X軸：ZoneFullMap向右為正 → 設備空間向右也為正
                // Y軸：ZoneFullMap向上為正 → 設備空間向下為正（需要翻轉）
                const deviceX = x_m + deviceCenterOffset + coordinateOffset;
                const deviceY = (deviceCenterOffset + coordinateOffset) - y_m; // 關鍵修正：減去y_m來翻轉Y軸
                
                // 轉換為像素座標（設備空間的像素座標）
                const pixelX = deviceX * targetPixelPerMeter;
                const pixelY = deviceY * targetPixelPerMeter;
                
                return {
                    x: Math.round(pixelX * scaleFactor),
                    y: Math.round(pixelY * scaleFactor)
                };
            });
            
            // 計算包圍矩形 (重要：x,y是左上角，width,height是尺寸)
            const minX = Math.min(...devicePoints.map(p => p.x));
            const minY = Math.min(...devicePoints.map(p => p.y));
            const maxX = Math.max(...devicePoints.map(p => p.x));
            const maxY = Math.max(...devicePoints.map(p => p.y));
            const width = maxX - minX;
            const height = maxY - minY;
            
            // 提取Zone ID
            const zoneId = quad.originalZoneID || quad.id || (index + 1);
            const zoneNumber = (typeof zoneId === 'number' && zoneId >= 100) ? 
                               zoneId % 100 : // 如果是複合ID (如101, 102)，取最後兩位
                               zoneId;         // 否則直接使用
            
            // 建立Zone數據結構（完全匹配原始系統格式）
            const zoneData = {
                x: minX,                              // 包圍矩形左上角X
                y: minY,                              // 包圍矩形左上角Y
                ZoneType: 1,                          // 固定為1
                width: width,                         // 矩形寬度
                height: height,                       // 矩形高度
                id: zoneNumber,                       // Zone編號(從1開始)
                isSelected: quad.isSelected || false,
                wasSelected: quad.wasSelected || false,
                imgZoneCancel: {},
                imgZoneResize: {},
                isPolygon: true,
                isRealPolygon: true,
                polygonPoints: devicePoints          // 四個角點座標陣列
            };
            
            convertedZones.push(zoneData);
            
            console.log(`Converted Zone ${quad.getDisplayZoneId()}:`, {
                zoneId: zoneNumber,
                worldCenter: {
                    x: (quad.points.reduce((sum, p) => sum + p.x, 0) / 4 / 100).toFixed(1),
                    y: (quad.points.reduce((sum, p) => sum + p.y, 0) / 4 / 100).toFixed(1)
                },
                relativeToMic: {
                    x: ((quad.points.reduce((sum, p) => sum + p.x, 0) / 4 - micDevice.x) / 100).toFixed(1),
                    y: ((quad.points.reduce((sum, p) => sum + p.y, 0) / 4 - micDevice.y) / 100).toFixed(1)
                },
                deviceBounds: {
                    x: (minX/scaleFactor).toFixed(0),
                    y: (minY/scaleFactor).toFixed(0),
                    width: (width/scaleFactor).toFixed(0),
                    height: (height/scaleFactor).toFixed(0)
                },
                polygonPoints: devicePoints.map((p, i) => 
                    `P${i}: (${(p.x/scaleFactor).toFixed(0)}, ${(p.y/scaleFactor).toFixed(0)})`
                )
            });
            
        } catch (error) {
            console.error(`Error converting zone ${index}:`, error);
        }
    });
    
    console.log(`Successfully converted ${convertedZones.length} zones to 12m×12m space`);
    return convertedZones;
}

var gZoneFullMap_MicConnectionData = {};

const gZoneFullMap_InterestedMicTypes = [
    gMicSelect_Index.Shure_MXA920_XY,
    gMicSelect_Index.Sennheiser_TCC2_XY,
    gMicSelect_Index.Sennheiser_TCCM_XY,
    gMicSelect_Index.Yamaha_RMCG_XY,
    gMicSelect_Index.AudioTechnica_ATND1061_XY
];

const gZoneFullMap_MicTypeNames = {
    [gMicSelect_Index.Shure_MXA920_XY]: "Shure:MXA920(Coordinate)",
    [gMicSelect_Index.Sennheiser_TCC2_XY]: "Sennheiser:TCC2(Coordinate)",
    [gMicSelect_Index.Sennheiser_TCCM_XY]: "Sennheiser:TCCM(Coordinate)",
    [gMicSelect_Index.Yamaha_RMCG_XY]: "Yamaha:RM-CG(Coordinate)",
    [gMicSelect_Index.AudioTechnica_ATND1061_XY]: "Audio-Technica:ATND1061(Coordinate)"
};

// function getMicSelectIndexByDeviceIndex(micDeviceIndex) {
//     return gMicDeviceIndexMapSelectIndex[micDeviceIndex] || -1;
// }

function isZoneFullMapInterestedMicType(micSelectIndex) {
    return gZoneFullMap_InterestedMicTypes.includes(micSelectIndex);
}

function updateZoneFullMapMicConnection(msg) {
    if (!msg.SoundSetting) return;
    
    const tabIndex = msg.SoundTabIndex;
    const micSelectIndex = getMicSelectIndexByDeviceIndex(msg.SoundSetting.MicDeviceIndex);
    
    if (isZoneFullMapInterestedMicType(micSelectIndex)) {
        gZoneFullMap_MicConnectionData[tabIndex] = {
            tabIndex: tabIndex,
            micSelectIndex: micSelectIndex,
            micTypeName: gZoneFullMap_MicTypeNames[micSelectIndex] || "Unknown",
            ipAddress: msg.SoundSetting.IPAddress || "",
            port: msg.SoundSetting.Port || 0,
            isConnect: msg.SoundSetting.isConnect || false,
            connectStatus: msg.SoundSetting.ConnectActionStatus || "Unknown",
            password: msg.SoundSetting.Password || "",
            apiKey: msg.SoundSetting.APIKey || ""
        };
    } 
    else 
    {
        if (gZoneFullMap_MicConnectionData[tabIndex]) {
            delete gZoneFullMap_MicConnectionData[tabIndex];
        }
    }
}

function getZoneFullMapAvailableMics() {
    const availableMics = [];
    
    Object.values(gZoneFullMap_MicConnectionData).forEach(micData => {
        if (micData.isConnect) {
            availableMics.push({
                value: micData.tabIndex,
                text: `Tab${micData.tabIndex}: ${micData.micTypeName}`,
                data: micData
            });
        }
    });
    
    return availableMics;
}

function getZoneFullMapMicDataByTab(tabIndex) {
    return gZoneFullMap_MicConnectionData[tabIndex] || null;
}

function generateZoneFullMapMicrophoneHTML(device) {
    const availableMics = getZoneFullMapAvailableMics();
    let micSelectHTML = '';
    
    micSelectHTML = `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Tab:</span>
            <select onchange="handleZoneFullMapMicSelection('${device.id}', this.value);" id="ZoneFullMap_connectedMic_${device.id}" 
                style="width: 140px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px; border: none;">`;
    
    const nullSelected = (device.connectedTabIndex === undefined || device.connectedTabIndex === '') ? 'selected' : '';
    micSelectHTML += `<option value="" ${nullSelected}>null</option>`;
    
    availableMics.forEach(mic => {
        const isAlreadyAssigned = ZoneFullMap_isTabAlreadyAssigned(mic.value, device.id);
        
        if (!isAlreadyAssigned || device.connectedTabIndex === mic.value) {
            const selected = device.connectedTabIndex === mic.value ? 'selected' : '';
            micSelectHTML += `<option value="${mic.value}" ${selected}>Tab${mic.value + 1}: ${mic.data.micTypeName}</option>`;
        }
    });
    
    micSelectHTML += `
            </select>
        </div>`;
    
    let ipAddress = 'null';
    if (device.connectedTabIndex !== undefined) {
        const micData = getZoneFullMapMicDataByTab(device.connectedTabIndex);
        if (micData && micData.ipAddress) {
            ipAddress = micData.ipAddress;
        }
    }
    
    micSelectHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">IP:</span>
            <span id="ZoneFullMap_ip_display_${device.id}" style="font-size: 18px;" class="ZoneFullMap_deviceInfo">${ipAddress}</span>
        </div>`;

    let connectStatus = 'null';
    if (device.connectedTabIndex !== undefined) {
        const micData = getZoneFullMapMicDataByTab(device.connectedTabIndex);
        if (micData) {
            connectStatus = micData.connectStatus;
        }
    }

    micSelectHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Status:</span>
            <span id="ZoneFullMap_status_display_${device.id}" style="font-size: 18px;" class="ZoneFullMap_deviceInfo">
                ${connectStatus}
            </span>
        </div>`;
    
    return micSelectHTML;
}

function handleZoneFullMapMicSelection(deviceId, selectedTabIndex) {
    console.log('handleZoneFullMapMicSelection called:', { deviceId, selectedTabIndex });
    
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device || device.type !== 'microphone') {
        console.error('Device not found or not a microphone:', deviceId);
        return;
    }
    
    const ipDisplay = document.getElementById(`ZoneFullMap_ip_display_${deviceId}`);
    const statusDisplay = document.getElementById(`ZoneFullMap_status_display_${deviceId}`);
    const deviceTitle = document.querySelector('#ZoneFullMap_device_info strong');
    
    if (device.connectedTabIndex !== undefined) {
        ZoneFullMap_clearMicrophoneZones(deviceId);
        
        delete gZoneFullMap_MicIdToTabMapping[device.id];
        delete gZoneFullMap_TabToMicIdMapping[device.connectedTabIndex];
        
        console.log('Cleared zones and mapping for previous tab:', device.connectedTabIndex);
    }
    
    if (selectedTabIndex === '') {
        device.connectedTabIndex = undefined;
        device.ip = '';
        device.name = `Microphone ${device.displayNumber}`;
        
        ZoneFullMap_updateMicrophoneIcon(device, 'Yamaha:RM-CG(Coordinate)');

        setTimeout(() => {
            ZoneFullMap_redrawQuadrilaterals();
        }, 50);
        
        console.log('Disconnected microphone from tab');
        
        ZoneFullMap_updateZoneInfo(null, 0, device.name);
        
    } else {
        const tabIndex = parseInt(selectedTabIndex);
        
        if (ZoneFullMap_isTabAlreadyAssigned(tabIndex, device.id)) {
            alert(`Tab${tabIndex + 1} is already assigned to another microphone!`);
            
            const selector = document.getElementById(`ZoneFullMap_connectedMic_${deviceId}`);
            if (selector) {
                selector.value = device.connectedTabIndex !== undefined ? device.connectedTabIndex : '';
            }
            return;
        }
        
        const micData = getZoneFullMapMicDataByTab(tabIndex);
        if (micData) {
            device.connectedTabIndex = tabIndex;
            device.ip = micData.ipAddress;
            device.microphoneType = micData.micTypeName;

            ZoneFullMap_updateMicrophoneIcon(device, micData.micTypeName);
            
            gZoneFullMap_MicIdToTabMapping[device.id] = tabIndex;
            gZoneFullMap_TabToMicIdMapping[tabIndex] = device.id;
            
            console.log('Connected microphone to tab:', {
                deviceId: device.id,
                tabIndex: tabIndex,
                micType: micData.micTypeName
            });
            
            ZoneFullMap_requestZoneConfigForDevice(tabIndex, device);
        }
    }

    const displayName = ZoneFullMap_getMicrophoneDisplayName(device);
    if (deviceTitle) {
        deviceTitle.textContent = displayName;
    }
    
    if (ipDisplay) {
        ipDisplay.textContent = device.ip || 'null';
    }
    
    if (statusDisplay) {
        const micData = device.connectedTabIndex !== undefined ? 
                       getZoneFullMapMicDataByTab(device.connectedTabIndex) : null;
        statusDisplay.textContent = micData ? micData.connectStatus : 'null';
    }
    
    if (gSelectedDevice && gSelectedDevice.id === deviceId) {
        setTimeout(() => {
            ZoneFullMap_highlightSelectedDevice();
        }, 50);
    }
    
    console.log('Updated microphone mapping:', {
        micIdToTab: gZoneFullMap_MicIdToTabMapping,
        tabToMicId: gZoneFullMap_TabToMicIdMapping
    });
}

function ZoneFullMap_requestZoneConfigForDevice(tabIndex, device) {
    const jsonmsg = {
        Command: "GetZoneModeConfig",
        SoundTabIndex: tabIndex
    };
    
    if (typeof sendMessage === 'function') {
        sendMessage("GetZoneModeConfig", jsonmsg);
        console.log(`GetZoneModeConfig sent for Tab${tabIndex + 1}: ${device.name}`);
        
        ZoneFullMap_showLoadingState(true);
        
        setTimeout(() => {
            ZoneFullMap_showLoadingState(false);
        }, 2000);
    } else {
        console.error('sendMessage function not available');
    }
}

function ZoneFullMap_showLoadingState(isLoading) {
    const applyBtn = document.getElementById('ZoneFullMapApplyBtn');
    const calibrationBtn = document.getElementById('ZoneFullMap_startCalibration_btn');
    
    if (applyBtn) {
        if (isLoading) {
            applyBtn.disabled = true;
            applyBtn.textContent = 'Loading Zones...';
        } else {
            applyBtn.disabled = false;
            applyBtn.textContent = 'Zone Change Apply';
        }
    }
    
    if (calibrationBtn) {
        if (isLoading) {
            calibrationBtn.disabled = true;
            calibrationBtn.textContent = 'Loading Zones...';
        } else {
            calibrationBtn.disabled = false;
            calibrationBtn.textContent = 'Camera Calibration';
        }
    }
}


var gZoneFullMap_MicIdToTabMapping = {};
var gZoneFullMap_TabToMicIdMapping = {};

function ZoneFullMap_isTabAlreadyAssigned(tabIndex, excludeDeviceId = null) {
    return ZoneFullMap_devices.some(device => 
        device.type === 'microphone' && 
        device.id !== excludeDeviceId && 
        device.connectedTabIndex === tabIndex
    );
}

function ZoneFullMap_getMicrophoneDisplayName(device) {
    if (device.connectedTabIndex !== undefined) {
        const micData = getZoneFullMapMicDataByTab(device.connectedTabIndex);
        if (micData) {
            return `Tab${device.connectedTabIndex + 1}: ${micData.micTypeName}`;
        } else {
            return `Tab${device.connectedTabIndex + 1}: ${device.name}`;
        }
    }
    return `Mic ${device.displayNumber}`;
}

function ZoneFullMap_getMicrophoneByTabIndex(tabIndex) {
    const micId = gZoneFullMap_TabToMicIdMapping[tabIndex];
    if (micId) {
        return ZoneFullMap_findDeviceById(micId);
    }
    return null;
}

function ZoneFullMap_getTabIndexByMicrophoneId(micId) {
    return gZoneFullMap_MicIdToTabMapping[micId];
}


const gZoneFullMap_MicTypeIcons = {
    "Yamaha:RM-CG(Coordinate)": './images/Web_RMCG.png',
    "Audio-Technica:ATND1061(Coordinate)": './images/Web_ATND1061.png',
    "Shure:MXA920(Coordinate)": './images/Web_MXA920.png',
    "Sennheiser:TCC2(Coordinate)": './images/Web_TCC2.png',
    "Sennheiser:TCCM(Coordinate)": './images/Web_TCCM.png'
};

function ZoneFullMap_getMicrophoneIcon(microphoneType) {
    return gZoneFullMap_MicTypeIcons[microphoneType] || './images/Web_RMCG.png';
}

function ZoneFullMap_updateMicrophoneIcon(device, newMicrophoneType) {
    if (!device || device.type !== 'microphone') return;

    device.microphoneType = newMicrophoneType;
    
    const newIconSrc = ZoneFullMap_getMicrophoneIcon(newMicrophoneType);
    device.imgSrc = newIconSrc;
    
    ZoneFullMap_preloadMicrophoneImage(device, () => {
        requestAnimationFrame(() => {
            ZoneFullMap_updateDevicePositions();
            
            ZoneFullMap_redrawQuadrilaterals();
            
            if (gSelectedDevice && gSelectedDevice.id === device.id) {
                ZoneFullMap_highlightSelectedDevice();
            }
            
            if (gDragSystem && gDragSystem.drawGrid) {
                gDragSystem.drawGrid();
            }
        });
        
        console.log(`Updated microphone icon for ${device.name}: ${newMicrophoneType} -> ${newIconSrc}`);
    });
    
    if (device.imageElement) {
        device.imageElement.src = newIconSrc;
    }
}

var gZoneFullMap_DeviceIdManager = {
    microphone: { usedIds: [], deletedIds: [], nextId: 1 },
    camera: { usedIds: [], deletedIds: [], nextId: 1 },
    vision: { usedIds: [], deletedIds: [], nextId: 1 },
    anchor: { usedIds: [], deletedIds: [], nextId: 1 }
};

function ZoneFullMap_initializeDeviceIdManager(deviceType) {
    if (!gZoneFullMap_DeviceIdManager[deviceType]) {
        gZoneFullMap_DeviceIdManager[deviceType] = {
            usedIds: [], deletedIds: [], nextId: 1
        };
    }
}

function ZoneFullMap_getNextDeviceId(deviceType) {
    ZoneFullMap_initializeDeviceIdManager(deviceType);
    const manager = gZoneFullMap_DeviceIdManager[deviceType];
    
    if (manager.deletedIds.length > 0) {
        manager.deletedIds.sort((a, b) => a - b);
        const reusedId = manager.deletedIds.shift();
        manager.usedIds.push(reusedId);
        manager.usedIds.sort((a, b) => a - b);
        return reusedId;
    }
    
    const newId = manager.nextId;
    manager.usedIds.push(newId);
    manager.nextId++;
    return newId;
}

function ZoneFullMap_releaseDeviceId(deviceType, deviceId) {
    ZoneFullMap_initializeDeviceIdManager(deviceType);
    const manager = gZoneFullMap_DeviceIdManager[deviceType];
    
    const usedIndex = manager.usedIds.indexOf(deviceId);
    if (usedIndex > -1) {
        manager.usedIds.splice(usedIndex, 1);
    }
    
    if (!manager.deletedIds.includes(deviceId)) {
        manager.deletedIds.push(deviceId);
        manager.deletedIds.sort((a, b) => a - b);
    }
}

function ZoneFullMap_initializeFromExistingDevices() {
    if (typeof ZoneFullMap_devices === 'undefined') return;
    
    Object.keys(gZoneFullMap_DeviceIdManager).forEach(type => {
        gZoneFullMap_DeviceIdManager[type] = {
            usedIds: [], deletedIds: [], nextId: 1
        };
    });
    
    const devicesByType = { microphone: [], camera: [], vision: [], anchor: [] };
    
    ZoneFullMap_devices.forEach(device => {
        if (devicesByType[device.type]) {
            const match = device.id.match(/_(\d+)$/);
            if (match) {
                devicesByType[device.type].push(parseInt(match[1]));
            }
        }
    });
    
    Object.keys(devicesByType).forEach(type => {
        const numbers = devicesByType[type];
        if (numbers.length > 0) {
            const manager = gZoneFullMap_DeviceIdManager[type];
            manager.usedIds = [...numbers].sort((a, b) => a - b);
            manager.nextId = Math.max(...numbers) + 1;
            
            const maxId = Math.max(...numbers);
            for (let i = 1; i < maxId; i++) {
                if (!numbers.includes(i)) {
                    manager.deletedIds.push(i);
                }
            }
            manager.deletedIds.sort((a, b) => a - b);
        }
    });
}

function ZoneFullMap_initExportImport() {
    const calibrationBtn = document.getElementById('ZoneFullMap_startCalibration_btn');
    if (!calibrationBtn) {
        return;
    }

    if (document.getElementById('ZoneFullMap_export_btn')) {
        return;
    }

    const exportBtn = document.createElement('button');
    exportBtn.id = 'ZoneFullMap_export_btn';
    exportBtn.className = 'ZoneFullMap_export_btn';
    exportBtn.textContent = 'Export';
    exportBtn.onclick = ZoneFullMap_export;

    const importBtn = document.createElement('button');
    importBtn.id = 'ZoneFullMap_import_btn';
    importBtn.className = 'ZoneFullMap_import_btn';
    importBtn.textContent = 'Import';
    importBtn.onclick = ZoneFullMap_import;

    calibrationBtn.parentNode.insertBefore(exportBtn, calibrationBtn.nextSibling);
    calibrationBtn.parentNode.insertBefore(importBtn, exportBtn.nextSibling);

}

function ZoneFullMap_export() {
    try {
        // 收集所有配置資料
        const configData = {
            metadata: {
                exportTime: new Date().toISOString(),
                version: "1.0.0"
            },
            devices: {
                microphones: [],
                cameras: [],
                visions: [],
                anchors: []
            },
            zones: [],
            connections: {
                micIdToTabMapping: { ...gZoneFullMap_MicIdToTabMapping },
                tabToMicIdMapping: { ...gZoneFullMap_TabToMicIdMapping }
            }
        };

        // 匯出設備
        ZoneFullMap_devices.forEach(device => {
            const deviceData = {
                id: device.id,
                type: device.type,
                name: device.name,
                x: device.x,
                y: device.y,
                status: device.status,
                baseWidthM: device.baseWidthM,
                canDelete: device.canDelete
            };

            switch (device.type) {
                case 'microphone':
                    Object.assign(deviceData, {
                        angle: device.angle || 0,
                        imgSrc: device.imgSrc,
                        ip: device.ip || '',
                        microphoneType: device.microphoneType || 'Yamaha:RM-CG(Coordinate)',
                        connectedTabIndex: device.connectedTabIndex,
                        displayNumber: device.displayNumber
                    });
                    configData.devices.microphones.push(deviceData);
                    break;

                case 'camera':
                    Object.assign(deviceData, {
                        angle: device.angle || 0,
                        fovAngle: device.fovAngle || 0,
                        offsetAngle: device.offsetAngle || 0,
                        isSelectedCamera: device.isSelectedCamera || false
                    });
                    configData.devices.cameras.push(deviceData);
                    break;

                case 'vision':

                    Object.assign(deviceData, {
                        angle: device.angle || 0,
                        fovAngle: device.fovAngle || 0,
                        offsetAngle: device.offsetAngle || 0,
                        isSelectedCamera: device.isSelectedCamera || false
                    });
                    configData.devices.visions.push(deviceData);
                    break;

                case 'anchor':
                    configData.devices.anchors.push(deviceData);
                    break;
            }
        });

        // 匯出Zone
        gZoneFullMap_Quadrilaterals.forEach(quad => {
            const zoneData = {
                id: quad.id,
                deviceId: quad.deviceId,
                points: quad.points.map(point => ({ x: point.x, y: point.y })),
                isSelected: quad.isSelected,
                color: quad.color,
                originalZoneID: quad.originalZoneID,
                originalZoneName: quad.originalZoneName,
                relativeToDevice: quad.relativeToDevice
            };
            configData.zones.push(zoneData);
        });

        // 生成檔案名稱
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').substring(0, 19);
        const filename = `ZoneMap_Config_${timestamp}.json`;

        // 創建並下載檔案
        const jsonString = JSON.stringify(configData, null, 2);
        const blob = new Blob([jsonString], { type: 'application/json;charset=utf-8' });
        const url = URL.createObjectURL(blob);
        
        const downloadLink = document.createElement('a');
        downloadLink.href = url;
        downloadLink.download = filename;
        downloadLink.style.display = 'none';
        
        document.body.appendChild(downloadLink);
        downloadLink.click();
        document.body.removeChild(downloadLink);
        
        URL.revokeObjectURL(url);

        // 顯示成功訊息
        const deviceCount = ZoneFullMap_devices.length;
        const zoneCount = gZoneFullMap_Quadrilaterals.length;
        alert(`配置匯出成功！\n設備: ${deviceCount}, Zone: ${zoneCount}\n檔案: ${filename}`);

        console.log(`配置已匯出: ${filename}`);

    } catch (error) {
        console.error('匯出失敗:', error);
        alert(`匯出失敗: ${error.message}`);
    }
}


function ZoneFullMap_import() {
    const fileInput = document.createElement('input');
    fileInput.type = 'file';
    fileInput.accept = '.json';
    fileInput.style.display = 'none';

    fileInput.onchange = function(e) {
        if (e.target.files && e.target.files[0]) {
            const file = e.target.files[0];
            
            if (!file.name.toLowerCase().endsWith('.json')) {
                alert('請選擇JSON格式的檔案');
                return;
            }

            if (ZoneFullMap_devices.length > 0) {
                if (!confirm(`目前有 ${ZoneFullMap_devices.length} 個設備。匯入將清除所有現有內容，確定繼續？`)) {
                    return;
                }
            }

            const reader = new FileReader();
            
            reader.onload = function(event) {
                try {
                    const configData = JSON.parse(event.target.result);
                    
                    ZoneFullMap_clearAllDevices();

                    let importedCounts = {
                        microphones: 0,
                        cameras: 0,
                        visions: 0,
                        anchors: 0,
                        zones: 0
                    };

                    if (configData.devices) {
                        if (configData.devices.microphones) {
                            configData.devices.microphones.forEach(micData => {
                                const device = {
                                    id: micData.id,
                                    type: 'microphone',
                                    name: micData.name,
                                    x: micData.x,
                                    y: micData.y,
                                    angle: micData.angle || 0,
                                    status: micData.status || 'Connected',
                                    imgSrc: micData.imgSrc || './images/Web_RMCG.png',
                                    baseWidthM: micData.baseWidthM || 0.6,
                                    canDelete: micData.canDelete !== false,
                                    imageLoaded: false,
                                    imageElement: null,
                                    connectedTabIndex: micData.connectedTabIndex,
                                    displayNumber: micData.displayNumber,
                                    ip: micData.ip || '',
                                    microphoneType: micData.microphoneType || 'Yamaha:RM-CG(Coordinate)'
                                };
                                ZoneFullMap_devices.push(device);
                                ZoneFullMap_preloadMicrophoneImage(device);
                                importedCounts.microphones++;
                            });
                        }

                        if (configData.devices.cameras) {
                            configData.devices.cameras.forEach(camData => {
                                const device = {
                                    id: camData.id,
                                    type: 'camera',
                                    name: camData.name,
                                    x: camData.x,
                                    y: camData.y,
                                    angle: camData.angle || 0,
                                    fovAngle: camData.fovAngle || 60,
                                    offsetAngle: camData.offsetAngle || 0,
                                    status: camData.status || 'Connected',
                                    baseWidthM: camData.baseWidthM || 0.4,
                                    isSelectedCamera: camData.isSelectedCamera || false,
                                    canDelete: camData.canDelete !== false
                                };
                                ZoneFullMap_devices.push(device);
                                importedCounts.cameras++;
                            });
                        }

                        if (configData.devices.visions) {
                            configData.devices.visions.forEach(visionData => {
                                const device = {
                                    id: visionData.id,
                                    type: 'vision',
                                    name: visionData.name,
                                    x: visionData.x,
                                    y: visionData.y,
                                    angle: visionData.angle || 0,
                                    status: visionData.status || 'Connected',
                                    baseWidthM: visionData.baseWidthM || 0.5,
                                    canDelete: visionData.canDelete !== false
                                };
                                ZoneFullMap_devices.push(device);
                                importedCounts.visions++;
                            });
                        }

                        if (configData.devices.anchors) {
                            configData.devices.anchors.forEach(anchorData => {
                                const device = {
                                    id: anchorData.id,
                                    type: 'anchor',
                                    name: anchorData.name,
                                    x: anchorData.x,
                                    y: anchorData.y,
                                    status: anchorData.status || 'Active',
                                    baseWidthM: anchorData.baseWidthM || 0.3,
                                    canDelete: anchorData.canDelete !== false
                                };
                                ZoneFullMap_devices.push(device);
                                importedCounts.anchors++;
                            });
                        }
                    }

                    if (configData.zones) {
                        configData.zones.forEach(zoneData => {
                            const quad = new ZoneFullMap_Quadrilateral(zoneData.points, zoneData.deviceId);
                            quad.id = zoneData.id;
                            quad.isSelected = zoneData.isSelected || false;
                            quad.color = zoneData.color || '#FFD700';
                            quad.originalZoneID = zoneData.originalZoneID;
                            quad.originalZoneName = zoneData.originalZoneName;
                            quad.relativeToDevice = zoneData.relativeToDevice;
                            quad.initializeColorScheme();
                            
                            gZoneFullMap_Quadrilaterals.push(quad);
                            if (!gZoneFullMap_ActiveRectangles.includes(quad.id)) {
                                gZoneFullMap_ActiveRectangles.push(quad.id);
                            }
                            importedCounts.zones++;
                        });
                    }

                    if (configData.connections) {
                        if (configData.connections.micIdToTabMapping) {
                            gZoneFullMap_MicIdToTabMapping = { ...configData.connections.micIdToTabMapping };
                        }
                        if (configData.connections.tabToMicIdMapping) {
                            gZoneFullMap_TabToMicIdMapping = { ...configData.connections.tabToMicIdMapping };
                        }
                    }

                    ZoneFullMap_updateButtonStates();
                    setTimeout(() => {
                        ZoneFullMap_updateDevicePositions();
                        ZoneFullMap_redrawQuadrilaterals();
                        if (gDragSystem && gDragSystem.drawGrid) {
                            gDragSystem.drawGrid();
                        }
                    }, 100);

                    const totalDevices = Object.values(importedCounts).slice(0, 4).reduce((a, b) => a + b, 0);
                    alert(`匯入成功！\n設備: ${totalDevices}, Zone: ${importedCounts.zones}`);
                    
                    console.log('匯入完成:', importedCounts);

                } catch (error) {
                    console.error('匯入失敗:', error);
                    alert(`匯入失敗: ${error.message}`);
                }
            };

            reader.onerror = function() {
                alert('檔案讀取失敗');
            };

            reader.readAsText(file, 'utf-8');
        }
    };

    document.body.appendChild(fileInput);
    fileInput.click();
    document.body.removeChild(fileInput);
}

function ZoneFullMap_updateSoundPosition(msg) {

    const zoneMapWindow = document.getElementById('Zone_Full_Map_popup_Window');
    if (!zoneMapWindow || zoneMapWindow.style.display === 'none') {
        return;
    }

    const micDevice = ZoneFullMap_getMicrophoneByTabIndex(msg.SoundTabIndex);
        
    if (!micDevice) {
        //console.warn(`No microphone found for tab ${msg.SoundTabIndex} in ZoneFullMap`);
        return;
    }

    // console.log('ZoneFullMap_updateSoundPosition received:', {
    //     tabIndex: msg.SoundTabIndex,
    //     posIndex: msg.PosIndex,
    //     isOn: msg.IsOn,
    //     isCallPreset: msg.IsCallPreset
    // });

    let status = 'off';
    if (msg.IsOn === true) {
        status = 'active';
    } else if (msg.IsCallPreset === true) {
        status = 'preset';
    }

    // if(msg.SoundTabIndex == 0 && msg.PosIndex == 0)
    // {
    //     console.log(`Setting status for Mic${msg.SoundTabIndex} Zone${msg.PosIndex}: ${status}`);
    // }

    //ZoneFullMap_setZoneStatus(msg.SoundTabIndex, msg.PosIndex, status);
    
    // ZoneFullMap_setZoneStatus(msg.SoundTabIndex, msg.PosIndex, status);

    // if (gSelectedDevice && gSelectedDevice.id === micDevice.id) {
    //     requestAnimationFrame(() => {
    //         ZoneFullMap_updateDevicePositions();
    //         ZoneFullMap_redrawQuadrilaterals();
    //         ZoneFullMap_highlightSelectedDevice();
    //     });
    // }
}


function ZoneFullMap_testZoneStatus(tabIndex, zoneIndex, status) {
    console.log(`Testing zone status: Tab${tabIndex + 1} Zone${zoneIndex + 1} = ${status}`);
    
    // Find the microphone device by tab index
    const micDevice = ZoneFullMap_getMicrophoneByTabIndex(tabIndex);
    
    if (!micDevice) {
        console.warn(`Microphone ${tabIndex + 1} not found`);
        return false;
    }
    
    // Initialize zone states if not exists
    if (!micDevice.zoneStates) {
        micDevice.zoneStates = {};
    }
    
    // Set the zone status
    micDevice.zoneStates[zoneIndex] = status;
    
    // Find the corresponding zone quadrilateral
    const deviceZones = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === micDevice.id);
    if (deviceZones.length > zoneIndex) {
        deviceZones[zoneIndex].currentStatus = status;
    }
    
    console.log(`Set Mic${tabIndex + 1} Zone${zoneIndex + 1} status to: ${status}`);
    
    // Redraw to show the status change
    ZoneFullMap_redrawWithStatusRings();
    
    // Auto-clear status after 3 seconds (if not 'off')
    if (status !== 'off') {
        setTimeout(() => {
            ZoneFullMap_testZoneStatus(tabIndex, zoneIndex, 'off');
        }, 3000);
    }
    
    return true;
}

function ZoneFullMap_redrawWithStatusRings() {
    // Update device positions (which includes drawing microphones)
    ZoneFullMap_updateDevicePositions();
    
    // Draw status rings
    ZoneFullMap_drawMicrophoneStatusRings();
    
    // Redraw zones
    ZoneFullMap_redrawQuadrilaterals();
    
    // Maintain selected device highlight
    if (gSelectedDevice) {
        ZoneFullMap_highlightSelectedDevice();
    }
}

function ZoneFullMap_drawMicrophoneStatusRings() {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_highLight');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    // Draw rings for each microphone with active zone states
    ZoneFullMap_devices.forEach(device => {
        if (device.type !== 'microphone' || !device.zoneStates) return;
        
        const widthPx = device.baseWidthM * scale * 100;
        const cx = ox + device.x * scale / 100;
        const cy = oy - device.y * scale / 100;
        
        // Check if any zones have active status
        const activeStates = Object.values(device.zoneStates).filter(status => status !== 'off');
        
        if (activeStates.length > 0) {
            // Determine ring color based on priority: active > preset
            let ringColor = '#FF9800'; // Default orange for preset
            if (activeStates.includes('active')) {
                ringColor = '#4CAF50'; // Green for active
            }
            
            // Draw animated ring
            const time = Date.now();
            const pulseEffect = (Math.sin(time / 300) + 1) / 2; // Slower pulse
            const baseRadius = widthPx / 2 + 8;
            const ringRadius = baseRadius + pulseEffect * 6;
            const alpha = 0.6 + pulseEffect * 0.4;
            
            ctx.save();
            ctx.beginPath();
            ctx.arc(cx, cy, ringRadius, 0, 2 * Math.PI);
            ctx.strokeStyle = ringColor;
            ctx.lineWidth = 4;
            ctx.globalAlpha = alpha;
            ctx.stroke();
            
            // Add inner glow effect
            ctx.beginPath();
            ctx.arc(cx, cy, ringRadius - 2, 0, 2 * Math.PI);
            ctx.strokeStyle = 'white';
            ctx.lineWidth = 1;
            ctx.globalAlpha = alpha * 0.5;
            ctx.stroke();
            
            ctx.restore();
        }
    });
}


function ZoneFullMap_runZoneStatusTests() {
    console.log('\n=== Starting Zone Status Tests ===');

    
    // Check if ZoneFullMap window is visible
    const popup = document.getElementById('Zone_Full_Map_popup_Window');
    if (!popup || popup.style.display === 'none') {
        console.error('❌ ZoneFullMap window is not open. Please open the Zone Full Map first.');
        alert('Please open the Zone Full Map window first by clicking the Zone Full Map button.');
        return;
    }
    
    // Check if microphones exist
    const microphones = ZoneFullMap_devices.filter(device => device.type === 'microphone');
    if (microphones.length === 0) {
        console.warn('❌ No microphones found for testing');
        alert('Please add microphones first before testing zone status\n\nUse "Add MIC." button to add microphones.');
        return;
    }
    
    console.log(`✅ Found ${microphones.length} microphones for testing`);
    
    // Check if any microphones have zones
    const microphonesWithZones = microphones.filter(mic => {
        const zones = gZoneFullMap_Quadrilaterals.filter(zone => zone.deviceId === mic.id);
        return zones.length > 0;
    });
    
    if (microphonesWithZones.length === 0) {
        console.warn('❌ No microphones have zones for testing');
        alert('Please create zones for microphones first before testing zone status\n\n1. Click "地圖編輯模式" button\n2. Select a microphone\n3. Draw zones by clicking and dragging');
        return;
    }
    
    console.log(`✅ Found ${microphonesWithZones.length} microphones with zones`);
    
    // Create a smarter test sequence based on available microphones and zones
    const testSequence = [];
    let delay = 0;
    
    microphonesWithZones.forEach((mic, micIndex) => {
        const tabIndex = mic.connectedTabIndex !== undefined ? mic.connectedTabIndex : micIndex;
        const zones = gZoneFullMap_Quadrilaterals.filter(zone => zone.deviceId === mic.id);
        
        // Test first few zones of each microphone
        zones.slice(0, 3).forEach((zone, zoneIndex) => {
            testSequence.push({
                delay: delay,
                tabIndex: tabIndex,
                zoneIndex: zoneIndex,
                status: zoneIndex % 2 === 0 ? 'active' : 'preset',
                description: `${mic.name} ${zone.getDisplayZoneId()} ${zoneIndex % 2 === 0 ? 'Active' : 'Preset'}`
            });
            delay += 1500; // 1.5 second intervals
        });
    });
    
    if (testSequence.length === 0) {
        console.error('❌ No valid test sequence could be created');
        return;
    }
    
    console.log(`🎬 Starting test sequence with ${testSequence.length} tests:`);
    testSequence.forEach((test, index) => {
        console.log(`  ${index + 1}. ${test.description} (delay: ${test.delay}ms)`);
    });
    
    // Execute test sequence
    testSequence.forEach(test => {
        setTimeout(() => {
            console.log(`🎯 Executing: ${test.description}`);
            ZoneFullMap_testZoneStatus(test.tabIndex, test.zoneIndex, test.status);
        }, test.delay);
    });
    
    // Clear all statuses after test
    const clearDelay = delay + 3000;
    setTimeout(() => {
        console.log('\n🧹 Clearing all test statuses');
        microphonesWithZones.forEach((mic, micIndex) => {
            const tabIndex = mic.connectedTabIndex !== undefined ? mic.connectedTabIndex : micIndex;
            const zones = gZoneFullMap_Quadrilaterals.filter(zone => zone.deviceId === mic.id);
            zones.forEach((zone, zoneIndex) => {
                ZoneFullMap_testZoneStatus(tabIndex, zoneIndex, 'off');
            });
        });
        console.log('✅ Zone Status Tests Complete');
    }, clearDelay);
}

function ZoneFullMap_updateSingleZoneStatus(deviceId, zoneIndex, status) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device || device.type !== 'microphone') return;
    
    // 更新狀態
    if (!device.zoneStates) device.zoneStates = {};
    device.zoneStates[zoneIndex] = status;
    
    // 找到對應的Zone
    const deviceZones = gZoneFullMap_Quadrilaterals.filter(quad => quad.deviceId === deviceId);
    if (deviceZones.length > zoneIndex) {
        const targetZone = deviceZones[zoneIndex];
        targetZone.currentStatus = status;
        
        // 只重繪這個Zone
        ZoneFullMap_redrawSingleZone(targetZone);
    }
    
    // 更新麥克風狀態環
    ZoneFullMap_updateSingleMicrophoneRing(device);
}

function ZoneFullMap_redrawSingleZone(zone) {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_rect');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    // 計算Zone的邊界
    const screenPoints = zone.points.map(point => ({
        x: ox + point.x * scale / 100,
        y: oy - point.y * scale / 100
    }));
    
    const bounds = {
        minX: Math.min(...screenPoints.map(p => p.x)) - 10,
        minY: Math.min(...screenPoints.map(p => p.y)) - 10,
        maxX: Math.max(...screenPoints.map(p => p.x)) + 10,
        maxY: Math.max(...screenPoints.map(p => p.y)) + 10
    };
    
    // 清除該Zone區域
    ctx.clearRect(bounds.minX, bounds.minY, 
                  bounds.maxX - bounds.minX, 
                  bounds.maxY - bounds.minY);
    
    // 重繪該Zone
    zone.draw(ctx);
}

function ZoneFullMap_updateSingleMicrophoneRing(device) {
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_highLight');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
    
    const widthPx = device.baseWidthM * scale * 100;
    const cx = ox + device.x * scale / 100;
    const cy = oy - device.y * scale / 100;
    
    const clearRadius = widthPx / 2 + 20;
    ctx.clearRect(cx - clearRadius, cy - clearRadius, 
                  clearRadius * 2, clearRadius * 2);
    
    if (device.zoneStates) {
        const activeStates = Object.values(device.zoneStates).filter(status => status !== 'off');
        
        if (activeStates.length > 0) {
            let ringColor = '#FF9800';
            if (activeStates.includes('active')) {
                ringColor = '#4CAF50';
            }
            
            const time = Date.now();
            const pulseEffect = (Math.sin(time / 300) + 1) / 2;
            const baseRadius = widthPx / 2 + 8;
            const ringRadius = baseRadius + pulseEffect * 6;
            const alpha = 0.6 + pulseEffect * 0.4;
            
            ctx.save();
            ctx.beginPath();
            ctx.arc(cx, cy, ringRadius, 0, 2 * Math.PI);
            ctx.strokeStyle = ringColor;
            ctx.lineWidth = 4;
            ctx.globalAlpha = alpha;
            ctx.stroke();
            ctx.restore();
        }
    }
}

const ZoneFullMap_DeviceVisualConfig = {
    camera: {
        baseCircleSize: 0.3,        // 基礎圓圈大小（米）
        circleColor: {
            normal: '#757575',       // 一般狀態顏色（灰色）
            selected: '#4CAF50'      // 選中狀態顏色（綠色）
        },
        arrow: {
            baseLength: 3,          // 基礎箭頭長度（像素）
            lengthMultiplier: 2,   // 長度倍數（相對於圓圈大小）
            hardwareWidth: 5,        // 硬體方向箭頭寬度
            fovWidth: 5,            // FOV方向箭頭寬度
            headSizeRatio: 0.6,     // 箭頭頭部大小比例
            dashPattern: [8, 4]     // 虛線模式
        },
        border: {
            width: 2,               // 邊框寬度
            color: '#000000'        // 邊框顏色
        }
    },
    bc200: {
        baseCircleSize: 0.3,        // BC-200圓圈稍大一點
        circleColor: {
            normal: '#2196F3',       // BC-200藍色
            selected: '#1976D2'      // 選中時深藍色
        },
        arrow: {
            baseLength: 3,          // BC-200箭頭更長
            lengthMultiplier: 2,   // 更大的長度倍數
            width: 5,               // 更粗的箭頭線條
            headSizeRatio: 0.6,     // 更大的箭頭頭部
            color: '#2196F3'        // 箭頭顏色
        },
        border: {
            width: 2,
            color: '#000000'
        }
    },
    microphone: {
        baseCircleSize: 0.6,        // 麥克風保持原大小
    }
};

const ZoneFullMap_RenderCache = {
    deviceAngles: new Map(),
    
    // 檢查角度是否有變化
    angleChanged: function(deviceId, angle, fov, offset) {
        const key = `${deviceId}`;
        const newValue = `${angle}_${fov}_${offset}`;
        const oldValue = this.deviceAngles.get(key);
        
        if (oldValue !== newValue) {
            this.deviceAngles.set(key, newValue);
            return true;
        }
        return false;
    }
};



let ZoneFullMap_lastRedrawTime = 0;

function ZoneFullMap_throttledRedraw() {
    const now = Date.now();
    if (now - ZoneFullMap_lastRedrawTime < 16) return; // 60fps 限制
    
    ZoneFullMap_lastRedrawTime = now;
    
    requestAnimationFrame(() => {
        ZoneFullMap_updateDevicePositions();
        ZoneFullMap_highlightSelectedDevice();
        if (gSelectedDevice) {
            ZoneFullMap_updateDeviceInfoPosition(gSelectedDevice);
        }
    });
}

function ZoneFullMap_normalizeAngle(angle) 
{
    return ((angle % 360) + 360) % 360;
}

function ZoneFullMap_angleToRadians(angle) 
{
    return ((angle - 90) * Math.PI) / 180;
}

function ZoneFullMap_updateDeviceDirection(deviceId, direction) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return;
    
    const newAngle = parseInt(direction);
    device.angle = newAngle;
    
    // 清除快取以強制重繪
    delete device._cachedRenderData;
    delete device._cachedAngle;
    
    console.log(`Updated ${device.name} direction to ${newAngle}°`);
    ZoneFullMap_updateDevicePositions();
}


function ZoneFullMap_updateDeviceFov(deviceId, fovAngle) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return;
    
    const newFov = Math.max(0, Math.min(360, parseInt(fovAngle) || 0));
    device.fovAngle = newFov;
    
    // 清除快取以強制重繪
    delete device._cachedRenderData;
    delete device._cachedAngle;
    
    console.log(`Updated ${device.name} FOV to ${newFov}°`);
    ZoneFullMap_updateDevicePositions();
}

function ZoneFullMap_updateDeviceOffset(deviceId, offsetAngle) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return;
    
    const newOffset = Math.max(-360, Math.min(360, parseInt(offsetAngle) || 0));
    device.offsetAngle = newOffset;
    
    // 清除快取以強制重繪
    delete device._cachedRenderData;
    delete device._cachedAngle;
    
    console.log(`Updated ${device.name} offset to ${newOffset}°`);
    ZoneFullMap_updateDevicePositions();
}

function ZoneFullMap_updateDevicePosition(deviceId, x, y) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return;
    
    const oldX = device.x;
    const oldY = device.y;
    
    device.x = x;
    device.y = y;
    
    if (device.type === 'microphone') {
        const deltaX = device.x - oldX;
        const deltaY = device.y - oldY;
        ZoneFullMap_syncDeviceZones(deviceId, deltaX, deltaY);
    }
    
    console.log(`Updated ${device.name} position to (${x}, ${y})`);
    ZoneFullMap_updateDevicePositions();
    
    if (gSelectedDevice && gSelectedDevice.id === deviceId) {
        ZoneFullMap_highlightSelectedDevice();
    }
}

function ZoneFullMap_updateDevicesInBatch(updates) {
    
    const updatedDevices = [];
    
    updates.forEach(update => {
        const device = ZoneFullMap_findDeviceById(update.deviceId);
        if (!device) {
            console.warn(`Device ${update.deviceId} not found`);
            return;
        }
        
        const { params } = update;
        let hasChanges = false;
        
        if (params.x !== undefined || params.y !== undefined) {
            const oldX = device.x;
            const oldY = device.y;
            
            if (params.x !== undefined) device.x = params.x;
            if (params.y !== undefined) device.y = params.y;
            
            if (device.type === 'microphone') {
                const deltaX = device.x - oldX;
                const deltaY = device.y - oldY;
                ZoneFullMap_syncDeviceZones(device.id, deltaX, deltaY);
            }
            
            hasChanges = true;
        }
        
        if (params.angle !== undefined) {
            device.angle = ZoneFullMap_normalizeAngle(params.angle);
            hasChanges = true;
        }
        
        if (params.fovAngle !== undefined) {
            device.fovAngle = Math.max(0, Math.min(360, params.fovAngle));
            hasChanges = true;
        }
        
        if (params.offsetAngle !== undefined) {
            device.offsetAngle = Math.max(-360, Math.min(360, params.offsetAngle));
            hasChanges = true;
        }
        
        if (hasChanges) {
            delete device._cachedRenderData;
            delete device._cachedAngle;
            
            updatedDevices.push({
                deviceId: device.id,
                deviceName: device.name,
                updatedParams: params
            });
        }
    });
    
    if (updatedDevices.length > 0) {
        ZoneFullMap_updateDevicePositions();
        
        if (gSelectedDevice) {
            ZoneFullMap_highlightSelectedDevice();
        }
        
        console.log('Batch update completed:', updatedDevices);
    }
    
    return updatedDevices;
}

function ZoneFullMap_exampleUsage() {
    ZoneFullMap_updateDevice('camera_1', { 
        x: 15000,        // 1.5米
        y: 25000,        // 2.5米
        angle: 90,       // 向右
        fovAngle: 60,    // 60度FOV
        offsetAngle: 15  // 偏移15度
    });
    
    // 只更新角度
    ZoneFullMap_updateDevice('vision_1', { 
        angle: 180,      // 向下
        fovAngle: 45,    // 45度FOV
        offsetAngle: -10 // 偏移-10度
    });
    
    // 只更新位置
    ZoneFullMap_updateDevice('camera_2', { 
        x: 30000,        // 3.0米
        y: 40000         // 4.0米
    });
    
    // // 批量更新
    // const batchUpdates = [
    //     {
    //         deviceId: 'camera_1',
    //         params: { angle: 0, fovAngle: 90, offsetAngle: 0 }
    //     },
    //     {
    //         deviceId: 'vision_1', 
    //         params: { x: 20000, y: 30000, angle: 270 }
    //     }
    // ];
    
    // ZoneFullMap_updateMultipleDevices(batchUpdates);
    
    // 獲取設備參數
    const cameraParams = ZoneFullMap_getDeviceParams('camera_1');
    console.log('Camera 1 params:', cameraParams);
    
    // 獲取所有相機
    const allCameras = ZoneFullMap_getDevicesByType('camera');
    console.log('All cameras:', allCameras.map(cam => cam.name));
}

function ZoneFullMap_getDeviceParams(deviceId) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return null;
    
    return {
        id: device.id,
        name: device.name,
        type: device.type,
        x: device.x,
        y: device.y,
        angle: device.angle || 0,
        fovAngle: device.fovAngle || 0,
        offsetAngle: device.offsetAngle || 0,
        status: device.status || 'Connected'
    };
}

function ZoneFullMap_getDevicesByType(deviceType) {
    return ZoneFullMap_devices.filter(device => device.type === deviceType);
}

function ZoneFullMap_updateMultipleDevices(updates) {
    
    const results = [];
    let totalUpdated = 0;
    
    updates.forEach(update => {
        const success = ZoneFullMap_updateDevice(update.deviceId, update.params);
        results.push({
            deviceId: update.deviceId,
            success: success,
            params: update.params
        });
        
        if (success) totalUpdated++;
    });
    
    console.log(`Batch update completed: ${totalUpdated}/${updates.length} devices updated`);
    return results;
}

function ZoneFullMap_updateDevice(deviceId, params) 
{
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) {
        console.warn(`Device ${deviceId} not found`);
        return false;
    }
    
    let hasChanges = false;
    const oldValues = {
        x: device.x,
        y: device.y,
        angle: device.angle,
        fovAngle: device.fovAngle,
        offsetAngle: device.offsetAngle
    };
    
    // 更新位置
    if (params.x !== undefined) {
        device.x = params.x;
        hasChanges = true;
    }
    
    if (params.y !== undefined) {
        device.y = params.y;
        hasChanges = true;
    }
    
    // 更新角度參數
    if (params.angle !== undefined) {
        device.angle = ZoneFullMap_normalizeAngle(params.angle);
        hasChanges = true;
    }
    
    if (params.fovAngle !== undefined) {
        device.fovAngle = Math.max(0, Math.min(360, params.fovAngle));
        hasChanges = true;
    }
    
    if (params.offsetAngle !== undefined) {
        device.offsetAngle = Math.max(-360, Math.min(360, params.offsetAngle));
        hasChanges = true;
    }
    
    if (hasChanges) {
        // 處理麥克風區域同步
        if (device.type === 'microphone' && (params.x !== undefined || params.y !== undefined)) {
            const deltaX = device.x - oldValues.x;
            const deltaY = device.y - oldValues.y;
            if (deltaX !== 0 || deltaY !== 0) {
                ZoneFullMap_syncDeviceZones(deviceId, deltaX, deltaY);
            }
        }
        
        // 清除渲染快取
        delete device._cachedRenderData;
        delete device._cachedAngle;
        
        // 更新畫布
        ZoneFullMap_updateDevicePositions();
        
        // 如果這個設備正被選中，更新highlight
        if (gSelectedDevice && gSelectedDevice.id === deviceId) {
            ZoneFullMap_highlightSelectedDevice();
            ZoneFullMap_updateDeviceInfoPosition(device);
        }
        
        console.log(`Updated device ${deviceId}:`, {
            deviceName: device.name,
            oldValues,
            newValues: params,
            deltaPosition: device.type === 'microphone' ? {
                x: device.x - oldValues.x,
                y: device.y - oldValues.y
            } : null
        });
    }
    
    return hasChanges;
}

/**
 * 將配置文件中的設備數據轉換為畫布座標系統
 * 座標系統說明：
 * - 測量原點 (0,0) 為測量設備位置
 * - Pan = 0° 指向 +Y 軸（北/前方）
 * - Pan = 90° 指向 +X 軸（東/右方）
 * - Pan = -90°/270° 指向 -X 軸（西/左方）
 * - Pan = ±180° 指向 -Y 軸（南/後方）
 */
function processConfigToCanvas(configData) {
    console.log('=== 開始處理配置數據 ===\n');
    
    // 解析配置數據
    const config = typeof configData === 'string' ? JSON.parse(configData) : configData;
    
    // 檢查單位來決定是否需要轉換
    const distanceUnit = config.calibrationParameters?.units?.distance || 'meters';
    const isMeters = distanceUnit.toLowerCase() === 'meters';
    const distanceMultiplier = isMeters ? 100 : 1;
    
    console.log(`距離單位: ${distanceUnit} (轉換倍率: ${distanceMultiplier})`);
    
    // 準備轉換後的數據結構
    const canvasData = {
        clearExisting: true,
        measurementDeviceHeight: 100,
        microphones: [],
        cameras: [],
        visions: []
    };
    
    // 角度映射表
    const angleMap = {
        '+X': 90,
        '-X': 270,
        '+Y': 0,
        '-Y': 180
    };
    
    // 處理 BC200 設備（視覺系統）
    if (config.devices?.BC200) {
        console.log('--- 處理 BC200 設備 ---');
        config.devices.BC200.forEach((device, index) => {
            console.log(`BC200 ${index + 1} (${device.id}):`);
            console.log(`  原始數據: 距離=${device.distance}${distanceUnit}, 水平角=${device.horizontalAngle}°, 垂直角=${device.verticalAngle}°`);
            
            const distanceCm = device.distance * distanceMultiplier;
            const position = MeasurementDevice.polarTo2D(
                distanceCm,
                device.horizontalAngle,
                device.verticalAngle
            );
            
            const deviceAngle = angleMap[device.axisDirection] || 0;
            
            canvasData.visions.push({
                x: position.x,
                y: position.y,
                angle: deviceAngle,
                id: device.id,
                deviceNumber: device.deviceNumber,
                quality: device.quality,
                measurementTime: device.measurementTime,
                status: device.status
            });
            
            console.log(`  轉換後位置: (${position.x_cm.toFixed(1)}, ${position.y_cm.toFixed(1)}) cm`);
            console.log(`  設備角度: ${deviceAngle}° (${device.axisDirection})`);
        });
    }
    
    // 處理 PTZ 攝影機
    if (config.devices?.PTZ) {
        console.log('\n--- 處理 PTZ 攝影機 ---');
        config.devices.PTZ.forEach((device, index) => {
            console.log(`PTZ ${index + 1} (${device.id}):`);
            console.log(`  原始數據: 距離=${device.distance}${distanceUnit}, 水平角=${device.horizontalAngle}°, 垂直角=${device.verticalAngle}°`);
            
            const distanceCm = device.distance * distanceMultiplier;
            const position = MeasurementDevice.polarTo2D(
                distanceCm,
                device.horizontalAngle,
                device.verticalAngle
            );
            
            const baseAngle = angleMap[device.axisDirection] || 0;
            
            const cameraConfig = {
                x: position.x,
                y: position.y,
                angle: baseAngle,
                id: device.id,
                deviceNumber: device.deviceNumber,
                quality: device.quality,
                measurementTime: device.measurementTime,
                status: device.status,
                axisDirection: device.axisDirection
            };
            
            // 只有在 currentPosition 存在時才處理 FOV
            if (device.currentPosition) {
                cameraConfig.fovAngle = device.currentPosition.pan;
                cameraConfig.currentViewAngle = (baseAngle + device.currentPosition.pan + 360) % 360;
                cameraConfig.tiltAngle = device.currentPosition.tilt;
                console.log(`  FOV偏移: ${cameraConfig.fovAngle}°`);
                console.log(`  當前視角方向: ${cameraConfig.currentViewAngle}°`);
            }
            
            canvasData.cameras.push(cameraConfig);
            
            console.log(`  轉換後位置: (${position.x_cm.toFixed(1)}, ${position.y_cm.toFixed(1)}) cm`);
            console.log(`  硬體基礎角度: ${baseAngle}° (${device.axisDirection})`);
        });
    }
    
    // 處理麥克風陣列 - 支援多組麥克風
    if (config.devices?.MIC) {
        console.log('\n--- 處理麥克風陣列 ---');
        
        const micArray = config.devices.MIC;
        
        // 檢查數據格式
        if (micArray[0]?.positions) {
            // 新格式：每個麥克風設備包含 positions 陣列
            micArray.forEach((micDevice, index) => {
                processSingleMicrophoneArray(micDevice, index);
            });
        } else {
            // 舊格式：根據 ID 分組處理
            const micGroups = {};
            
            // 根據 ID 將麥克風點分組
            micArray.forEach(mic => {
                if (!micGroups[mic.id]) {
                    micGroups[mic.id] = {
                        id: mic.id,
                        points: {}
                    };
                }
                micGroups[mic.id].points[mic.location] = mic;
            });
            
            // 處理每一組麥克風
            Object.values(micGroups).forEach((group, index) => {
                console.log(`\n麥克風陣列 ${index + 1} (${group.id}):`);
                
                const rightFront = group.points.RightFront;
                const center = group.points.Center;
                const rightBack = group.points.RightBack;
                
                if (rightFront && center && rightBack) {
                    processMicrophonePoints(rightFront, center, rightBack, group.id, index);
                } else {
                    console.log('  警告：麥克風組不完整，跳過');
                }
            });
        }
        
        // 處理單組麥克風（新格式）
        function processSingleMicrophoneArray(micDevice, index) {
            console.log(`\n麥克風陣列 ${index + 1} (${micDevice.id}):`);
            
            const rightFront = micDevice.positions.find(p => 
                p.location === 'RightFront' || p.position === 'RightFront'
            );
            const center = micDevice.positions.find(p => 
                p.location === 'Center' || p.position === 'Center'
            );
            const rightBack = micDevice.positions.find(p => 
                p.location === 'RightBack' || p.position === 'RightBack'
            );
            
            if (rightFront && center && rightBack) {
                processMicrophonePoints(rightFront, center, rightBack, micDevice.id, index);
            }
        }
        
        // 處理麥克風三點測量
        function processMicrophonePoints(rightFront, center, rightBack, micId, index) {
            console.log('  找到完整的麥克風陣列配置');
            
            // 建立測量數據格式
            const measurements = {
                rightFront: {
                    distance: rightFront.distance * distanceMultiplier,
                    pan: rightFront.horizontalAngle,
                    tilt: rightFront.verticalAngle
                },
                center: {
                    distance: center.distance * distanceMultiplier,
                    pan: center.horizontalAngle,
                    tilt: center.verticalAngle
                },
                rightBack: {
                    distance: rightBack.distance * distanceMultiplier,
                    pan: rightBack.horizontalAngle,
                    tilt: rightBack.verticalAngle
                }
            };
            
            console.log('  麥克風測量數據:');
            console.log(`    右前: 距離=${measurements.rightFront.distance.toFixed(1)}cm, pan=${measurements.rightFront.pan}°`);
            console.log(`    中心: 距離=${measurements.center.distance.toFixed(1)}cm, pan=${measurements.center.pan}°`);
            console.log(`    右後: 距離=${measurements.rightBack.distance.toFixed(1)}cm, pan=${measurements.rightBack.pan}°`);
            
            // 使用 MeasurementDevice 處理麥克風數據
            const processed = MeasurementDevice.processMicrophoneMeasurement(measurements);
            
            const micConfig = {
                x: processed.position.x,
                y: processed.position.y,
                angle: processed.angle,
                ip: `192.168.1.${100 + index}`,
                microphoneType: 'Shure_MXA310',
                id: micId,
                status: 'active'
            };
            
            // 如果有品質數據，計算平均值
            if (rightFront.quality !== undefined) {
                micConfig.quality = ((rightFront.quality || 0) + 
                                    (center.quality || 0) + 
                                    (rightBack.quality || 0)) / 3;
            }
            
            console.log(`  最終麥克風配置:`);
            console.log(`    位置: (${(micConfig.x/100).toFixed(1)}, ${(micConfig.y/100).toFixed(1)}) cm`);
            console.log(`    角度: ${micConfig.angle}°`);
            if (micConfig.quality) {
                console.log(`    平均測量品質: ${micConfig.quality.toFixed(2)}%`);
            }
            
            canvasData.microphones.push(micConfig);
        }
    }
    
    console.log('\n=== 轉換完成 ===');
    console.log('最終畫布數據統計:', {
        麥克風數量: canvasData.microphones.length,
        攝影機數量: canvasData.cameras.length,
        視覺系統數量: canvasData.visions.length
    });
    console.log('詳細數據:', canvasData);
    
    // 呼叫批次添加函數來顯示設備
    if (typeof ZoneFullMap_batchAddDevices === 'function') {
        console.log('\n呼叫 ZoneFullMap_batchAddDevices 顯示設備...');
        ZoneFullMap_batchAddDevices(canvasData);
    } else {
        console.log('\n注意: ZoneFullMap_batchAddDevices 函數不存在，無法自動顯示設備');
    }
    
    return canvasData;
}

/**
 * 直接使用提供的配置數據進行轉換
 */
function convertAndDisplayDevices() {
    // 使用您提供的配置數據
    const configData = {
        "calibrationParameters": {
            "coordinateSystem": "cartesian",
            "referenceFrame": {
                "orientation": "NED",
                "origin": [
                    0,
                    0,
                    0
                ]
            },
            "units": {
                "angle": "degrees",
                "distance": "meters"
            }
        },
        "devices": {
            "BC200": [
                {
                    "axisDirection": "-Y",
                    "distance": 5.6,
                    "horizontalAngle": -4.14,
                    "id": "BC200_001",
                    "sensorReading": {
                        "x": 0,
                        "y": 0
                    },
                    "status": "active",
                    "verticalAngle": 13.09
                }
            ],
            "MIC": [
                {
                    "arrayPosition": 1,
                    "axisDirection": "-",
                    "distance": 4.9,
                    "horizontalAngle": 6.64,
                    "id": "MIC_001",
                    "location": "RightFront",
                    "status": "active",
                    "verticalAngle": 16.6
                },
                {
                    "arrayPosition": 2,
                    "axisDirection": "-",
                    "distance": 4.7,
                    "horizontalAngle": 5.93,
                    "id": "MIC_001",
                    "location": "RightBack",
                    "status": "active",
                    "verticalAngle": 17.51
                },
                {
                    "arrayPosition": 3,
                    "axisDirection": "+Y",
                    "distance": 4.8,
                    "horizontalAngle": 3.35,
                    "id": "MIC_001",
                    "location": "Center",
                    "status": "active",
                    "verticalAngle": 17.03
                },
                {
                    "arrayPosition": 1,
                    "axisDirection": "-",
                    "distance": 2.1,
                    "horizontalAngle": 5.72,
                    "id": "MIC_002",
                    "location": "RightFront",
                    "status": "active",
                    "verticalAngle": 42.66
                },
                {
                    "arrayPosition": 2,
                    "axisDirection": "-",
                    "distance": 1.8,
                    "horizontalAngle": 7.09,
                    "id": "MIC_002",
                    "location": "RightBack",
                    "status": "active",
                    "verticalAngle": 55.52
                },
                {
                    "arrayPosition": 3,
                    "axisDirection": "+Y",
                    "distance": 1.9,
                    "horizontalAngle": -3.25,
                    "id": "MIC_002",
                    "location": "Center",
                    "status": "active",
                    "verticalAngle": 48.79
                }
            ],
            "PTZ": [
                {
                    "axisDirection": "-X",
                    "currentPosition": {
                        "pan": 0,
                        "tilt": 0
                    },
                    "distance": 4.2,
                    "horizontalAngle": 9.66,
                    "id": "PTZ1",
                    "limits": {
                        "panMax": 180,
                        "panMin": -180,
                        "tiltMax": 90,
                        "tiltMin": -90
                    },
                    "status": "active",
                    "verticalAngle": 16.18
                },
                {
                    "axisDirection": "+X",
                    "currentPosition": {
                        "pan": 0,
                        "tilt": 0
                    },
                    "distance": 2.8,
                    "horizontalAngle": -12.48,
                    "id": "PTZ2",
                    "limits": {
                        "panMax": 180,
                        "panMin": -180,
                        "tiltMax": 90,
                        "tiltMin": -90
                    },
                    "status": "active",
                    "verticalAngle": 25.96
                }
            ],
            "calibrationPoint": {
                "description": "Calibration origin",
                "distance": 0,
                "horizontalAngle": 0,
                "verticalAngle": 0
            }
        },
        "site": "Global Mapping Calibration",
        "timestamp": "2025-09-10T06:05:54ZZ",
        "version": "1.0"
    };

    // 執行轉換並顯示
    return processConfigToCanvas(configData);
}

var gZoneFullMap_CameraConnectionData = {};
var gZoneFullMap_CurrentDeviceId = null;
var gZoneFullMap_CurrentCameraIP = null;

function updateZoneFullMapCameraConnection() {
    gZoneFullMap_CameraConnectionData = {};
    
    if (gCamConnectStatusMap && gCamConnectStatusMap.size > 0) {
        let index = 0;
        gCamConnectStatusMap.forEach((status, displayName) => {
            if (checkCamConnectStatus(displayName)) {
                gZoneFullMap_CameraConnectionData[index] = {
                    cameraIndex: index,
                    displayName: displayName,
                    ipAddress: extractIPFromDisplayName(displayName),
                    isConnect: checkCamConnectStatus(displayName),
                    connectStatus: status
                };
                index++;
            }
        });
    }
}

function getZoneFullMapAvailableCameras() {
    const availableCameras = [];
    
    Object.values(gZoneFullMap_CameraConnectionData).forEach(camData => {
        if (camData.isConnect) {
            availableCameras.push({
                value: camData.displayName,
                text: camData.displayName,
                data: camData
            });
        }
    });
    
    return availableCameras;
}

function getZoneFullMapCameraDataByName(displayName) {
    for (let key in gZoneFullMap_CameraConnectionData) {
        if (gZoneFullMap_CameraConnectionData[key].displayName === displayName) {
            return gZoneFullMap_CameraConnectionData[key];
        }
    }
    return null;
}

function ZoneFullMap_isCameraAlreadyAssigned(cameraDisplayName, excludeDeviceId) {
    if (!cameraDisplayName || cameraDisplayName === "-1" || cameraDisplayName === "null" || cameraDisplayName === "") {
        return false;
    }
    
    const allDevices = [...gZoneFullMap_Microphones, ...gZoneFullMap_Cameras, ...gZoneFullMap_Visions];
    
    for (const device of allDevices) {
        if (device.id !== excludeDeviceId && device.connectedCamera === cameraDisplayName) {
            return true;
        }
    }
    
    return false;
}

function generateZoneFullMapCameraHTML(device) {
    const availableCameras = getZoneFullMapAvailableCameras();
    let cameraSelectHTML = '';
    
    cameraSelectHTML = `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Camera:</span>
            <select onchange="handleZoneFullMapCameraSelection('${device.id}', this.value);" 
                id="ZoneFullMap_connectedCamera_${device.id}" 
                style="width: 180px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px; border: none;">`;
    
    const nullSelected = (!device.connectedCamera || device.connectedCamera === '') ? 'selected' : '';
    cameraSelectHTML += `<option value="" ${nullSelected}>null</option>`;
    
    availableCameras.forEach(camera => {
        const selected = device.connectedCamera === camera.value ? 'selected' : '';
        cameraSelectHTML += `<option value="${camera.value}" ${selected}>${camera.text}</option>`;
    });
    
    cameraSelectHTML += `
            </select>
        </div>`;
    
    let ipAddress = 'null';
    let connectStatus = 'null';
    
    if (device.connectedCamera && device.connectedCamera !== '') {
        const camData = getZoneFullMapCameraDataByName(device.connectedCamera);
        if (camData) {
            ipAddress = camData.ipAddress || 'null';
            connectStatus = camData.connectStatus || 'null';
        }
    }
    
    cameraSelectHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">IP:</span>
            <span id="ZoneFullMap_camera_ip_display_${device.id}" style="font-size: 18px;" class="ZoneFullMap_deviceInfo">${ipAddress}</span>
        </div>
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Status:</span>
            <span id="ZoneFullMap_camera_status_display_${device.id}" style="font-size: 18px;" class="ZoneFullMap_deviceInfo">
                ${connectStatus}
            </span>
        </div>`;

    // const isPTZEnabled = device.connectedCamera && connectStatus !== 'null' && connectStatus !== 'Unknown';
    let isPTZEnabled = true;//test
    cameraSelectHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 120px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">PTZ control:</span>
            <button id="ZoneFullMap_ptz_btn_${device.id}" 
                onclick="ZoneFullMap_PTZControl('${device.id}')" 
                class="ZoneFullMap_Btn_ptzctrl"
                ${isPTZEnabled ? '' : 'disabled'}>
                <span>PTZ</span>
            </button>
        </div>`;
    
    return cameraSelectHTML;
}

function handleZoneFullMapCameraSelection(deviceId, cameraValue) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return;
    
    device.connectedCamera = cameraValue;
    
    const ipDisplay = document.getElementById(`ZoneFullMap_camera_ip_display_${deviceId}`);
    const statusDisplay = document.getElementById(`ZoneFullMap_camera_status_display_${deviceId}`);
    const ptzBtn = document.getElementById(`ZoneFullMap_ptz_btn_${deviceId}`);
    
    if (cameraValue && cameraValue !== "") {
        const camData = getZoneFullMapCameraDataByName(cameraValue);
        if (camData) {
            if (ipDisplay) ipDisplay.textContent = camData.ipAddress || 'null';
            if (statusDisplay) statusDisplay.textContent = camData.connectStatus || 'null';
            
            if (ptzBtn) {
                const isEnabled = camData.connectStatus !== 'null' && camData.connectStatus !== 'Unknown';
                ptzBtn.disabled = !isEnabled;
            }
        }
    } else {
        if (ipDisplay) ipDisplay.textContent = 'null';
        if (statusDisplay) statusDisplay.textContent = 'null';
        if (ptzBtn) ptzBtn.disabled = true;
    }
}

function extractIPFromDisplayName(displayName) {
    const match = displayName.match(/\(([^)]+)\)/);
    if (match && match[1]) {
        return match[1];
    }
    return '';
}

function ZoneFullMap_PTZControl(deviceId) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device || !device.connectedCamera) {
        alert('Please select a camera first');
        return;
    }
    
    const cameraIP = extractIPFromDisplayName(device.connectedCamera);
    // const cameraIP = "192.168.1.100";//for test
    const cameraName = device.connectedCamera;
    const isDante = cameraName.indexOf("DanteCH") >= 0;
    
    // 儲存當前設備資訊
    gZoneFullMap_CurrentDeviceId = deviceId;
    gZoneFullMap_CurrentCameraIP = cameraIP;

    ZoneFullMap_createCameraPTZWindows();

    const CameraPTZ = document.getElementById('cameralist_FullMapCameraPTZWindows');
    if (CameraPTZ) {
        CameraPTZ.style.display = 'block';
        
        const jsonmsg = {
            Command: "SetWebPreview",
            IPAddress: isDante ? cameraName : gZoneFullMap_CurrentCameraIP,
            IsPreview: true
        };
        
        sendMessageCameralist(jsonmsg.Command, jsonmsg);
        
        GetPTZInfo(cameraIP);


        console.log('cameraIP : ',cameraIP);
    }
}

function ZoneFullMap_closePTZWindow() {
    const cameraPTZWindow = document.getElementById('cameralist_FullMapCameraPTZWindows');
    if (cameraPTZWindow) {
        cameraPTZWindow.style.display = 'none';
    }
    
    if (gZoneFullMap_CurrentDeviceId && gZoneFullMap_CurrentCameraIP) {
        const device = ZoneFullMap_findDeviceById(gZoneFullMap_CurrentDeviceId);
        const cameraName = device ? device.connectedCamera : '';
        // const isDante = cameraName.indexOf("DanteCH") >= 0;
        
        const jsonmsg = {
            Command: "SetWebPreview",
            IPAddress: gZoneFullMap_CurrentCameraIP,
            IsPreview: false
        };
        
        if (typeof sendMessageCameralist === 'function') {
            sendMessageCameralist(jsonmsg.Command, jsonmsg);
        }
    }
    
    // 啟用左側選單
    if (typeof enabledLeftMenuOnPopWindowClose === 'function') {
        enabledLeftMenuOnPopWindowClose();
    }
    
    // 清除當前設備資訊
    gZoneFullMap_CurrentDeviceId = null;
    gZoneFullMap_CurrentCameraIP = null;

    if(gIsCalibrationMode)
    {
        gIsCalibrationMode = false;
        console.log('ZoneFullMap_closePTZWindow  gIsCalibrationMode');
        const BC200ViewWindow = document.getElementById('cameralist_FullMapBC200ViewWindows');
        if (BC200ViewWindow) 
        {
            console.log('ZoneFullMap_closePTZWindow  -> ZoneFullMap_closeBC200ViewWindow');
            ZoneFullMap_closeBC200ViewWindow();
        }
        setTimeout(function(){
            ZoneFullMap_openDeviceConfigMap();
        },30);

    }
}

function ZoneFullMap_createCameraPTZWindows() {
    if (gIsCalibrationMode) 
    {
        var ZoneMapCameraSettingWindow = document.getElementById('cameralist_FullMapCameraPTZWindows');
        ZoneMapCameraSettingWindow.innerHTML = '';
        ZoneMapCameraSettingWindow.className = 'ZoneFullMap_PTZ_Centered';
        ZoneMapCameraSettingWindow.style.height = 'auto';

        ZoneMapCameraSettingWindow.style.userSelect = 'none';
        ZoneMapCameraSettingWindow.style.webkitUserSelect = 'none';
        ZoneMapCameraSettingWindow.style.msUserSelect = 'none';
        ZoneMapCameraSettingWindow.style.mozUserSelect = 'none';

        var modalPTZ = document.createElement('div');
        modalPTZ.className = 'ZoneFullMap_PTZ_Modal';
        modalPTZ.style.backgroundColor = 'black';
        modalPTZ.style.display = 'block';
        modalPTZ.id = 'cameralist_PTZModal';
        applyNoDragStyle(modalPTZ);

        var modalDialog = document.createElement('div');
        applyNoDragStyle(modalDialog);

        var modalContent = document.createElement('div');
        applyNoDragStyle(modalContent);

        // Title Bar
        var titleRow = document.createElement('tr');
        var titleBarRowCell = document.createElement('td');
        titleBarRowCell.className = 'CameraSettingtitleBarRow';
        titleBarRowCell.style.cursor = 'move';
        titleBarRowCell.id = 'PTZ_titleBar';
        applyNoDragStyle(titleBarRowCell);

        var logoCell = document.createElement('td');
        logoCell.className = 'title-bar-cell logo-cell';
        applyNoDragStyle(logoCell);

        var logoImage = document.createElement('img');
        logoImage.id = 'logo_image';
        logoImage.className = 'mic-popup-Window-logo-Image';
        logoImage.src = '../imagesaibox/CamConnect.png';
        logoImage.draggable = false;
        applyNoDragStyle(logoImage);
        logoCell.appendChild(logoImage);

        var titleCell = document.createElement('td');
        titleCell.className = 'title-bar-cell title-cell';
        titleCell.style.margin = '10px';
        applyNoDragStyle(titleCell);

        var titleLabel = document.createElement('label');
        titleLabel.className = 'Font_Arial_18_bold mic-xy-title';
        titleLabel.textContent = 'PTZ Camera';
        applyNoDragStyle(titleLabel);
        titleCell.appendChild(titleLabel);

        var closeButtonCell = document.createElement('td');
        closeButtonCell.className = 'title-bar-cell close-button-cell';
        applyNoDragStyle(closeButtonCell);

        var closeButton = document.createElement('button');
        closeButton.id = 'ZoneFullMap_close_CameraSetting_Window';
        closeButton.className = 'close-popup-btn';
        closeButton.setAttribute("onclick", "ZoneFullMap_closePTZWindow(this)");
        applyNoDragStyle(closeButton);
        closeButtonCell.appendChild(closeButton);

        titleBarRowCell.appendChild(logoCell);
        titleBarRowCell.appendChild(titleCell);
        titleBarRowCell.appendChild(closeButtonCell);
        titleRow.appendChild(titleBarRowCell);

        // Video Display
        var modalPTZBody = document.createElement('div');
        modalPTZBody.className = 'modalPTZ-body';
        applyNoDragStyle(modalPTZBody);

        var imgContents = document.createElement('div');
        imgContents.id = 'cameralist_imgcontents';
        imgContents.style.cssText = 'position: relative; width: 100%; height: 100%; display: flex; justify-content: center; align-items: center;';
        applyNoDragStyle(imgContents);

        var imgLoader = document.createElement('img');
        imgLoader.id = 'ZoneFullMap_loader';
        imgLoader.className = 'CameraSettingimgloader';
        imgLoader.draggable = false;
        applyNoDragStyle(imgLoader);

        imgContents.appendChild(imgLoader);
        modalPTZBody.appendChild(imgContents);
        modalContent.appendChild(titleRow);
        modalContent.appendChild(modalPTZBody);

        // 第一行：Camera Selection + Zoom Ratio
        var cameraSelectContainer = document.createElement('div');
        cameraSelectContainer.style.cssText = 'background: #000000; display: flex; gap: 15px; align-items: flex-start;';
        applyNoDragStyle(cameraSelectContainer);

        // Camera Selection
        var deviceContainer = document.createElement('div');
        deviceContainer.style.cssText = 'display: flex; flex-direction: column; gap: 10px; min-width: 250px; margin-top : 20px; margin-left : 50px;';
        applyNoDragStyle(deviceContainer);

        var selectCameraLabelContainer = document.createElement('div');
        selectCameraLabelContainer.style.cssText = 'display: flex; align-items: center; gap: 10px;';
        applyNoDragStyle(selectCameraLabelContainer);

        var label_DeviceType = document.createElement('label');
        label_DeviceType.id = 'Label_SelectCamera_DeviceType';
        label_DeviceType.className = 'Font_Arial_16_bold';
        label_DeviceType.style.color = '#89cfd8';
        label_DeviceType.innerText = window.LanguageManager.getTranslatedText("Select_Camera");
        applyNoDragStyle(label_DeviceType);

        var selectCameraImgContainer = document.createElement('div');
        selectCameraImgContainer.className = 'BC200btnstreamabo';
        selectCameraImgContainer.style.position = 'relative';
        applyNoDragStyle(selectCameraImgContainer);

        var selectCameraImg = document.createElement('img');
        selectCameraImg.id = 'BC200SelectCameraImg';
        selectCameraImg.src = './imagesaibox/btn_stream_about_normal.png';
        selectCameraImg.alt = 'Camera Image';
        selectCameraImg.style.cssText = 'width: 20px; height: 20px; display: block;';
        selectCameraImg.draggable = false;
        applyNoDragStyle(selectCameraImg);

        var BC200SelectCameraImgtooltip = document.createElement('div');
        BC200SelectCameraImgtooltip.className = 'BC200SelectCameraImgtooltip Font_Arial_14';
        BC200SelectCameraImgtooltip.innerText = window.LanguageManager.getTranslatedText("Camera_and_BC200_Calibration");
        BC200SelectCameraImgtooltip.style.cssText = 'padding: 8px; margin-left: 40px; margin-top: -15px;';
        applyNoDragStyle(BC200SelectCameraImgtooltip);


        selectCameraLabelContainer.appendChild(label_DeviceType);
        selectCameraLabelContainer.appendChild(selectCameraImgContainer);
        selectCameraImgContainer.appendChild(selectCameraImg);
        selectCameraImgContainer.appendChild(BC200SelectCameraImgtooltip);

        var select_deviceTypeSel = document.createElement('select');
        select_deviceTypeSel.id = 'ZoneFullMap_PTZ_CameraListSel';
        select_deviceTypeSel.className = 'ZoneFullMap_BC200CameraListSel';
        select_deviceTypeSel.setAttribute('onchange', 'selectCalibrationCameraOnChange()');
        applyNoDragStyle(select_deviceTypeSel);

        deviceContainer.appendChild(selectCameraLabelContainer);
        deviceContainer.appendChild(select_deviceTypeSel);

        // Zoom Ratio Selection
        var zoomRatioContainer = document.createElement('div');
        zoomRatioContainer.style.cssText = 'display: flex; flex-direction: column; gap: 10px; min-width: 150px; margin-top : 20px; margin-left : -60px;';
        applyNoDragStyle(zoomRatioContainer);

        var label_CameraZoomRatio = document.createElement('label');
        label_CameraZoomRatio.id = 'Label_SelectCamera_ZoomRatio';
        label_CameraZoomRatio.className = 'Font_Arial_16_bold';
        label_CameraZoomRatio.style.color = '#89cfd8';
        label_CameraZoomRatio.innerText = 'Zoom Ratio';
        applyNoDragStyle(label_CameraZoomRatio);

        var select_ZoomRatioSel = document.createElement('select');
        select_ZoomRatioSel.id = 'ZoneFullMap_PTZ_ZoomRationListSel';
        select_ZoomRatioSel.className = 'ZoneFullMap_BC200CameraZoomRatioSel';
        applyNoDragStyle(select_ZoomRatioSel);

        ['12x', '20x', '30x'].forEach((text, index) => {
            var option = document.createElement('option');
            option.text = text;
            option.value = [12, 20, 30][index];
            select_ZoomRatioSel.appendChild(option);
        });
        select_ZoomRatioSel.value = 20;

        zoomRatioContainer.appendChild(label_CameraZoomRatio);
        zoomRatioContainer.appendChild(select_ZoomRatioSel);

        cameraSelectContainer.appendChild(deviceContainer);
        cameraSelectContainer.appendChild(zoomRatioContainer);


        var actionButtonsContainer = document.createElement('div');
        actionButtonsContainer.style.cssText = 'display: flex; flex-direction: column; gap: 10px; min-width: 150px; margin-top : 20px; margin-left : -30px; ';
        applyNoDragStyle(actionButtonsContainer);

        var ResetButton = document.createElement('button');
        ResetButton.id = 'gZoneFullMap_PTZ_CalibrationResetButton';
        ResetButton.type = 'button';
        ResetButton.className = 'BC200PanTiltBtn_style';
        ResetButton.style.width = "150px";
        //ResetButton.style.cssText = ' background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%); border: 2px solid #89cfd8; border-radius: 8px; color: #89cfd8; font-size: 14px; font-weight: bold; cursor: pointer;';
        ResetButton.innerText = window.LanguageManager.getTranslatedText("Pan_Tilt_Reset");
        ResetButton.setAttribute('onclick', 'onPanTiltResetClick()');
        applyNoDragStyle(ResetButton);

        var Pan90UpButton = document.createElement('button');
        Pan90UpButton.id = 'gZoneFullMap_PTZ_Pan90UpButton';
        Pan90UpButton.type = 'button';
        Pan90UpButton.className = 'BC200PanTiltBtn_style';
        Pan90UpButton.style.width = "120px";
        //Pan90UpButton.style.cssText = ' background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%); border: 2px solid #89cfd8; border-radius: 8px; color: #89cfd8; font-size: 14px; font-weight: bold; cursor: pointer;';
        Pan90UpButton.innerText = 'Pan 90° Up';
        Pan90UpButton.setAttribute('onclick', 'onPan90UpClick()');
        applyNoDragStyle(Pan90UpButton);

        actionButtonsContainer.appendChild(ResetButton);
        actionButtonsContainer.appendChild(Pan90UpButton);

        cameraSelectContainer.appendChild(actionButtonsContainer);

        modalContent.appendChild(cameraSelectContainer);

        var labelDataContainer = document.createElement('div');
        labelDataContainer.style.cssText = 'margin-top : 20px; margin-left : 50px;';
        applyNoDragStyle(labelDataContainer);

        var Label_Calbriatoin1 = document.createElement('label');
        Label_Calbriatoin1.id = 'Label_BC200_Camera_Title';
        Label_Calbriatoin1.className = 'Font_Arial_16_bold';
        Label_Calbriatoin1.textContent = window.LanguageManager.getTranslatedText("Coordinate_Calbriatoin_Center_Zoom");
        applyNoDragStyle(Label_Calbriatoin1);

        var Label_Calbriatoin2 = document.createElement('label');
        Label_Calbriatoin2.className = 'Font_Arial_16_bold';
        Label_Calbriatoin2.style.cssText = 'margin-left : 15px;';
        Label_Calbriatoin2.textContent = "(";
        applyNoDragStyle(Label_Calbriatoin2);

        var Label_Calbriatoin_Pan = document.createElement('label');
        Label_Calbriatoin_Pan.className = 'Font_Arial_16_bold';
        Label_Calbriatoin_Pan.id = 'Label_BC200_Camera_Pan';
        Label_Calbriatoin_Pan.style.cssText = 'margin-left : 15px;';
        Label_Calbriatoin_Pan.textContent = window.LanguageManager.getTranslatedText("null");
        applyNoDragStyle(Label_Calbriatoin_Pan);

        var Label_Calbriatoin3 = document.createElement('label');
        Label_Calbriatoin3.className = 'Font_Arial_16_bold';
        Label_Calbriatoin3.style.cssText = 'margin-left : 15px;';
        Label_Calbriatoin3.textContent = "/";
        applyNoDragStyle(Label_Calbriatoin3);

        var Label_Calbriatoin_Tilt = document.createElement('label');
        Label_Calbriatoin_Tilt.className = 'Font_Arial_16_bold';
        Label_Calbriatoin_Tilt.id = 'Label_BC200_Camera_Tilt';
        Label_Calbriatoin_Tilt.style.cssText = 'margin-left : 15px;';
        Label_Calbriatoin_Tilt.textContent = window.LanguageManager.getTranslatedText("null");
        applyNoDragStyle(Label_Calbriatoin_Tilt);

        var Label_Calbriatoin4 = document.createElement('label');
        Label_Calbriatoin4.className = 'Font_Arial_16_bold';
        Label_Calbriatoin4.textContent = "/";
        Label_Calbriatoin4.style.cssText = 'margin-left : 15px;';
        applyNoDragStyle(Label_Calbriatoin4);

        var Label_Calbriatoin_Zoom = document.createElement('label');
        Label_Calbriatoin_Zoom.className = 'Font_Arial_16_bold';
        Label_Calbriatoin_Zoom.id = 'Label_BC200_Camera_Zoom';
        Label_Calbriatoin_Zoom.textContent = window.LanguageManager.getTranslatedText("null");
        Label_Calbriatoin_Zoom.style.cssText = 'margin-left : 15px;';
        applyNoDragStyle(Label_Calbriatoin_Zoom);

        var Label_Calbriatoin5 = document.createElement('label');
        Label_Calbriatoin5.className = 'Font_Arial_16_bold';
        Label_Calbriatoin5.style.cssText = 'margin-left : 15px;';
        Label_Calbriatoin5.textContent = ") ";
        applyNoDragStyle(Label_Calbriatoin5);

        labelDataContainer.appendChild(Label_Calbriatoin1);
        labelDataContainer.appendChild(Label_Calbriatoin2);
        labelDataContainer.appendChild(Label_Calbriatoin_Pan);
        labelDataContainer.appendChild(Label_Calbriatoin3);
        labelDataContainer.appendChild(Label_Calbriatoin_Tilt);
        labelDataContainer.appendChild(Label_Calbriatoin4);
        labelDataContainer.appendChild(Label_Calbriatoin_Zoom);
        labelDataContainer.appendChild(Label_Calbriatoin5);

        modalContent.appendChild(labelDataContainer);

        var panTiltControlContainer = document.createElement('div');
        panTiltControlContainer.style.cssText = 'padding-top: 20px; padding-bottom: 20px; padding-left: 20px; padding-right: 20px; background: #000000; display: flex; gap: 40px; align-items: center; justify-content: center;';
        applyNoDragStyle(panTiltControlContainer);

        var divPanTiltZone = document.createElement('div');
        divPanTiltZone.id = 'Div_PanTiltZone';
        divPanTiltZone.style.cssText = 'padding-left: 180px;';
        applyNoDragStyle(divPanTiltZone);

        var panTiltBtnGrid = document.createElement('ul');
        panTiltBtnGrid.className = 'ZoneFullMap_PanTiltBtnGrid';
        applyNoDragStyle(panTiltBtnGrid);

        var buttons = [
            { id: 'zoneFullMap_ptUpLeftButton',     className: 'PT_UpLeft_Btn',      },
            { id: 'zoneFullMap_ptUpButton',         className: 'PT_Up_Btn PT_Up_img',},
            { id: 'zoneFullMap_ptUpRightButton',    className: 'PT_UpRight_Btn',     },
            { id: 'zoneFullMap_ptLeftButton',       className: 'PT_Left_Btn',        },
            { id: 'zoneFullMap_HomeButton',         className: 'PT_Home_Btn',        },
            { id: 'zoneFullMap_ptRightButton',      className: 'PT_Right_Btn',       },
            { id: 'zoneFullMap_ptDownLeftButton',   className: 'PT_DownLeft_Btn',    },
            { id: 'zoneFullMap_ptDownButton',       className: 'PT_Down_Btn',        },
            { id: 'zoneFullMap_ptDownRightButton',  className: 'PT_DownRight_Btn',   },
        ];

        buttons.forEach(function(button) {
            var li = document.createElement('li');
            applyNoDragStyle(li);
            
            var btn = document.createElement('button');
            btn.id = button.id;
            btn.className = button.className;
            applyNoDragStyle(btn);
            
            btn.addEventListener('mousedown', function() {
                var dir = 'LeftDown';
                if(btn.id == 'zoneFullMap_ptUpLeftButton') dir = 'LeftUp';
                else if(btn.id == 'zoneFullMap_ptUpButton') dir = 'Up';
                else if(btn.id == 'zoneFullMap_ptUpRightButton') dir = 'RightUp';
                else if(btn.id == 'zoneFullMap_ptLeftButton') dir = 'Left';
                else if(btn.id == 'zoneFullMap_HomeButton') dir = 'home';
                else if(btn.id == 'zoneFullMap_ptRightButton') dir = 'Right';
                else if(btn.id == 'zoneFullMap_ptDownLeftButton') dir = 'LeftDown';
                else if(btn.id == 'zoneFullMap_ptDownButton') dir = 'Down';
                else if(btn.id == 'zoneFullMap_ptDownRightButton') dir = 'RightDown';
                
                ptCmdRun = true;
                setTimeout(function() {
                    zoneFullMap_sendPanTiltCmd(dir);
                }, 100);
            });

            if(btn.id != 'zoneFullMap_HomeButton') {
                btn.addEventListener('mouseup', function() {
                    zoneFullMap_stopPanTiltMoving();
                });
                btn.addEventListener('mouseleave', function() {
                    zoneFullMap_stopPanTiltMoving();
                });
            }

            li.appendChild(btn);
            panTiltBtnGrid.appendChild(li);
        });

        divPanTiltZone.appendChild(panTiltBtnGrid);
        panTiltControlContainer.appendChild(divPanTiltZone);

        var zoomButtonsContainer = document.createElement('div');
        zoomButtonsContainer.style.cssText = 'display: flex; flex-direction: column; gap: 10px;';
        applyNoDragStyle(zoomButtonsContainer);

        var zoomInBtn = document.createElement('button');
        zoomInBtn.id = 'ZoneFullMap_PTZ_zoomTeleButton';
        zoomInBtn.className = 'ZoomPlus_Btn';
        zoomInBtn.style.padding = '5px 10px';
        applyNoDragStyle(zoomInBtn);
        zoomInBtn.addEventListener('mousedown', function() {
            gCameraCalibrationZoomCmdRun = true;
            setTimeout(() => sendBC200CameraZoomCmd('In'), 100);
        });
        zoomInBtn.addEventListener('mouseup', () => sendBC200CameraZoomStopCmd());
        zoomInBtn.addEventListener('mouseleave', () => sendBC200CameraZoomStopCmd());

        var zoomOutBtn = document.createElement('button');
        zoomOutBtn.id = 'ZoneFullMap_PTZ_zoomWideButton';
        zoomOutBtn.className = 'ZoomMinus_Btn';
        zoomOutBtn.style.padding = '5px 10px';
        applyNoDragStyle(zoomOutBtn);
        zoomOutBtn.addEventListener('mousedown', function() {
            gCameraCalibrationZoomCmdRun = true;
            setTimeout(() => sendBC200CameraZoomCmd('Out'), 100);
        });
        zoomOutBtn.addEventListener('mouseup', () => sendBC200CameraZoomStopCmd());
        zoomOutBtn.addEventListener('mouseleave', () => sendBC200CameraZoomStopCmd());

        zoomButtonsContainer.appendChild(zoomInBtn);
        zoomButtonsContainer.appendChild(zoomOutBtn);


        var saveButtonsContainer = document.createElement('div');
        saveButtonsContainer.style.cssText = 'padding: 20px;';
        applyNoDragStyle(saveButtonsContainer);

        var SaveButton = document.createElement('button');
        SaveButton.id = 'gZoneFullMap_PTZ_SaveButton';
        SaveButton.type = 'button';
        SaveButton.className = 'BC200PanTiltBtn_style';
        SaveButton.innerText = 'Save';
        SaveButton.setAttribute('onclick', 'onSaveButtonClick()');
        applyNoDragStyle(Pan90UpButton);

        saveButtonsContainer.appendChild(SaveButton);

        panTiltControlContainer.appendChild(zoomButtonsContainer);
        panTiltControlContainer.appendChild(saveButtonsContainer);
        modalContent.appendChild(panTiltControlContainer);

        modalDialog.appendChild(modalContent);
        modalPTZ.appendChild(modalDialog);
        ZoneMapCameraSettingWindow.appendChild(modalPTZ);
        makeDraggable(ZoneMapCameraSettingWindow, titleBarRowCell);

        imgContents.addEventListener('mousemove', function(e) {
            const rect = imgContents.getBoundingClientRect();
            const x = Math.round(e.clientX - rect.left);
            const y = Math.round(e.clientY - rect.top);
            window.gPTZ_CurrentX = x;
            window.gPTZ_CurrentY = y;
        });

        function addPTZCrosshair() {
            const imgContents = document.getElementById('cameralist_imgcontents');
            if (!imgContents) return;
            
            const oldH = imgContents.querySelector('.ptz-crosshair-h');
            const oldV = imgContents.querySelector('.ptz-crosshair-v');
            if (oldH) oldH.remove();
            if (oldV) oldV.remove();
            
            var crosshairHorizontal = document.createElement('div');
            crosshairHorizontal.className = 'ptz-crosshair-h';
            crosshairHorizontal.style.cssText = 'position: absolute; width: 100px; height: 2px; background: limegreen; top: 50%; left: 50%; transform: translate(-50%, -50%); z-index: 10; pointer-events: none;';
            applyNoDragStyle(crosshairHorizontal);
            
            var crosshairVertical = document.createElement('div');
            crosshairVertical.className = 'ptz-crosshair-v';
            crosshairVertical.style.cssText = 'position: absolute; width: 2px; height: 100px; background: limegreen; top: 50%; left: 50%; transform: translate(-50%, -50%); z-index: 10; pointer-events: none;';
            applyNoDragStyle(crosshairVertical);
            
            imgContents.appendChild(crosshairHorizontal);
            imgContents.appendChild(crosshairVertical);
        }

        if (imgLoader) {
            imgLoader.addEventListener('load', addPTZCrosshair);
            if (imgLoader.complete && imgLoader.naturalWidth > 0) {
                addPTZCrosshair();
            }
        }
        window.addPTZCrosshair = addPTZCrosshair;

    }
    else
    {
        var ZoneMapCameraSettingWindow = document.getElementById('cameralist_FullMapCameraPTZWindows');
        ZoneMapCameraSettingWindow.innerHTML = '';
        ZoneMapCameraSettingWindow.className = 'ZoneFullMap_PTZ_Centered';
        
        ZoneMapCameraSettingWindow.style.userSelect = 'none';
        ZoneMapCameraSettingWindow.style.webkitUserSelect = 'none';
        ZoneMapCameraSettingWindow.style.msUserSelect = 'none';
        ZoneMapCameraSettingWindow.style.mozUserSelect = 'none';

        console.log('ZoneFullMap_createCameraPTZWindows test!!!!');

        var modalPTZ = document.createElement('div');
        modalPTZ.className = 'ZoneFullMap_PTZ_Modal';
        modalPTZ.style.backgroundColor = 'black';
        modalPTZ.style.display = 'block';
        modalPTZ.id = 'cameralist_PTZModal';
        applyNoDragStyle(modalPTZ);

        var modalDialog = document.createElement('div');
        applyNoDragStyle(modalDialog);

        var modalContent = document.createElement('div');
        applyNoDragStyle(modalContent);

        var titleRow = document.createElement('tr');
        var titleBarRowCell = document.createElement('td');
        titleBarRowCell.className = 'CameraSettingtitleBarRow';
        titleBarRowCell.style.cursor = 'move';
        titleBarRowCell.id = 'PTZ_titleBar';
        applyNoDragStyle(titleBarRowCell);

        var logoCell = document.createElement('td');
        logoCell.className = 'title-bar-cell logo-cell';
        applyNoDragStyle(logoCell);

        var logoImage = document.createElement('img');
        logoImage.id = 'logo_image';
        logoImage.className = 'mic-popup-Window-logo-Image';
        logoImage.src = '../imagesaibox/CamConnect.png';
        logoImage.draggable = false;
        applyNoDragStyle(logoImage);
        logoCell.appendChild(logoImage);

        var titleCell = document.createElement('td');
        titleCell.className = 'title-bar-cell title-cell';
        titleCell.style.margin = '10px';
        applyNoDragStyle(titleCell);

        var titleLabel = document.createElement('label');
        titleLabel.className = 'Font_Arial_18_bold mic-xy-title';
        titleLabel.textContent = 'PTZ Camera';
        applyNoDragStyle(titleLabel);
        titleCell.appendChild(titleLabel);

        var closeButtonCell = document.createElement('td');
        closeButtonCell.className = 'title-bar-cell close-button-cell';
        applyNoDragStyle(closeButtonCell);

        var closeButton = document.createElement('button');
        closeButton.id = 'ZoneFullMap_close_CameraSetting_Window';
        closeButton.className = 'close-popup-btn';
        closeButton.setAttribute("onclick", "ZoneFullMap_closePTZWindow(this)");
        applyNoDragStyle(closeButton);
        closeButtonCell.appendChild(closeButton);

        titleBarRowCell.appendChild(logoCell);
        titleBarRowCell.appendChild(titleCell);
        titleBarRowCell.appendChild(closeButtonCell);

        titleRow.appendChild(titleBarRowCell);

        var modalPTZBody = document.createElement('div');
        modalPTZBody.className = 'modalPTZ-body';
        applyNoDragStyle(modalPTZBody);

        var imgContents = document.createElement('div');
        imgContents.id = 'cameralist_imgcontents';
        imgContents.style.cssText = 'position: relative; width: 100%; height: 100%; display: flex; justify-content: center; align-items: center;';
        applyNoDragStyle(imgContents);

        var imgLoader = document.createElement('img');
        imgLoader.id = 'ZoneFullMap_loader';
        imgLoader.className = 'CameraSettingimgloader';
        imgLoader.draggable = false;
        applyNoDragStyle(imgLoader);

        imgContents.appendChild(imgLoader);
        modalPTZBody.appendChild(imgContents);

        modalContent.appendChild(titleRow);
        modalContent.appendChild(modalPTZBody);

        // ============ 保留所有原有的 PTZ 控制功能 ============
        var directionMirrorDiv = document.createElement('div');
        directionMirrorDiv.className = 'CameraSettingDirectionMirrorDiv';
        applyNoDragStyle(directionMirrorDiv);

        var labelMirrorFlip = document.createElement('label');
        labelMirrorFlip.className = 'Font_Arial_14_bold CameraSetting_MirrorFlip_label';
        labelMirrorFlip.textContent = window.LanguageManager.getTranslatedText("Mirror_Flip");
        applyNoDragStyle(labelMirrorFlip);

        var selectMirrorFlip = document.createElement('select');
        selectMirrorFlip.id = 'zoneFullMap_PTZMirrorFlipType';
        selectMirrorFlip.className = 'selectH24 selectW100';
        selectMirrorFlip.setAttribute('onchange', 'zoneFullMap_PTZMirrorFlipTypeChange(this)'); 
        applyNoDragStyle(selectMirrorFlip);

        [window.LanguageManager.getTranslatedText("OFF"), 
         window.LanguageManager.getTranslatedText("Mirror"), 
         window.LanguageManager.getTranslatedText("Flip"), 
         window.LanguageManager.getTranslatedText("Mirror_add_Flip")].forEach((value, index) => {
            var option = document.createElement('option');
            option.value = index;
            option.textContent = value;
            applyNoDragStyle(option);
            selectMirrorFlip.appendChild(option);
        });

        directionMirrorDiv.appendChild(labelMirrorFlip);
        directionMirrorDiv.appendChild(selectMirrorFlip);

        var divPanTiltZone = document.createElement('div');
        divPanTiltZone.id = 'Div_PanTiltZone';
        divPanTiltZone.className = 'PanTiltZoneDiv';
        applyNoDragStyle(divPanTiltZone);

        var panTiltBtnGrid = document.createElement('ul');
        panTiltBtnGrid.className = 'ZoneFullMap_noCalibrationPanTiltBtnGrid';
        applyNoDragStyle(panTiltBtnGrid);

        var buttons = [
            { id: 'zoneFullMap_ptUpLeftButton',     className: 'PT_UpLeft_Btn',      },
            { id: 'zoneFullMap_ptUpButton',         className: 'PT_Up_Btn PT_Up_img',},
            { id: 'zoneFullMap_ptUpRightButton',    className: 'PT_UpRight_Btn',     },
            { id: 'zoneFullMap_ptLeftButton',       className: 'PT_Left_Btn',        },
            { id: 'zoneFullMap_HomeButton',         className: 'PT_Home_Btn',        },
            { id: 'zoneFullMap_ptRightButton',      className: 'PT_Right_Btn',       },
            { id: 'zoneFullMap_ptDownLeftButton',   className: 'PT_DownLeft_Btn',    },
            { id: 'zoneFullMap_ptDownButton',       className: 'PT_Down_Btn',        },
            { id: 'zoneFullMap_ptDownRightButton',  className: 'PT_DownRight_Btn',   },
        ];

        buttons.forEach(function(button) {
            var li = document.createElement('li');
            applyNoDragStyle(li);

            var btn = document.createElement('button');
            btn.id = button.id;
            btn.className = button.className;
            applyNoDragStyle(btn);

            btn.addEventListener('mousedown', function() {                                
                var dir = 'LeftDown';
                if(btn.id == 'zoneFullMap_ptUpLeftButton')
                    dir = 'LeftUp';
                else if(btn.id == 'zoneFullMap_ptUpButton')
                    dir = 'Up';
                else if(btn.id == 'zoneFullMap_ptUpRightButton')
                    dir = 'RightUp';
                else if(btn.id == 'zoneFullMap_ptLeftButton')
                    dir = 'Left';
                else if(btn.id == 'zoneFullMap_HomeButton')
                    dir = 'home';
                else if(btn.id == 'zoneFullMap_ptRightButton')
                    dir = 'Right';
                else if(btn.id == 'zoneFullMap_ptDownLeftButton')
                    dir = 'LeftDown';
                else if(btn.id == 'zoneFullMap_ptDownButton')
                    dir = 'Down';
                else if(btn.id == 'zoneFullMap_ptDownRightButton')
                    dir = 'RightDown';
                ptCmdRun = true;
                setTimeout(function () {
                    zoneFullMap_sendPanTiltCmd(dir);
                }, 100);
            });

            if(btn.id != 'zoneFullMap_HomeButton')
            {
                btn.addEventListener('mouseup', function() {                                
                    zoneFullMap_stopPanTiltMoving();
                });
                btn.addEventListener('mouseleave', function() {                                
                    zoneFullMap_stopPanTiltMoving();
                });
            }

            li.appendChild(btn);
            panTiltBtnGrid.appendChild(li);
        });

        divPanTiltZone.appendChild(panTiltBtnGrid);

        var divPresetZone = document.createElement('div');
        divPresetZone.id = 'Div_presetZone';
        divPresetZone.className = 'presetZoneDiv';
        applyNoDragStyle(divPresetZone);

        var labelPreset = document.createElement('label');
        labelPreset.className = 'CameraSetting_presetLabel Font_Arial_14_bold';
        labelPreset.textContent = window.LanguageManager.getTranslatedText("Preset");
        applyNoDragStyle(labelPreset);

        var inputPreset = document.createElement('input');
        inputPreset.type = 'text';
        inputPreset.id = 'zoneFullMap_presetInput';
        inputPreset.className = 'presetInput';
        inputPreset.oninput = function() { this.value = this.value.replace(/[^\d]/g, ''); };
        applyNoDragStyle(inputPreset);

        var ulPresetBtnGrid = document.createElement('ul');
        ulPresetBtnGrid.className = 'ZoneFullMap_noCalibrationpresetBtnGrid';
        applyNoDragStyle(ulPresetBtnGrid);

        function zoneFullMap_storeButtonfunc() {
            var presetValue = document.getElementById('zoneFullMap_presetInput').value;
            zoneFullMap_sendStorePresetCmd(presetValue);
            document.getElementById('zoneFullMap_presetInput').value = "";
        }

        function zoneFullMap_recallButtonfunc() {
            var presetValue = document.getElementById('zoneFullMap_presetInput').value;
            zoneFullMap_sendRecallPresetCmd(presetValue);
            document.getElementById('zoneFullMap_presetInput').value = "";
        }

        var presetButtons = [
            { id: 'zoneFullMap_presetNum1Button',   class: 'presetBtn',      text: '1'  ,   action: function() { zoneFullMap_resetTextAction("1"); } },
            { id: 'zoneFullMap_presetNum2Button',   class: 'presetBtn',      text: '2'  ,   action: function() { zoneFullMap_resetTextAction("2"); } },
            { id: 'zoneFullMap_presetNum3Button',   class: 'presetBtn',      text: '3'  ,   action: function() { zoneFullMap_resetTextAction("3"); } },
            { id: 'zoneFullMap_presetNum4Button',   class: 'presetBtn',      text: '4'  ,   action: function() { zoneFullMap_resetTextAction("4"); } },
            { id: 'zoneFullMap_presetNum5Button',   class: 'presetBtn',      text: '5'  ,   action: function() { zoneFullMap_resetTextAction("5"); } },
            { id: 'zoneFullMap_presetNum6Button',   class: 'presetBtn',      text: '6'  ,   action: function() { zoneFullMap_resetTextAction("6"); } },
            { id: 'zoneFullMap_presetNum7Button',   class: 'presetBtn',      text: '7'  ,   action: function() { zoneFullMap_resetTextAction("7"); } },
            { id: 'zoneFullMap_presetNum8Button',   class: 'presetBtn',      text: '8'  ,   action: function() { zoneFullMap_resetTextAction("8"); } },
            { id: 'zoneFullMap_presetNum9Button',   class: 'presetBtn',      text: '9'  ,   action: function() { zoneFullMap_resetTextAction("9"); } },
            { id: 'zoneFullMap_presetStoreButton',  class: 'presetBtn',      text: window.LanguageManager.getTranslatedText("Store"), action: function() { zoneFullMap_storeButtonfunc();} },
            { id: 'zoneFullMap_presetNum0Button',   class: 'presetBtn',      text: '0'  ,   action: function() { zoneFullMap_resetTextAction("0"); } },
            { id: 'zoneFullMap_presetRecallButton', class: 'presetClearBtn', text: ''   ,   action: function() { zoneFullMap_recallButtonfunc(); } },
        ];

        presetButtons.forEach(function(button) {
            var li = document.createElement('li');
            applyNoDragStyle(li);

            var btn = document.createElement('button');
            btn.id = button.id;
            btn.className = button.class;
            applyNoDragStyle(btn);

            if (button.text) {
                var span = document.createElement('span');
                span.className = 'presetBtnText';
                span.textContent = button.text;
                applyNoDragStyle(span);
                btn.appendChild(span);
            }
            if (button.action) {
                btn.addEventListener('click', button.action);
            }
            li.appendChild(btn);
            ulPresetBtnGrid.appendChild(li);
        });

        divPresetZone.appendChild(labelPreset);
        divPresetZone.appendChild(inputPreset);
        divPresetZone.appendChild(ulPresetBtnGrid);

        var divAFcousZone = document.createElement('div');
        divAFcousZone.id = 'cameralist_Div_AFcousZone';
        divAFcousZone.className = 'CameraSetting_AFcousZoneDiv';
        applyNoDragStyle(divAFcousZone);

        var table = document.createElement('table');
        table.style.borderCollapse = 'separate';
        table.style.borderSpacing = '15px 15px';
        applyNoDragStyle(table);

        var tr1 = document.createElement('tr');
        tr1.className = 'align-Td';
        applyNoDragStyle(tr1);

        var td1 = document.createElement('td');
        applyNoDragStyle(td1);
        var label1 = document.createElement('label');
        label1.className = 'switch_afmf';
        applyNoDragStyle(label1);
        var input = document.createElement('input');
        input.type = 'checkbox';
        input.id = 'zoneFullMap_focusModeCheckboxInput';
        input.setAttribute('onchange', 'zoneFullMap_focusModeChange(this)');
        applyNoDragStyle(input);
        var span = document.createElement('span');
        span.className = 'slider_afmf';
        applyNoDragStyle(span);
        label1.appendChild(input);
        label1.appendChild(span);
        td1.appendChild(label1);

        var td2 = document.createElement('td');
        applyNoDragStyle(td2);
        var buttonNear = document.createElement('button');
        buttonNear.id = 'cameralist_focusNearButton';
        buttonNear.className = 'FocusPlus_Btn';
        applyNoDragStyle(buttonNear);
        buttonNear.addEventListener('mousedown', function() {                                
            focusCmdRun = true;
            setTimeout(function () {
                zoneFullMap_sendManualFocusCmd('Near');
            }, 100);
        });
        buttonNear.addEventListener('mouseup', function() {                                
            zoneFullMap_stopFocusMoving();
        });
        buttonNear.addEventListener('mouseleave', function() {                                
            zoneFullMap_stopFocusMoving();
        });
        td2.appendChild(buttonNear);

        var td3 = document.createElement('td');
        applyNoDragStyle(td3);
        var buttonFar = document.createElement('button');
        buttonFar.id = 'cameralist_focusFarButton';
        buttonFar.className = 'FocusMinus_Btn';
        applyNoDragStyle(buttonFar);
        buttonFar.addEventListener('mousedown', function() {                                
            focusCmdRun = true;
            setTimeout(function () {
                zoneFullMap_sendManualFocusCmd('Far');
            }, 100);
        });
        buttonFar.addEventListener('mouseup', function() {                                
            zoneFullMap_stopFocusMoving();
        });
        buttonFar.addEventListener('mouseleave', function() {                                
            zoneFullMap_stopFocusMoving();
        });
        td3.appendChild(buttonFar);

        tr1.appendChild(td1);
        tr1.appendChild(td2);
        tr1.appendChild(td3);

        var tr2 = document.createElement('tr');
        tr2.className = 'align-Td';
        applyNoDragStyle(tr2);

        [window.LanguageManager.getTranslatedText("AF_MF"),
         window.LanguageManager.getTranslatedText("Label_FocusPlus"),
         window.LanguageManager.getTranslatedText("Label_FocusMinus")].forEach(function(labelText, index) {
            var td = document.createElement('td');
            applyNoDragStyle(td);
            var label = document.createElement('label');
            label.className = 'AFMF_label Font_Arial_14_bold';
            label.textContent = labelText;
            applyNoDragStyle(label);
            if (index === 0) {
                label.id = 'Label_AFMF';
            } else if (index === 1) {
                label.id = 'Label_FocusPlus';
            } else {
                label.id = 'Label_FocusMinus';
            }
            td.appendChild(label);
            tr2.appendChild(td);
        });

        var tr3 = document.createElement('tr');
        tr3.className = 'align-Td';
        applyNoDragStyle(tr3);

        var td4 = document.createElement('td');
        applyNoDragStyle(td4);
        var labelZoom = document.createElement('label');
        labelZoom.className = 'AFMF_label Font_Arial_14_bold';
        labelZoom.id = 'cameralist_Label_FocusMinus';
        labelZoom.textContent = window.LanguageManager.getTranslatedText("Zoom");
        applyNoDragStyle(labelZoom);
        td4.appendChild(labelZoom);

        var td5 = document.createElement('td');
        applyNoDragStyle(td5);
        var zoomTeleButton = document.createElement('button');
        zoomTeleButton.id = 'cameralist_zoomTeleButton';
        zoomTeleButton.className = 'ZoomPlus_Btn';
        applyNoDragStyle(zoomTeleButton);
        zoomTeleButton.addEventListener('mousedown', function() {                                
            zoomCmdRun = true;
            setTimeout(function () {
                zoneFullMap_sendZoomCmd('In');
            }, 100);
        });
        zoomTeleButton.addEventListener('mouseup', function() {                                
            zoneFullMap_sendZoomStopCmd();
        });
        zoomTeleButton.addEventListener('mouseleave', function() {                                
            zoneFullMap_sendZoomStopCmd();
        });
        td5.appendChild(zoomTeleButton);

        var td6 = document.createElement('td');
        applyNoDragStyle(td6);
        var zoomWideButton = document.createElement('button');
        zoomWideButton.id = 'cameralist_zoomWideButton';
        zoomWideButton.className = 'ZoomMinus_Btn';
        applyNoDragStyle(zoomWideButton);
        zoomWideButton.addEventListener('mousedown', function() {                                
            zoomCmdRun = true;
            setTimeout(function () {
                zoneFullMap_sendZoomCmd('Out');
            }, 100);
        });
        zoomWideButton.addEventListener('mouseup', function() {                                
            zoneFullMap_sendZoomStopCmd();
        });
        zoomWideButton.addEventListener('mouseleave', function() {                                
            zoneFullMap_sendZoomStopCmd();
        });
        td6.appendChild(zoomWideButton);

        tr3.appendChild(td4);
        tr3.appendChild(td5);
        tr3.appendChild(td6);

        table.appendChild(tr1);
        table.appendChild(tr2);
        table.appendChild(tr3);
        divAFcousZone.appendChild(table);

        var modalFooter = document.createElement('div');
        modalFooter.className = 'modal-footer';
        modalFooter.style.backgroundColor = 'black';
        modalFooter.style.marginTop = '8px';
        applyNoDragStyle(modalFooter);

        modalContent.appendChild(directionMirrorDiv);
        modalContent.appendChild(divPanTiltZone);
        modalContent.appendChild(divPresetZone);
        modalContent.appendChild(divAFcousZone);
        modalContent.appendChild(modalFooter);

        modalDialog.appendChild(modalContent);
        modalPTZ.appendChild(modalDialog);
        ZoneMapCameraSettingWindow.appendChild(modalPTZ);

        makeDraggable(ZoneMapCameraSettingWindow, titleBarRowCell);

    }
}

function makeDraggable(element, handle) {
    let pos1 = 0, pos2 = 0, pos3 = 0, pos4 = 0;
    
    handle.onmousedown = dragMouseDown;

    function dragMouseDown(e) {
        e = e || window.event;
        e.preventDefault();
        
        if (element.style.transform && element.style.transform.includes('translate')) {
            const rect = element.getBoundingClientRect();
            element.style.top = rect.top + 'px';
            element.style.left = rect.left + 'px';
            element.style.transform = 'none';
        }
        
        pos3 = e.clientX;
        pos4 = e.clientY;
        document.onmouseup = closeDragElement;
        document.onmousemove = elementDrag;
    }

    function elementDrag(e) {
        e = e || window.event;
        e.preventDefault();
        
        pos1 = pos3 - e.clientX;
        pos2 = pos4 - e.clientY;
        pos3 = e.clientX;
        pos4 = e.clientY;
        
        element.style.top = (element.offsetTop - pos2) + "px";
        element.style.left = (element.offsetLeft - pos1) + "px";
    }

    function closeDragElement() {
        document.onmouseup = null;
        document.onmousemove = null;
    }
}

function zoneFullMap_sendRecallPresetCmd(strSet) {
    var jsonmsg = {};
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    jsonmsg.Command   = "SetToCallPreset";
    jsonmsg.IPAddress = cameraIP;
    jsonmsg.Preset    = parseInt(strSet,10);
    sendMessageCameralist("SetToCallPreset",jsonmsg);
}

function zoneFullMap_sendPanTiltCmd(dir) {

    var ptzcmd;
    var jsonmsg = {};
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    if (dir == 'stop') {
        ptzcmd = "SetPanTiltStop";
    } else if(dir == 'home') {
        ptzcmd = "SetPanTiltHome";
    } else {
        ptzcmd = "SetPanTiltStart";
        jsonmsg.Direction = dir;
    }

    jsonmsg.Command   = ptzcmd;
    jsonmsg.IPAddress = cameraIP;
    sendMessage(ptzcmd,jsonmsg);
}

function zoneFullMap_stopPanTiltMoving() {
    if (ptCmdRun == false)
        return;
    ptCmdRun = false;
    clearTimeout($(this).data('timer'));
    dir = 'stop';
    setTimeout(function () {
        zoneFullMap_sendPanTiltCmd(dir);
    }, 300);
}


function zoneFullMap_sendStorePresetCmd(strSet) {
    var jsonmsg = {};
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    jsonmsg.Command   = "SetToSavePreset";
    jsonmsg.IPAddress = cameraIP;
    jsonmsg.Preset    = parseInt(strSet,10);
    sendMessageCameralist("SetToSavePreset",jsonmsg);
}

function zoneFullMap_resetTextAction(strSet) {
    var strVal = $("#zoneFullMap_presetInput").val();
    switch (strVal.length) {
        case 0:
            strVal = strVal.concat(strSet);
            break;
        case 1:
            if (strVal == "0") {
                strVal = strSet;
            } else {
                strVal = strVal.concat(strSet);
            }
            break;
        case 2:
            if (strSet < "6") {
                if (strVal <= "25") {
                    strVal = strVal.concat(strSet);
                }
                //else do nothing
            } else {
                if (strVal <= "24") {
                    strVal = strVal.concat(strSet);
                }
            }
            break;
        case 3:
            //do nothing
            break;
        case 4:
            strVal = "";
            break;
    }
    $("#zoneFullMap_presetInput").val(strVal);
}

function zoneFullMap_updatedPTZFocusMode(msg) {

    if(gIsControlByGui)
        return;

    console.log('Mode:', msg.Mode);
    
    if(msg.Mode == "Auto"){
        if(document.getElementById("zoneFullMap_focusModeCheckboxInput"))
            document.getElementById("zoneFullMap_focusModeCheckboxInput").checked = true;
    }
    else
    {
        if(document.getElementById("zoneFullMap_focusModeCheckboxInput"))
            document.getElementById("zoneFullMap_focusModeCheckboxInput").checked = false;
    }


}

function zoneFullMap_updatedMirrorFlipType(msg) {

    if(gIsControlByGui)
        return;

    //console.log('Type:', msg.Type);
    if(document.getElementById("zoneFullMap_PTZMirrorFlipType"))
        document.getElementById("zoneFullMap_PTZMirrorFlipType").value = msg.Type;

}

function zoneFullMap_focusModeChange(e) {
    var jsonmsg  = {};
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    var focusMode;

    if (e.checked == false) {
        focusMode = "Manual";
    }
    else {
        focusMode = "Auto";
    }

    jsonmsg.Command   = "SetFocusMode";
    jsonmsg.IPAddress = cameraIP;
    jsonmsg.Mode      = focusMode;
    sendMessageCameralist("SetFocusMode",jsonmsg);
}

function zoneFullMap_sendManualFocusCmd(dir) {
    var focuscmd;
    var jsonmsg = {};
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    if (dir == 'stop') {
        focuscmd = "SetFocusStop";
    }
    else {
        focuscmd = "SetFocusStart";
        jsonmsg.Direction = dir;
    }

    jsonmsg.Command   = focuscmd;
    jsonmsg.IPAddress = cameraIP;
    sendMessageCameralist(focuscmd,jsonmsg);
}

function zoneFullMap_stopFocusMoving() {
    if (focusCmdRun == false)
        return;
    focusCmdRun = false;
    clearTimeout($(this).data('timer'));

    setTimeout(function () {
        zoneFullMap_sendManualFocusCmd('stop');
    }, 500);
}

function zoneFullMap_sendZoomStopCmd() {
    if (zoomCmdRun == false)
        return;
    zoomCmdRun = false;
    clearTimeout($(this).data('timer'));

    setTimeout(function () {
        zoneFullMap_sendZoomCmd('Stop');
    }, 500);
}

function zoneFullMap_sendZoomCmd(telewide) {
    var zoomcmd;
    var jsonmsg = {};
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    if (telewide == 'Stop') {
        zoomcmd = "SetZoomStop";
    }
    else {
        zoomcmd = "SetZoomStart";
        jsonmsg.Direction = telewide;
    }

    jsonmsg.Command   = zoomcmd;
    jsonmsg.IPAddress = cameraIP;
    sendMessageCameralist(zoomcmd,jsonmsg);
}

function zoneFullMap_PTZMirrorFlipTypeChange() {
    var jsonmsg = {};
    var value = document.getElementById("zoneFullMap_PTZMirrorFlipType").value;
    var cameraIP = gZoneFullMap_CurrentCameraIP;

    jsonmsg.Command   = "SetMirrorFlip";
    jsonmsg.IPAddress = cameraIP;
    jsonmsg.Type      = parseInt(value,10);
    sendMessageCameralist("SetMirrorFlip",jsonmsg);
}

function generateZoneFullMapVisionHTML(device) {
    const availableVisions = getZoneFullMapAvailableVisions();

    let visionSelectHTML = '';
    
    visionSelectHTML = `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 170px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Auxiliary Camera:</span>
            <select onchange="handleZoneFullMapVisionSelection('${device.id}', this.value);" 
                id="ZoneFullMap_connectedCamera_${device.id}" 
                style="width: 180px; height: 28px; background-color: white; color: black; padding: 2px 4px; border-radius: 3px; border: none;">`;
    
    const nullSelected = (!device.connectedCamera || device.connectedCamera === '') ? 'selected' : '';
    visionSelectHTML += `<option value="" ${nullSelected}>null</option>`;
    
    availableVisions.forEach(vision => {
        const selected = device.connectedCamera === vision.value ? 'selected' : '';
        visionSelectHTML += `<option value="${vision.value}" ${selected}>${vision.text}</option>`;
        console.log('generateZoneFullMapVisionHTML : ',device.connectedCamera,'vision.value : ',vision.value);
    });
    
    visionSelectHTML += `</select></div>`;
    
    let ipAddress = 'null';
    let connectStatus = 'null';
    
    if (device.connectedCamera && device.connectedCamera !== '') {
        const camData = getZoneFullMapVisionDataByName(device.connectedCamera);
        if (camData) {
            ipAddress = camData.ipAddress || 'null';
            connectStatus = camData.connectStatus || 'null';
        }
    }
    
    visionSelectHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 170px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">IP:</span>
            <span id="ZoneFullMap_vision_ip_display_${device.id}" style="font-size: 18px;" class="ZoneFullMap_deviceInfo">${ipAddress}</span>
        </div>
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 170px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Status:</span>
            <span id="ZoneFullMap_vision_status_display_${device.id}" style="font-size: 18px;" class="ZoneFullMap_deviceInfo">
                ${connectStatus}
            </span>
        </div>`;

    const isPTZEnabled = device.connectedCamera && connectStatus !== 'null' && connectStatus !== 'Unknown';
    
    visionSelectHTML += `
        <div style="display: flex; align-items: center; margin-bottom: 8px;">
            <span style="width: 170px; font-size: 18px; display: inline-block;" class="ZoneFullMap_deviceInfo">Info:</span>
            <button id="ZoneFullMap_BC200View_btn_${device.id}" 
                onclick="ZoneFullMap_BC200View('${device.id}')" 
                class="ZoneFullMap_Btn_ptzctrl"
                ${isPTZEnabled ? '' : 'disabled'}>
                <span>View</span>
            </button>
        </div>`;
    
    return visionSelectHTML;
}

function handleZoneFullMapVisionSelection(deviceId, cameraValue) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device) return;
    
    console.log(device,cameraValue);
    device.connectedCamera = cameraValue;
    
    const ipDisplay = document.getElementById(`ZoneFullMap_vision_ip_display_${deviceId}`);
    const statusDisplay = document.getElementById(`ZoneFullMap_vision_status_display_${deviceId}`);
    const ptzBtn = document.getElementById(`ZoneFullMap_BC200View_btn_${deviceId}`);
    
    if (cameraValue && cameraValue !== "") {
        const camData = getZoneFullMapVisionDataByName(cameraValue);
        if (camData) {
            if (ipDisplay) ipDisplay.textContent = camData.ipAddress || 'null';
            if (statusDisplay) statusDisplay.textContent = camData.connectStatus || 'null';
            
            if (ptzBtn) {
                const isEnabled = camData.connectStatus !== 'null' && camData.connectStatus !== 'Unknown';
                ptzBtn.disabled = !isEnabled;
            }
        }
    } else {
        if (ipDisplay) ipDisplay.textContent = 'null';
        if (statusDisplay) statusDisplay.textContent = 'null';
        if (ptzBtn) ptzBtn.disabled = true;
    }
}

var gZoneFullMap_VisionConnectionData = {};

function updateZoneFullMapVisionConnection(bc200CameraArray) {
    //console.log('updateZoneFullMapVisionConnection : ',bc200CameraArray);
    gZoneFullMap_VisionConnectionData = {};
    
    if (!bc200CameraArray || bc200CameraArray.length === 0) {
        //console.log('updateZoneFullMapVisionConnection - No BC200 cameras found');
        return;
    }
    
    let index = 0;
    bc200CameraArray.forEach(camera => {
        // No Disconnected camera
        // if (camera.ConnectStatus !== "Disconnected" && camera.IsConnect === "1") {
            
        // }

        gZoneFullMap_VisionConnectionData[index] = {
            visionIndex: index,
            friendlyName: camera.FriendlyName,
            ipAddress: camera.IPAddress,
            macAddress: camera.MacAddress,
            modelName: camera.ModelName,
            isConnect: camera.IsConnect === "1",
            connectStatus: camera.ConnectStatus,
            isLumensCam: camera.IsLumensCam
        };
        index++;

    });
    
    //console.log('gZoneFullMap_VisionConnectionData updated:', gZoneFullMap_VisionConnectionData);
}

function getZoneFullMapAvailableVisions() {
    const availableVisions = [];
    
    Object.values(gZoneFullMap_VisionConnectionData).forEach(visionData => {
        if (visionData.isConnect) {
            
        }

        availableVisions.push({
            value: `${visionData.friendlyName}(${visionData.ipAddress})`,
            text: `${visionData.friendlyName}(${visionData.ipAddress})`,
            data: visionData
        });

    });
    
    return availableVisions;
}

function getZoneFullMapVisionDataByName(displayName) {
    for (let key in gZoneFullMap_VisionConnectionData) {
        const visionData = gZoneFullMap_VisionConnectionData[key];
        const fullName = `${visionData.friendlyName}(${visionData.ipAddress})`;
        if (fullName === displayName) {
            return visionData;
        }
    }
    return null;
}

function updatedGetZoneFullMapBC200CameraCalibrationSetting(msg)
{
    //console.log('updatedGetZoneFullMapBC200CameraCalibrationSetting --- >',msg);
    if (msg.BC200ChangX !== undefined && msg.BC200ChangY !== undefined)
    {
        var xLabel = document.getElementById('Label_BC200_Camera_X');
        var yLabel = document.getElementById('Label_BC200_Camera_Y');
        
        if (xLabel) {
            xLabel.textContent = msg.BC200ChangX.toString();
        }
        if (yLabel) {
            yLabel.textContent = msg.BC200ChangY.toString();
        }
    }
    if( 'BC200SelectObject' in msg)
    {
        let bc200Data = msg.BC200SelectObject;
        
        if (bc200Data && 'getLoudestMicIndex' in bc200Data && 'getCallCameraIP' in bc200Data && 'getCallCameraPreset' in bc200Data) 
        {
            gBC200TriggerInfo.getLoudestMicIndex = bc200Data.getLoudestMicIndex+1;
            gBC200TriggerInfo.lastUpdateTime = new Date();
            gBC200TriggerInfo.getCallCameraIP = bc200Data.getCallCameraIP;

            if(bc200Data.getCallCameraPreset != -1)
                gBC200TriggerInfo.getCallCameraPreset = bc200Data.getCallCameraPreset;
            //console.log('getLoudestMicIndex = > ',msg);
            
            updateZoneFullMapBC200InfoDisplay();
        }


        if (bc200Data && 'BC200IP' in bc200Data && 'BC200Model' in bc200Data && 'BC200MAC' in bc200Data) 
        {
            gBC200TriggerInfo.BC200IP = bc200Data.BC200IP;
            gBC200TriggerInfo.BC200Model = bc200Data.BC200Model;
            gBC200TriggerInfo.BC200MAC = bc200Data.BC200MAC;
            gBC200TriggerInfo.getLoudestMicIndex = bc200Data.getLoudestMicIndex+1;
            gBC200TriggerInfo.lastUpdateTime = new Date();
            gBC200TriggerInfo.getCallCameraIP = bc200Data.getCallCameraIP;
            gBC200TriggerInfo.getTrigerPersonX = bc200Data.getTrigerPersonX;
            gBC200TriggerInfo.getTrigerPersonY = bc200Data.getTrigerPersonY;
            gBC200TriggerInfo.getTrigerTiltAngle = bc200Data.getTrigerTiltAngle;

            //console.log('BC200SelectObject = > ',msg);
            
            updateZoneFullMapBC200InfoDisplay();
        }

        if(bc200Data && 'justSendLoudestMicIndex' in bc200Data)
        {
            gBC200TriggerInfo.getLoudestMicIndex = bc200Data.justSendLoudestMicIndex+1;
            updateZoneFullMapBC200InfoJustMicIndexDisplay();
        }
    }
    
}

function updateZoneFullMapBC200InfoDisplay() {
    var infoPanel = document.getElementById('ZoneFullMap_trigger_info_panel');
    
    if (!infoPanel) {
        return;
    }
    
    const newModelValue = (gBC200TriggerInfo.BC200Model + " ( " + gBC200TriggerInfo.BC200IP + " )") || 'N/A';
    const newMicIndexValue = gBC200TriggerInfo.getLoudestMicIndex;
    
    let cameraName = '';

    // console.log('gConnectedCameras : ', gConnectedCameras);

    if (gBC200TriggerInfo.getCallCameraIP && gConnectedCameras.length > 0) {
        // console.log(`(Camera IP: ${gBC200TriggerInfo.getCallCameraIP})`);
        
        const matchedCamera = gConnectedCameras.find(camera => 
            camera.IPAddress === gBC200TriggerInfo.getCallCameraIP
        );
        
        if (matchedCamera)
        {
            cameraName = matchedCamera.DisplayName;
            // console.log(`找到對應相機: ${cameraName} (IP: ${gBC200TriggerInfo.getCallCameraIP})`);
            // console.log('相機詳細資訊:', {
            //     DisplayName: matchedCamera.DisplayName,
            //     FriendlyName: matchedCamera.FriendlyName,
            //     ModelName: matchedCamera.ModelName,
            //     MacAddress: matchedCamera.MacAddress,
            //     Type: matchedCamera.Type
            // });
        }
    }

    gBC200TriggerInfo.getCallCameraFullname = cameraName;
    const newCameraValue = gBC200TriggerInfo.getCallCameraFullname;
    const newCameraPresetValue = gBC200TriggerInfo.getCallCameraPreset;
    
    if (gLastBC200DisplayValues.model !== newModelValue) {
        if(document.getElementById('ZoneFullMap_model_value'))
            document.getElementById('ZoneFullMap_model_value').textContent = newModelValue;
        gLastBC200DisplayValues.model = newModelValue;
    }
    
    if (gLastBC200DisplayValues.micIndex !== newMicIndexValue) {

        gLastBC200DisplayValues.micIndex = newMicIndexValue;
    }

    if(document.getElementById('ZoneFullMap_mic_index_value'))
        document.getElementById('ZoneFullMap_mic_index_value').textContent = newMicIndexValue;
    
    if (gLastBC200DisplayValues.camera !== newCameraValue) {
        gLastBC200DisplayValues.camera = newCameraValue;
    }

    if(document.getElementById('ZoneFullMap_camera_value'))
        document.getElementById('ZoneFullMap_camera_value').textContent = newCameraValue;

    if (gLastBC200DisplayValues.camerapreset !== newCameraPresetValue) {
        gLastBC200DisplayValues.camerapreset = newCameraPresetValue;
    }

    if(document.getElementById('ZoneFullMap_camera_preset_value'))
        document.getElementById('ZoneFullMap_camera_preset_value').textContent = newCameraPresetValue;
}

function updateZoneFullMapBC200InfoJustMicIndexDisplay()
{

    var infoPanel = document.getElementById('ZoneFullMap_trigger_info_panel');
    
    if (!infoPanel) {
        return;
    }

    const newMicIndexValue = gBC200TriggerInfo.getLoudestMicIndex;

    if (gLastBC200DisplayValues.micIndex !== newMicIndexValue) {
        document.getElementById('ZoneFullMap_mic_index_value').textContent = newMicIndexValue;
        gLastBC200DisplayValues.micIndex = newMicIndexValue;
    }

}

function ZoneFullMapVoiceTrackingChange(e)
{
    let SettingModeStatus;

    if(e.checked == true){
        document.getElementById("ZoneFullMapVoiceTrackingCheckbox").checked = true;
        gVoiceTrackingStatus = true;
    }
    else{
        document.getElementById("ZoneFullMapVoiceTrackingCheckbox").checked = false;
        gVoiceTrackingStatus = false;
    }

    SettingModeStatus = document.getElementById("ZoneFullMapVoiceTrackingCheckbox").checked;

    sendMessageSettings("SetVoiceTracking", SettingModeStatus);
}

function updatedZoneFullMapNimbleSystemEyeSetup(msg) {
    const nimbleEyeEnable = document.getElementById("NimbleEyeCheckCheckboxInput");
    const nimbleVisionZoneDetectionEnable = document.getElementById("VisionZoneDetectionCheckCheckboxInput");
    const config = msg.Config;
    const CameraObject = msg.CameraObject;
    const AssignInitMicArray = msg.AssignInitMicArray;
    const AssignDataStatus = msg.AssignDataStatus;
    const MicSettingArray = msg.SoundSettingArray;
    const Location = msg.Location;
    const ImageData = msg.ImageData;

    if(CameraObject)
    {
        //console.log('updatedZoneFullMapNimbleSystemEyeSetup : ',msg);
    }

    if(ImageData)
    {
        //console.log('updatedZoneFullMapNimbleSystemEyeSetup -->',ImageData);
        var imgLoader = document.getElementById('BC200View_loader');
        if (imgLoader) {
            try {
                const img = new Image();
                img.src = 'data:image/jpeg;base64,' + ImageData;
                img.onload = function() {
                    imgLoader.src = img.src;
                    if (!gGetBC200Stream) {
                        gGetBC200Stream = true;
                    }

                    if (gIsCalibrationMode && typeof window.addBC200Crosshair === 'function') {
                        window.addBC200Crosshair();
                    }
                };
                img.onerror = function() {
                    console.log("Failed to load BC-200 image: invalid data.");
                };
            } catch (e) {
                console.log("Error processing BC-200 image data:", e);
            }
        }
        // if(!gGetBC200Stream)
        //     gGetBC200Stream = true;

        // if(gGetCameraStream && gGetBC200Stream)
        // {
        //     gUnBlockWaitMsg = true;
        //     newUnblockUIforPage();
        // }
    }

    if (msg.CameraObject && msg.CameraObject.BC200CameraArray) {
        updateZoneFullMapVisionConnection(msg.CameraObject.BC200CameraArray);
    }

    if (Location && Location.NimbleEyeData) 
    {

        if (ZoneFullMapRenderController.currentMode === 'paused') {
            return;
        }
        //console.log('ZoneFullMap Location -->', Location);
        
        const nimbleEyeData = Location.NimbleEyeData;
        const BC200IP = nimbleEyeData.BC200IP;
        const soundArray = nimbleEyeData.SoundArray;
        
        if (soundArray && soundArray.length > 0) 
        {
            const bc200Device = ZoneFullMap_devices.find(d => 
                d.type === 'vision' && 
                d.connectedCamera && 
                getZoneFullMapVisionDataByName(d.connectedCamera)?.ipAddress === BC200IP
            );

            if (!bc200Device) 
            {
                const beforeCount = ZoneFullMap_visionPoints.length;
                ZoneFullMap_visionPoints = ZoneFullMap_visionPoints.filter(p => p.bc200IP !== BC200IP);
                const afterCount = ZoneFullMap_visionPoints.length;
                
                if (beforeCount !== afterCount) 
                {
                    ZoneFullMap_drawVisionDataPoint_DoubleBuffered();
                }
                
                return;
            }
            
            if (bc200Device) {
                ZoneFullMap_lastVisionDataReceivedTime = Date.now();
                
                ZoneFullMap_visionPoints.forEach(p => {
                    if (p.bc200IP === BC200IP) {
                        p.pendingRemoval = true;
                    }
                });
                
                soundArray.forEach(sound => {
                    const deviceIP = sound.DeviceIP;
                    const personXYArray = sound.PersonXYModeArray;
                    
                    if (personXYArray && personXYArray.length > 0) {
                        personXYArray.forEach(person => {
                            if (person.PersonExistFlag) {
                                const x = person.BoxPanCalibX;
                                const y = person.BoxPanCalibD;
                                const distance = person.BoxDistance;
                                
                                //console.log(`Adding ZoneFullMap vision point from device ${deviceIP} - X: ${x}, Y: ${y}, Distance: ${distance}`);
                                
                                const visionPoint = {
                                    offsetX: x,
                                    offsetY: y,
                                    distance: distance,
                                    deviceIP: deviceIP,
                                    bc200IP: BC200IP,
                                    timestamp: Date.now(),
                                    opacity: 1,
                                    pendingRemoval: false
                                };
                                
                                ZoneFullMap_visionPoints.push(visionPoint);

                            }
                        });
                    }
                });
                
                const toleranceTime = 100;
                const currentTime = Date.now();
                ZoneFullMap_visionPoints = ZoneFullMap_visionPoints.filter(p => {
                    if (p.bc200IP !== BC200IP) return true;
                    if (!p.pendingRemoval) return true;
                    return (currentTime - p.timestamp) < toleranceTime;
                });

                ZoneFullMap_drawVisionDataPoint_DoubleBuffered();
            }
        }
    }
}


//v1
// const ZoneFullMapDoubleBufferRenderer = {
//     buffers: {
//         visionPoint: null
//     },
    
//     contexts: {
//         visionPoint: null
//     },
    
//     pendingRenders: {
//         visionPoint: false
//     },
    
//     renderQueues: {
//         visionPoint: []
//     },
    
//     latestTransform: {
//         visionPoint: null
//     },
    
//     initialize() {
//         this.createBuffer('visionPoint');
//     },
    
//     createBuffer(type) {
//         const canvasId = 'ZoneFullMap_xy_canvas_visionPoint';
//         const canvas = document.getElementById(canvasId);
//         if (!canvas) {
//             return;
//         }
        
//         const buffer = document.createElement('canvas');
//         const rect = canvas.getBoundingClientRect();
        
//         buffer.width = canvas.width;
//         buffer.height = canvas.height;
//         buffer.style.width = rect.width + 'px';
//         buffer.style.height = rect.height + 'px';
        
//         const ctx = buffer.getContext('2d');
//         if (!ctx) {
//             return;
//         }
        
//         this.buffers[type] = buffer;
//         this.contexts[type] = ctx;
//     },
    
//     batchRender(type, renderFunction, transform = null) {
//         if (transform) {
//             this.latestTransform[type] = {
//                 x: transform.x,
//                 y: transform.y,
//                 scale: transform.scale
//             };
//         }
        
//         this.renderQueues[type].push(renderFunction);
        
//         if (!this.pendingRenders[type]) {
//             this.pendingRenders[type] = true;
//             requestAnimationFrame(() => this.executeRender(type));
//         }
//     },
    
//     executeRender(type) {
//         const startTime = performance.now();
        
//         const buffer = this.buffers[type];
//         const ctx = this.contexts[type];
        
//         if (!buffer || !ctx) {
//             this.pendingRenders[type] = false;
//             return;
//         }
        
//         const canvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
//         if (!canvas) {
//             this.pendingRenders[type] = false;
//             return;
//         }
        
//         const mainCtx = canvas.getContext('2d');
//         const dpr = window.devicePixelRatio || 1;
        
//         const w = buffer.width / dpr;
//         const h = buffer.height / dpr;
        
//         ctx.clearRect(0, 0, buffer.width, buffer.height);
        
//         const transform = this.latestTransform[type] || gDragSystem.canvasTransform;
        
//         while (this.renderQueues[type].length > 0) {
//             const renderFunction = this.renderQueues[type].shift();
//             renderFunction(ctx, w, h, transform);  // 傳入 transform
//         }
        
//         mainCtx.save();
//         mainCtx.setTransform(1, 0, 0, 1, 0, 0);
//         mainCtx.clearRect(0, 0, canvas.width, canvas.height);
//         mainCtx.drawImage(buffer, 0, 0);
//         mainCtx.restore();
        
//         this.pendingRenders[type] = false;
//     },

//     clear(type) {
//         const canvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
//         if (canvas) {
//             const ctx = canvas.getContext('2d');
//             ctx.save();
//             ctx.setTransform(1, 0, 0, 1, 0, 0);
//             ctx.clearRect(0, 0, canvas.width, canvas.height);
//             ctx.restore();
//         }
//     },
    
//     resize() {
//         this.createBuffer('visionPoint');
//     }
// };

const ZoneFullMapDoubleBufferRenderer = {
    buffers: {
        visionPoint: null
    },
    
    contexts: {
        visionPoint: null
    },
    
    pendingRenders: {
        visionPoint: false
    },
    
    renderQueues: {
        visionPoint: []
    },
    
    latestTransform: {
        visionPoint: null
    },
    
    debugMode: true,
    lastLoggedSize: null,
    _firstRenderLogged: false,
    
    initialize() {
        this.createBuffer('visionPoint');
    },
    
    createBuffer(type) {
        const canvasId = 'ZoneFullMap_xy_canvas_visionPoint';
        const canvas = document.getElementById(canvasId);
        if (!canvas) {
            //console.error('[Buffer] Canvas not found:', canvasId);
            return;
        }
        
        const buffer = document.createElement('canvas');
        
        buffer.width = canvas.width;
        buffer.height = canvas.height;
        
        if (this.debugMode) {
            const currentSize = `${canvas.width}x${canvas.height}`;
            if (this.lastLoggedSize !== currentSize) {
                // console.log('[Buffer Create]', {
                //     type: type,
                //     canvasSize: { width: canvas.width, height: canvas.height },
                //     bufferSize: { width: buffer.width, height: buffer.height },
                //     canvasStyleSize: { width: canvas.style.width, height: canvas.style.height }
                // });
                this.lastLoggedSize = currentSize;
            }
        }
        
        const ctx = buffer.getContext('2d');
        if (!ctx) {
            //console.error('[Buffer] Failed to get context');
            return;
        }
        
        this.buffers[type] = buffer;
        this.contexts[type] = ctx;
    },
    
    // ===== 新增：檢查並同步尺寸的輔助函數 =====
    ensureBufferSize(type) {
        const canvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
        if (!canvas) return false;
        
        const buffer = this.buffers[type];
        
        if (!buffer || buffer.width !== canvas.width || buffer.height !== canvas.height) {
            if (this.debugMode && buffer) {
                // console.log('[Buffer Resize Detected in batchRender]', {
                //     bufferSize: { width: buffer.width, height: buffer.height },
                //     canvasSize: { width: canvas.width, height: canvas.height }
                // });
            }
            this.createBuffer(type);
            return true;
        }
        return false;
    },
    
    batchRender(type, renderFunction, transform = null) {
        this.ensureBufferSize(type);
        
        if (transform) {
            this.latestTransform[type] = {
                x: transform.x,
                y: transform.y,
                scale: transform.scale
            };
        }
        
        this.renderQueues[type].push(renderFunction);
        
        if (!this.pendingRenders[type]) {
            this.pendingRenders[type] = true;
            requestAnimationFrame(() => this.executeRender(type));
        }
    },
    
    executeRender(type) {
        const canvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
        if (!canvas) {
            this.pendingRenders[type] = false;
            return;
        }
        
        this.ensureBufferSize(type);
        
        const buffer = this.buffers[type];
        const ctx = this.contexts[type];
        
        if (!buffer || !ctx) {
            this.pendingRenders[type] = false;
            return;
        }
        
        const mainCtx = canvas.getContext('2d');
        
        const w = canvas.width;
        const h = canvas.height;
        
        if (this.debugMode && !this._firstRenderLogged) {
            // console.log('[Buffer First Render]', {
            //     type: type,
            //     usingCanvasSize: { w, h },
            //     bufferSize: { width: buffer.width, height: buffer.height },
            //     canvasSize: { width: canvas.width, height: canvas.height },
            //     transform: this.latestTransform[type]
            // });
            this._firstRenderLogged = true;
        }
        
        ctx.clearRect(0, 0, buffer.width, buffer.height);
        
        const transform = this.latestTransform[type] || gDragSystem.canvasTransform;
        
        while (this.renderQueues[type].length > 0) {
            const renderFunction = this.renderQueues[type].shift();
            renderFunction(ctx, w, h, transform);
        }
        
        mainCtx.save();
        mainCtx.setTransform(1, 0, 0, 1, 0, 0);
        mainCtx.clearRect(0, 0, canvas.width, canvas.height);
        mainCtx.drawImage(buffer, 0, 0);
        mainCtx.restore();
        
        this.pendingRenders[type] = false;
    },
    
    clear(type) {
        const canvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
        if (canvas) {
            const ctx = canvas.getContext('2d');
            ctx.save();
            ctx.setTransform(1, 0, 0, 1, 0, 0);
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            ctx.restore();
        }
    },
    
    resize() {
        this._firstRenderLogged = false;
        this.createBuffer('visionPoint');
    }
};

let ZoneFullMap_visionPoints = [];
let ZoneFullMap_lastVisionDataReceivedTime = 0;

const ZoneFullMapBatchUpdater = {
    pendingUpdates: {
        visionPoint: []
    },
    
    updateTimers: {
        visionPoint: null
    },
    
    batchInterval: {
        visionPoint: 33    // 30fps
    },
    
    addUpdate: function(point) {
        //console.log('Adding update to batch:', point);
        this.pendingUpdates.visionPoint.push(point);
        
        if (!this.updateTimers.visionPoint) {
            this.updateTimers.visionPoint = setTimeout(() => {
                this.processBatch();
            }, this.batchInterval.visionPoint);
        }
    },
    
    processBatch: function() {
        const updates = this.pendingUpdates.visionPoint;
        this.pendingUpdates.visionPoint = [];
        this.updateTimers.visionPoint = null;
    
        if (updates.length === 0) return;
        
        updates.forEach(update => {
            let merged = false;
            
            for (let i = 0; i < ZoneFullMap_visionPoints.length; i++) {
                const existing = ZoneFullMap_visionPoints[i];
                if (existing.deviceIP === update.deviceIP &&
                    existing.bc200IP === update.bc200IP) {
                    
                    existing.offsetX = update.offsetX;
                    existing.offsetY = update.offsetY;
                    existing.distance = update.distance;
                    existing.timestamp = Date.now();
                    existing.opacity = 1;
                    existing.pendingRemoval = false;
                    
                    merged = true;
                    //console.log('Updated existing point with C++ smoothed values:', existing);
                    break;
                }
            }
            
            if (!merged) {
                //console.log('Adding new vision point:', update);
                ZoneFullMap_visionPoints.push(update);
            }
        });
        
        if (ZoneFullMap_visionPoints.length > 150) {
            ZoneFullMap_visionPoints.sort((a, b) => b.timestamp - a.timestamp);
            ZoneFullMap_visionPoints = ZoneFullMap_visionPoints.slice(0, 150);
        }
        
        ZoneFullMap_lastVisionDataReceivedTime = Date.now();
        ZoneFullMap_drawVisionDataPoint_DoubleBuffered();
    }
};

var gZoneFullMap_BC200ShowCenterline = true;


function ZoneFullMap_drawVisionDataPoint_DoubleBuffered(providedTransform = null) 
{

    if (ZoneFullMapRenderController.isMoving && !ZoneFullMapRenderController.isDraggingDevice) 
    {
        const visionCanvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
        if (visionCanvas) {
            const ctx = visionCanvas.getContext('2d');
            ctx.clearRect(0, 0, visionCanvas.width, visionCanvas.height);
        }
        return;
    }

    if (!ZoneFullMapDoubleBufferRenderer.buffers.visionPoint) 
    {
        ZoneFullMapDoubleBufferRenderer.initialize();
    }
    
    const currentTransform = providedTransform || gDragSystem.canvasTransform;
    
    if (!currentTransform || isNaN(currentTransform.x) || isNaN(currentTransform.y) || isNaN(currentTransform.scale)) {
        console.warn('[Vision] Invalid transform:', currentTransform);
        return;
    }
    
    ZoneFullMapDoubleBufferRenderer.batchRender('visionPoint', (ctx, w, h, transform) => {
        const { x: ox, y: oy, scale } = transform || currentTransform;
        
        if (ZoneFullMap_visionPoints.length === 0) {
            return;
        }
        
        ctx.save();
        
        const deviceGroups = groupZoneFullMapPointsByDevice(ZoneFullMap_visionPoints);
        
        Object.entries(deviceGroups).forEach(([deviceId, points]) => {
            if (points.length === 0) return;
            
            const colorScheme = ZoneFullMap_getVisionColorScheme(deviceId);
            
            points.forEach((visionPoint) => {
                const bc200Device = ZoneFullMap_devices.find(d => 
                    d.type === 'vision' && 
                    d.connectedCamera && 
                    getZoneFullMapVisionDataByName(d.connectedCamera)?.ipAddress === visionPoint.bc200IP
                );
                
                if (!bc200Device) return;
                
                if (gZoneFullMap_deviceVisibility && gZoneFullMap_deviceVisibility[bc200Device.id] === false) {
                    return;
                }
                
                const deviceDirection = bc200Device.angle || 0;
                const fovAngle = bc200Device.fovAngle || 0;
                const offsetAngle = bc200Device.offsetAngle || 0;
            
                const coordinateTransform = {
                    0:  { flipVisionX: false, flipVisionY: false, flipOffsetX: false, flipOffsetY: false, swapXY: false },
                    180:{ flipVisionX: false, flipVisionY: false, flipOffsetX: false, flipOffsetY: false, swapXY: false },
                    90: { flipVisionX: false, flipVisionY: true,  flipOffsetX: true, flipOffsetY: true, swapXY: false },
                    270:{ flipVisionX: false, flipVisionY: true,  flipOffsetX: true, flipOffsetY: true, swapXY: false }
                };
                
                const transform = coordinateTransform[deviceDirection];
                
                let visionXmm, visionYmm, offsetXmm, offsetYmm;
                
                if (transform.swapXY) 
                {
                    visionXmm = visionPoint.offsetY * 10000 * (transform.flipVisionX ? -1 : 1);
                    visionYmm = visionPoint.offsetX * 10000 * (transform.flipVisionY ? -1 : 1);
                    offsetXmm = bc200Device.offsetY * 10000 * (transform.flipOffsetX ? -1 : 1);
                    offsetYmm = bc200Device.offsetX * 10000 * (transform.flipOffsetY ? -1 : 1);
                } 
                else 
                {
                    visionXmm = visionPoint.offsetX * 10000 * (transform.flipVisionX ? -1 : 1);
                    visionYmm = visionPoint.offsetY * 10000 * (transform.flipVisionY ? -1 : 1);
                    offsetXmm = bc200Device.offsetX * 10000 * (transform.flipOffsetX ? -1 : 1);
                    offsetYmm = bc200Device.offsetY * 10000 * (transform.flipOffsetY ? -1 : 1);
                }

                const totalAngle = (deviceDirection - offsetAngle - fovAngle) % 360;
                const angleRad = (totalAngle * Math.PI) / 180;

                const rotatedvisionX = visionXmm * Math.cos(angleRad) - visionYmm * Math.sin(angleRad);
                const rotatedvisionY = visionXmm * Math.sin(angleRad) + visionYmm * Math.cos(angleRad);
                const rotatedOffsetX = offsetXmm * Math.cos(angleRad) - offsetYmm * Math.sin(angleRad);
                const rotatedOffsetY = offsetXmm * Math.sin(angleRad) + offsetYmm * Math.cos(angleRad);
                
                const virtualCenterWorldX = bc200Device.x + rotatedOffsetX;
                const virtualCenterWorldY = bc200Device.y + rotatedOffsetY;
                
                const soundWorldX = virtualCenterWorldX + rotatedvisionX;
                const soundWorldY = virtualCenterWorldY + rotatedvisionY;
                
                const cx = ox + soundWorldX * scale / 100;
                const cy = oy - soundWorldY * scale / 100;
                
                const baseLineWidth = 3;
                const baseRangeLineWidth = 4;
                
                const dynamicLineWidth = Math.max(1, baseLineWidth / scale);
                const dynamicRangeLineWidth = Math.max(1.5, baseRangeLineWidth / scale);
                
                const baseRadius = 8;
                const radiusPx = Math.max(baseRadius, baseRadius * scale);
                ctx.fillStyle = colorScheme.primary;
                ctx.strokeStyle = colorScheme.primary;
                ctx.lineWidth = dynamicLineWidth;
                
                ctx.beginPath();
                ctx.arc(cx, cy, radiusPx, 0, 2 * Math.PI);
                ctx.fill();
                ctx.stroke();
                
                const rangeSideMm = 16000;
                const rangeRadiusPx = rangeSideMm * scale / 200;
                
                ctx.save();
                ctx.strokeStyle = colorScheme.primary;
                ctx.lineWidth = dynamicRangeLineWidth;
                ctx.setLineDash([5, 5]);
                ctx.beginPath();
                ctx.arc(cx, cy, rangeRadiusPx, 0, 2 * Math.PI);
                ctx.stroke();
                ctx.restore();

                const hasOffset = (bc200Device.offsetX !== 0 || bc200Device.offsetY !== 0);
                
                if (hasOffset) 
                {
                    const virtualCenterCx = ox + virtualCenterWorldX * scale / 100;
                    const virtualCenterCy = oy - virtualCenterWorldY * scale / 100;
                    
                    ctx.save();
                    
                    const baseCenterRadius = 6;
                    const centerRadiusPx = Math.max(baseCenterRadius, baseCenterRadius * scale);
                    ctx.fillStyle = colorScheme.primary;
                    ctx.strokeStyle = '#FFFFFF';
                    ctx.lineWidth = dynamicLineWidth;
                    
                    ctx.beginPath();
                    ctx.arc(virtualCenterCx, virtualCenterCy, centerRadiusPx, 0, 2 * Math.PI);
                    ctx.fill();
                    ctx.stroke();
                    
                    const crossSize = centerRadiusPx + 4;
                    ctx.strokeStyle = colorScheme.primary;
                    ctx.lineWidth = dynamicLineWidth;
                    
                    ctx.beginPath();
                    ctx.moveTo(virtualCenterCx - crossSize, virtualCenterCy);
                    ctx.lineTo(virtualCenterCx + crossSize, virtualCenterCy);
                    ctx.moveTo(virtualCenterCx, virtualCenterCy - crossSize);
                    ctx.lineTo(virtualCenterCx, virtualCenterCy + crossSize);
                    ctx.stroke();
                    
                    ctx.restore();
                    
                    if (gZoneFullMap_BC200ShowCenterline === true) 
                    {
                        ctx.save();
                        ctx.strokeStyle = colorScheme.primary;
                        ctx.lineWidth = dynamicLineWidth;
                        ctx.setLineDash([8, 4]);
                        ctx.beginPath();
                        ctx.moveTo(cx, cy);
                        ctx.lineTo(virtualCenterCx, virtualCenterCy);
                        ctx.stroke();
                        ctx.restore();
                    }
                    
                    if (gZoneFullMap_BC200ShowCenterline === true) 
                    {
                        ctx.save();
                        const deviceCx = ox + bc200Device.x * scale / 100;
                        const deviceCy = oy - bc200Device.y * scale / 100;
                        ctx.strokeStyle = colorScheme.primary;
                        ctx.lineWidth = Math.max(0.75, (dynamicLineWidth * 0.75));
                        ctx.setLineDash([4, 8]);
                        ctx.globalAlpha = 0.5;
                        ctx.beginPath();
                        ctx.moveTo(virtualCenterCx, virtualCenterCy);
                        ctx.lineTo(deviceCx, deviceCy);
                        ctx.stroke();
                        ctx.restore();
                    }
                }
                else
                {
                    if (gZoneFullMap_BC200ShowCenterline === true) 
                    {
                        ctx.save();
                        const deviceCx = ox + bc200Device.x * scale / 100;
                        const deviceCy = oy - bc200Device.y * scale / 100;
                        ctx.strokeStyle = colorScheme.primary;
                        ctx.lineWidth = dynamicLineWidth;
                        ctx.setLineDash([8, 4]);
                        ctx.beginPath();
                        ctx.moveTo(cx, cy);
                        ctx.lineTo(deviceCx, deviceCy);
                        ctx.stroke();
                        ctx.restore();
                    }
                }
            });
        });
        
        ctx.restore();
    }, currentTransform);
}


function groupZoneFullMapPointsByOpacity(points) {
    const groups = {};
    points.forEach(point => {
        const opacity = point.opacity || 1;
        if (!groups[opacity]) {
            groups[opacity] = [];
        }
        groups[opacity].push(point);
    });
    return groups;
}

function groupZoneFullMapPointsByDevice(points) 
{
    const groups = {};
    
    points.forEach((point, index) => {
        
        const bc200Device = ZoneFullMap_devices.find(d => 
            d.type === 'vision' && 
            d.connectedCamera && 
            getZoneFullMapVisionDataByName(d.connectedCamera)?.ipAddress === point.bc200IP
        );
        
        if (bc200Device) 
        {
            const deviceId = bc200Device.id;
            if (!groups[deviceId]) 
            {
                groups[deviceId] = [];
            }
            groups[deviceId].push(point);
        }
    });
    
    return groups;
}

function hexToRgb(hex) 
{
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? {
        r: parseInt(result[1], 16),
        g: parseInt(result[2], 16),
        b: parseInt(result[3], 16)
    } : { r: 255, g: 0, b: 0 };
}

function ZoneFullMap_addVisionPointFromLocation(offsetX, offsetY, distance, deviceIP, bc200IP) 
{
    const visionPoint = {
        offsetX: offsetX,
        offsetY: offsetY,
        distance: distance,
        deviceIP: deviceIP,
        bc200IP: bc200IP,
        timestamp: Date.now(),
        opacity: 1,
        pendingRemoval: false
    };
    
    ZoneFullMapBatchUpdater.addUpdate(visionPoint);
}

function initZoneFullMapVisionPointRenderer() 
{
    const canvas = document.getElementById('ZoneFullMap_xy_canvas_visionPoint');
    if (canvas) {
        const container = canvas.parentElement;
        if (container) {
            const rect = container.getBoundingClientRect();
            const dpr = window.devicePixelRatio || 1;
            
            canvas.width = rect.width;
            canvas.height = rect.height;
            
            canvas.style.width = rect.width + 'px';
            canvas.style.height = rect.height + 'px';
            
            // console.log('Canvas initialized with size:', {
            //     actual: { w: canvas.width, h: canvas.height },
            //     display: { w: rect.width, h: rect.height },
            //     dpr: dpr
            // });
        }
    }
    
    ZoneFullMapDoubleBufferRenderer.initialize();
    window.addEventListener('resize', onZoneFullMapCanvasResize);
}

function clearZoneFullMapVisionPoints() {
    ZoneFullMap_visionPoints = [];
    if(ZoneFullMapDoubleBufferRenderer)
        ZoneFullMapDoubleBufferRenderer.clear('visionPoint');
}

function onZoneFullMapCanvasResize() {
    if(ZoneFullMapDoubleBufferRenderer)
        ZoneFullMapDoubleBufferRenderer.resize();
}

const ZoneFullMapRenderController = {
    renderModes: {
        FULL: 'full',
        GRID_ONLY: 'grid',
        PAUSED: 'paused'
    },
    
    currentMode: 'full',
    dataBuffer: [],
    
    isMoving: false,
    isDraggingDevice: false,
    rafId: null,
    lastRenderTime: 0,
    minRenderInterval: 16,
    needsRedraw: false,
    pendingGridRedraw: false,
    
    setMode: function(mode) {
        this.currentMode = mode;
    },
    
    canAcceptData: function() {
        return this.currentMode !== this.renderModes.PAUSED;
    },
    
    canRenderVisionPoints: function() {
        return this.currentMode === this.renderModes.FULL;
    },
    
    bufferData: function(data) {
        if (this.currentMode === this.renderModes.GRID_ONLY) {
            this.dataBuffer.push({
                data: data,
                timestamp: Date.now()
            });
            return true;
        }
        return false;
    },
    
    flushBuffer: function() {
        const bufferedData = this.dataBuffer;
        this.dataBuffer = [];
        
        bufferedData.forEach(item => {
            if (Date.now() - item.timestamp < 500) {
                this.processBufferedData(item.data);
            }
        });
    },
    
    processBufferedData: function(data) {
        if (typeof ZoneFullMap_drawVisionDataPoint_DoubleBuffered === 'function') {
            ZoneFullMap_drawVisionDataPoint_DoubleBuffered();
        }
    },
    
    requestGridRedraw: function() {
        this.pendingGridRedraw = true;
        if (this.isMoving) {
            this.scheduleRender();
        }
    },
    
    requestRedraw: function() {
        this.needsRedraw = true;
        this.scheduleRender();
    },
    
    scheduleRender: function() {
        if (this.rafId) {
            return;
        }
        
        this.rafId = requestAnimationFrame(() => {
            const now = performance.now();
            
            if (now - this.lastRenderTime >= this.minRenderInterval) {
                
                if (this.pendingGridRedraw && this.isMoving) {
                    if (gDragSystem && gDragSystem.gridCanvas) {
                        if (!this.isDraggingDevice) {
                            gDragSystem._performGridRedraw();
                        }
                        
                        const currentTransform = {
                            x: gDragSystem.canvasTransform.x,
                            y: gDragSystem.canvasTransform.y,
                            scale: gDragSystem.canvasTransform.scale
                        };
                        
                        if (typeof ZoneFullMap_updateDevicePositions === 'function') {
                            this._updateDevicesImmediately(currentTransform);
                        }
                        
                        if (typeof ZoneFullMap_redrawQuadrilaterals === 'function') {
                            ZoneFullMap_redrawQuadrilaterals(currentTransform);
                        }
                        
                        if (typeof gSelectedDevice !== 'undefined' && gSelectedDevice && 
                            typeof ZoneFullMap_highlightSelectedDevice === 'function') {
                            ZoneFullMap_highlightSelectedDevice(currentTransform);
                        }
                    }
                    this.pendingGridRedraw = false;
                }

                if (this.needsRedraw) {
                    this.render();
                    this.needsRedraw = false;
                }
                
                this.lastRenderTime = now;
            }
            
            this.rafId = null;
            

            if (this.isMoving) {
                this.scheduleRender();
            }
        });
    },
    
    _updateDevicesImmediately: function(transform) {
        const { x: ox, y: oy, scale } = transform;
        
        const canvases = {
            mic: document.getElementById('ZoneFullMap_xy_canvas_mic'),
            camera: document.getElementById('ZoneFullMap_xy_canvas_camera'),      
            bc200: document.getElementById('ZoneFullMap_xy_canvas_bc200'),        
            anchor: document.getElementById('ZoneFullMap_xy_canvas_anchor'),
            soundPoint: document.getElementById('ZoneFullMap_xy_canvas_soundPoint')
        };
        
        if (!canvases.mic || !canvases.camera || !canvases.bc200 || !canvases.anchor || !canvases.soundPoint) {
            return;
        }
        
        const contexts = {
            mic: canvases.mic.getContext('2d'),
            camera: canvases.camera.getContext('2d'),
            bc200: canvases.bc200.getContext('2d'),
            anchor: canvases.anchor.getContext('2d'),
            soundPoint: canvases.soundPoint.getContext('2d')
        };
        
        const w = canvases.mic.width;
        const h = canvases.mic.height;
        
        Object.values(contexts).forEach(ctx => ctx.clearRect(0, 0, w, h));
        
        const devicesByType = {
            microphone: [],
            camera: [],
            vision: [],
            anchor: []
        };
        
        if (typeof ZoneFullMap_devices !== 'undefined') {
            ZoneFullMap_devices.forEach(device => {
                if (gZoneFullMap_deviceVisibility && gZoneFullMap_deviceVisibility[device.id] === false) {
                    return;
                }
                
                if (devicesByType[device.type]) {
                    devicesByType[device.type].push(device);
                }
            });
        }
        
        devicesByType.camera.forEach(device => {
            if (typeof ZoneFullMap_DeviceVisualConfig !== 'undefined') {
                const config = ZoneFullMap_DeviceVisualConfig.camera;
                const widthPx = config.baseCircleSize * scale * 100;
                const cx = ox + device.x * scale / 100;
                const cy = oy - device.y * scale / 100;
                
                if (typeof ZoneFullMap_drawCameraArrows === 'function') {
                    ZoneFullMap_drawCameraArrows(contexts.camera, device, cx, cy, widthPx, config);
                }
                if (typeof ZoneFullMap_drawCameraCircle === 'function') {
                    ZoneFullMap_drawCameraCircle(contexts.camera, device, cx, cy, widthPx, config);
                }
            }
        });
        
        devicesByType.vision.forEach(device => {
            if (typeof ZoneFullMap_DeviceVisualConfig !== 'undefined') {
                const config = ZoneFullMap_DeviceVisualConfig.bc200;
                const widthPx = config.baseCircleSize * scale * 100;
                const cx = ox + device.x * scale / 100;
                const cy = oy - device.y * scale / 100;
                
                if (typeof ZoneFullMap_drawBC200Arrow === 'function') {
                    ZoneFullMap_drawBC200Arrow(contexts.bc200, device, cx, cy, widthPx, config);
                }
                if (typeof ZoneFullMap_drawBC200Circle === 'function') {
                    ZoneFullMap_drawBC200Circle(contexts.bc200, device, cx, cy, widthPx, config);
                }
            }
        });
        
        devicesByType.microphone.forEach(device => {
            const widthPx = device.baseWidthM * scale * 100;
            const cx = ox + device.x * scale / 100;
            const cy = oy - device.y * scale / 100;
            
            if (typeof ZoneFullMap_drawMicrophoneOnCanvas === 'function') {
                ZoneFullMap_drawMicrophoneOnCanvas(contexts.mic, device, cx, cy, widthPx);
            }
        });
        
        devicesByType.anchor.forEach(device => {
            const widthPx = device.baseWidthM * scale * 100;
            const cx = ox + device.x * scale / 100;
            const cy = oy - device.y * scale / 100;
            
            if (typeof ZoneFullMap_trackAnchorMovement === 'function') {
                ZoneFullMap_trackAnchorMovement(device, cx, cy);
            }
            if (typeof ZoneFullMap_drawAnchorBasic === 'function') {
                ZoneFullMap_drawAnchorBasic(contexts.anchor, device, cx, cy, widthPx);
            }
        });
        
        if (typeof ZoneFullMap_manageAnchorBreathingAnimation === 'function') {
            ZoneFullMap_manageAnchorBreathingAnimation();
        }
    },
    
    render: function() {
        if (typeof redrawAllCanvases === 'function') {
            redrawAllCanvases();
        }
    },
    
    startMoving: function() {
        if (this.isMoving) return;
        
        this.isMoving = true;
        this.isDraggingDevice = false;
        this.setMode(this.renderModes.PAUSED);
        ZoneFullMapDoubleBufferRenderer.clear('visionPoint');
        this.scheduleRender();
    },
    
    startDraggingDevice: function() {
        if (this.isMoving && this.isDraggingDevice) return;
        
        this.isMoving = true;
        this.isDraggingDevice = true;
        this.setMode(this.renderModes.PAUSED);
        ZoneFullMapDoubleBufferRenderer.clear('visionPoint');
        this.scheduleRender();
    },
    
    stopMoving: function() {
        if (!this.isMoving) return;
        
        this.isMoving = false;
        this.isDraggingDevice = false;
        this.setMode(this.renderModes.FULL);
        this.flushBuffer();
        
        this.needsRedraw = true;
        this.scheduleRender();
        
        setTimeout(() => {
            if (typeof ZoneFullMap_drawVisionDataPoint_DoubleBuffered === 'function') {
                ZoneFullMap_drawVisionDataPoint_DoubleBuffered();
            }
        }, 50);
    }
};

function ZoneFullMap_showDeviceSelector(hitDevices, mouseX, mouseY, isRightClick = false) {
    ZoneFullMap_closeDeviceSelector();
    
    const selector = document.createElement('div');
    selector.id = 'ZoneFullMap_device_selector';
    selector.className = 'ZoneFullMap_device_selector';
    selector.isRightClick = isRightClick;
    
    const headerHTML = `
        <div class="ZoneFullMap_device_selector_header">
            <span>Select Device (${hitDevices.length})</span>
            <span class="ZoneFullMap_device_selector_close">✕</span>
        </div>
    `;
    
    const listHTML = hitDevices.map((hit, index) => {
        const device = hit.device;
        const isMic = device.type === 'microphone';
        const deviceName = device.name || `Device ${device.id}`;
        
        return `
            <div class="ZoneFullMap_device_selector_item" data-device-id="${device.id}" data-device-index="${index}">
                <div class="ZoneFullMap_device_selector_info">
                    <div class="ZoneFullMap_device_selector_name">${deviceName}</div>
                    <div class="ZoneFullMap_device_selector_id">ID: ${device.id}</div>
                </div>
            </div>
        `;
    }).join('');
    
    selector.innerHTML = headerHTML + '<div class="ZoneFullMap_device_selector_list">' + listHTML + '</div>';
    
    document.body.appendChild(selector);
    
    const canvas = document.querySelector('.ZoneFullMap_canvas_td');
    if (!canvas) {
        ZoneFullMap_closeDeviceSelector();
        return;
    }
    
    const canvasRect = canvas.getBoundingClientRect();
    const selectorRect = selector.getBoundingClientRect();
    const selectorWidth = selectorRect.width;
    const selectorHeight = selectorRect.height;
    
    let x = mouseX + canvasRect.left + 10;
    let y = mouseY + canvasRect.top + 10;
    
    const margin = 10;
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;
    
    if (x + selectorWidth > viewportWidth - margin) {
        x = mouseX + canvasRect.left - selectorWidth - margin;
    }
    
    if (y + selectorHeight > viewportHeight - margin) {
        y = mouseY + canvasRect.top - selectorHeight - margin;
    }
    
    x = Math.max(margin, x);
    y = Math.max(margin, y);
    
    selector.style.left = x + 'px';
    selector.style.top = y + 'px';

    selector.hitDevices = hitDevices;
    gOverlappingDeviceSelector = selector;
    console.log('Stored hitDevices in selector:', selector.hitDevices);

    selector.addEventListener('click', ZoneFullMap_handleSelectorClick);

    setTimeout(() => {
        document.addEventListener('click', ZoneFullMap_handleSelectorOutsideClick);
    }, 100);
}

function ZoneFullMap_handleSelectorClick(e) {
    if (e.target.classList.contains('ZoneFullMap_device_selector_close')) {
        ZoneFullMap_closeDeviceSelector();
        return;
    }
    
    const item = e.target.closest('.ZoneFullMap_device_selector_item');
    if (item) {
        const deviceIndex = parseInt(item.dataset.deviceIndex);
        
        if (gOverlappingDeviceSelector && gOverlappingDeviceSelector.hitDevices) {
            const hitInfo = gOverlappingDeviceSelector.hitDevices[deviceIndex];
            
            if (hitInfo) {
                gSelectedDevice = hitInfo.device;
                ZoneFullMap_closeDeviceSelector();
                ZoneFullMap_highlightSelectedDevice();
                
                console.log('Selected device from selector:', hitInfo.device.name || hitInfo.device.id);
            }
        }
    }
}

function ZoneFullMap_findDeviceById(deviceId) {
    if (!ZoneFullMap_devices || !deviceId) return null;
    return ZoneFullMap_devices.find(device => device.id == deviceId);
}

function ZoneFullMap_selectDevice(device, screenX, screenY) {
    if (!device) return;
    gSelectedDevice = device;
    
    ZoneFullMap_closeDeviceSelector();
    
    ZoneFullMap_handleDeviceClick(device, screenX, screenY);
    ZoneFullMap_highlightSelectedDevice();
}

function ZoneFullMap_selectDeviceOnly(device, screenX, screenY) {
    if (!device) return;
    gSelectedDevice = device;
    ZoneFullMap_closeDeviceSelector();
    ZoneFullMap_highlightSelectedDevice();
    
}

function ZoneFullMap_closeDeviceSelector() {
    if (gOverlappingDeviceSelector) {
        gOverlappingDeviceSelector.remove();
        gOverlappingDeviceSelector = null;
    }
    
    document.removeEventListener('click', ZoneFullMap_handleSelectorOutsideClick);
}

function ZoneFullMap_handleSelectorOutsideClick(e) {
    if (gOverlappingDeviceSelector && !gOverlappingDeviceSelector.contains(e.target)) {
        ZoneFullMap_closeDeviceSelector();
    }
}

function ZoneFullMap_cleanupDeviceSelector() {
    if (gOverlappingDeviceSelector) {
        gOverlappingDeviceSelector.remove();
        gOverlappingDeviceSelector = null;
    }
    document.removeEventListener('click', ZoneFullMap_handleSelectorOutsideClick);
}

function ZoneFullMap_openDeviceConfigMap() {
    try {
        let configMapWindow = document.getElementById('Device_Config_Map_Window');
        const configBtn = document.getElementById('ZoneFullMapDeviceConfigBtn');
        
        if (configMapWindow) {
            configMapWindow.remove();
            if (configBtn) {
                configBtn.classList.remove('active');
            }
            return;
        }
        
        if (configBtn) {
            configBtn.classList.add('active');
        }
        
        const devices = {
            microphones: [],
            cameras: [],
            visions: [],
            total: 0
        };

        if (typeof ZoneFullMap_devices !== 'undefined' && ZoneFullMap_devices.length > 0) {
            ZoneFullMap_devices.forEach(device => {
                switch (device.type) {
                    case 'microphone':
                        devices.microphones.push({
                            id: device.id,
                            name: device.name,
                            x: device.x,
                            y: device.y
                        });
                        break;
                    case 'camera':
                        devices.cameras.push({
                            id: device.id,
                            name: device.name,
                            x: device.x,
                            y: device.y
                        });
                        break;
                    case 'vision':
                        devices.visions.push({
                            id: device.id,
                            name: device.name,
                            x: device.x,
                            y: device.y
                        });
                        break;
                }
            });
        }

        devices.total = devices.microphones.length + devices.cameras.length + devices.visions.length;

        if (devices.total === 0) {
            alert('Please add devices to the canvas first');
            if (configBtn) {
                configBtn.classList.remove('active');
            }
            return;
        }

        configMapWindow = document.createElement('div');
        configMapWindow.id = 'Device_Config_Map_Window';
        configMapWindow.className = 'popup-window';
        configMapWindow.style.cssText = `
            position: fixed;
            top: 10%;
            right: 2%;
            width: 90%;
            max-width: 1200px;
            max-height: 800px;
            height: 85vh;
            background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
            border: 2px solid #89cfd8;
            border-radius: 12px;
            z-index: 10000;
            display: flex;
            flex-direction: column;
            box-shadow: 0 25px 70px rgba(0, 0, 0, 0.9);
        `;

        if (!document.getElementById('dcm-styles')) {
            const styleSheet = document.createElement('style');
            styleSheet.id = 'dcm-styles';
            styleSheet.innerHTML = `
                .ZoneFullMap_device_config_map_btn.active {
                    background: linear-gradient(180deg, rgba(137, 207, 216, 1) 0%, rgba(137, 207, 216, 1) 100%) !important;
                    color: rgba(19, 19, 19, 1) !important;
                    transform: translateY(-1px) !important;
                    box-shadow: 0 4px 12px rgba(137, 207, 216, 0.2) !important;
                }
                
                .ZoneFullMap_device_config_map_btn.active .systemImg {
                    content: url("../imageslm/menu_system_hover_pressed.png") !important;
                    filter: drop-shadow(0 0 3px #89cfd8) !important;
                }
                
                .dcm-relation-btn {
                    width: 100%;
                    padding: 16px;
                    margin-bottom: 12px;
                    background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
                    border: 2px solid rgba(137, 207, 216, 0.3);
                    border-radius: 10px;
                    color: #89cfd8;
                    cursor: pointer;
                    text-align: left;
                    transition: all 0.2s ease;
                    font-family: Arial;
                }
                
                .dcm-relation-btn.active {
                    background: linear-gradient(180deg, rgba(137, 207, 216, 1) 0%, rgba(137, 207, 216, 1) 100%);
                    border-color: rgba(137, 207, 216, 1);
                    color: rgba(19, 19, 19, 1);
                    box-shadow: 0 4px 12px rgba(137, 207, 216, 0.2);
                    transform: translateX(8px);
                }
            `;
            document.head.appendChild(styleSheet);
        }

        const titleBar = document.createElement('div');
        titleBar.id = 'deviceConfigMapTitleBar';
        titleBar.style.cssText = `
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 18px 25px;
            background: rgba(0, 0, 0, 0.6);
            border-bottom: 2px solid rgba(137, 207, 216, 0.3);
            border-radius: 12px 12px 0 0;
            cursor: move;
            user-select: none;
        `;

        titleBar.innerHTML = `
            <div style="display: flex; align-items: center; gap: 15px;">
                <img src="../imagesaibox/CamConnect.png" style="width: 28px; height: 28px;">
                <span style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold; margin-top: 5px;">設備校準配置</span>
                <span style="color: rgba(137, 207, 216, 0.7); font-size: 16px;">
                    (${devices.visions.length} BC200, ${devices.microphones.length} MIC, ${devices.cameras.length} Camera)
                </span>
            </div>
            <button id="closeDeviceConfigMapBtn" onclick="ZoneFullMap_closeDeviceConfigMap()" style="
                background: rgba(255, 255, 255, 0.05);
                border: 2px solid #89cfd8;
                color: #89cfd8;
                width: 36px;
                height: 36px;
                border-radius: 50%;
                cursor: pointer;
                font-size: 20px;
                transition: all 0.3s;
                font-weight: bold;
                display: flex;
                justify-content: center;
                align-items: center;
                line-height: 1;
                padding: 0;
            " onmouseover="this.style.background='#ff4444'; this.style.borderColor='#ff4444'; this.style.color='white'; this.style.transform='rotate(90deg)'" 
               onmouseout="this.style.background='rgba(255,255,255,0.05)'; this.style.borderColor='#89cfd8'; this.style.color='#89cfd8'; this.style.transform='rotate(0)'">✕</button>
        `;

        const contentArea = document.createElement('div');
        contentArea.style.cssText = `
            flex: 1;
            display: flex;
            overflow: hidden;
        `;

        const leftPanel = document.createElement('div');
        leftPanel.style.cssText = `
            width: 320px;
            height: 800px;
            background: rgba(0, 0, 0, 0.4);
            border-right: 2px solid rgba(137, 207, 216, 0.2);
            padding: 25px;
            overflow-y: hidden;
        `;

        let relationButtons = '<div style="color: #89cfd8; font-size: 20px;  font-family: Arial; font-weight: bold; margin-bottom: 25px;letter-spacing: 1px;">選擇設備校準</div>';
        let buttonIndex = 0;
        
        if (devices.visions.length > 0 && devices.microphones.length > 0) {
            relationButtons += `
                <button id="dcm-btn-vm" class="dcm-relation-btn ${buttonIndex === 0 ? 'active' : ''}" 
                        data-source-type="vision" data-target-type="microphone">
                    <div style="font-size: 20px; font-weight: bold; margin-bottom: 6px;">BC-200 × Microphone</div>
                </button>
            `;
            buttonIndex++;
        }
        
        if (devices.visions.length > 0 && devices.cameras.length > 0) {
            relationButtons += `
                <button id="dcm-btn-vc" class="dcm-relation-btn ${buttonIndex === 0 ? 'active' : ''}" 
                        data-source-type="vision" data-target-type="camera">
                    <div style="font-size: 20px; font-weight: bold; margin-bottom: 6px;">BC-200 × Camera</div>
                </button>
            `;
            buttonIndex++;
        }
        
        if (devices.cameras.length > 0 && devices.microphones.length > 0) {
            relationButtons += `
                <button id="dcm-btn-cm" class="dcm-relation-btn ${buttonIndex === 0 ? 'active' : ''}" 
                        data-source-type="camera" data-target-type="microphone">
                    <div style="font-size: 20px; font-weight: bold; margin-bottom: 6px;">Camera × Microphone</div>
                </button>
            `;
        }

        leftPanel.innerHTML = relationButtons;
        
        setTimeout(() => {
            document.querySelectorAll('.dcm-relation-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.querySelectorAll('.dcm-relation-btn').forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                    
                    const sourceType = this.getAttribute('data-source-type');
                    const targetType = this.getAttribute('data-target-type');
                    
                    const sourceSelect = document.querySelector('#Device_Config_Map_Window select[data-type="source"]');
                    const targetSelect = document.querySelector('#Device_Config_Map_Window select[data-type="target"]');
                    
                    if (sourceSelect && targetSelect) {
                        let sourceHTML = '';
                        let targetHTML = '';
                        
                        if (sourceType === 'vision') {
                            sourceHTML = devices.visions.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
                        } else if (sourceType === 'microphone') {
                            sourceHTML = devices.microphones.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
                        } else if (sourceType === 'camera') {
                            sourceHTML = devices.cameras.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
                        }
                        
                        if (targetType === 'vision') {
                            targetHTML = devices.visions.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
                        } else if (targetType === 'microphone') {
                            targetHTML = devices.microphones.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
                        } else if (targetType === 'camera') {
                            targetHTML = devices.cameras.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
                        }
                        
                        sourceSelect.innerHTML = sourceHTML;
                        targetSelect.innerHTML = targetHTML;
                    }

                    const calibrationContainer = document.getElementById('ZoneFullMap_calibration_button_container');
                    if (calibrationContainer) 
                    {
                        if (sourceType === 'vision' && targetType === 'camera') 
                        {
                            calibrationContainer.style.display = 'block';
                        } 
                        else 
                        {
                            calibrationContainer.style.display = 'none';
                        }
                    }
                });
            });

            const firstActiveBtn = document.querySelector('.dcm-relation-btn.active');
            if (firstActiveBtn) 
            {
                const sourceType = firstActiveBtn.getAttribute('data-source-type');
                const targetType = firstActiveBtn.getAttribute('data-target-type');
                const calibrationContainer = document.getElementById('ZoneFullMap_calibration_button_container');
                
                if (calibrationContainer && sourceType === 'vision' && targetType === 'camera') 
                {
                    calibrationContainer.style.display = 'block';
                }
            }
        }, 100);

        const centerPanel = document.createElement('div');
        centerPanel.style.cssText = `
            flex: 1;
            padding: 30px;
            overflow-y: hidden;
        `;

        let sourceOptions = '';
        let targetOptions = '';
        
        if (devices.visions.length > 0 && devices.microphones.length > 0) {
            sourceOptions = devices.visions.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
            targetOptions = devices.microphones.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
        } else if (devices.visions.length > 0 && devices.cameras.length > 0) {
            sourceOptions = devices.visions.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
            targetOptions = devices.cameras.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
        } else if (devices.cameras.length > 0 && devices.microphones.length > 0) {
            sourceOptions = devices.cameras.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
            targetOptions = devices.microphones.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
        } else if (devices.visions.length > 0) {
            sourceOptions = devices.visions.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
            targetOptions = '<option value="null">No target device</option>';
        } else if (devices.microphones.length > 0) {
            sourceOptions = devices.microphones.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
            targetOptions = '<option value="null">No target device</option>';
        } else if (devices.cameras.length > 0) {
            sourceOptions = devices.cameras.map(d => `<option value="${d.id}">${d.name}</option>`).join('');
            targetOptions = '<option value="null">No target device</option>';
        }

        centerPanel.innerHTML = `
            <div style="background: rgba(0, 0, 0, 0.4); border-radius: 12px; padding: 25px; margin-bottom: 25px;">
                <h3 style="color: #89cfd8; font-size: 16px; margin-bottom: 25px; font-weight: 600;">選擇設備</h3>
                <div style="display: flex; align-items: center; gap: 25px;">
                    <div style="flex: 1;">
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 14px; display: block; margin-bottom: 10px; font-weight: 500;">來源設備</label>
                        <select data-type="source" style="width: 100%; padding: 12px; background: white; border: 2px solid #89cfd8; border-radius: 8px; color: black; font-size: 15px;">
                            ${sourceOptions}
                        </select>
                    </div>
                    <span style="font-size: 28px; color: #89cfd8;">→</span>
                    <div style="flex: 1;">
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 14px; display: block; margin-bottom: 10px; font-weight: 500;">目標設備</label>
                        <select data-type="target" style="width: 100%; padding: 12px; background: white; border: 2px solid #89cfd8; border-radius: 8px; color: black; font-size: 15px;">
                            ${targetOptions}
                        </select>
                    </div>
                </div>
            </div>

            <div style="background: rgba(0, 0, 0, 0.4); border-radius: 12px; padding: 25px; margin-bottom: 25px;">
                <h3 style="color: #89cfd8; font-size: 16px; margin-bottom: 25px; font-weight: 600;">相關參數</h3>
                <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">
                    <div>
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 13px; display: block; margin-bottom: 8px; font-weight: 500;">偏移中心 X (m)</label>
                        <input type="number" value="0" step="0.1" style="width: 100%; padding: 10px; background: white; border: 2px solid rgba(137, 207, 216, 0.4); border-radius: 6px; color: black; font-size: 15px;">
                    </div>
                    <div>
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 13px; display: block; margin-bottom: 8px; font-weight: 500;">偏移中心 Y (m)</label>
                        <input type="number" value="0" step="0.1" style="width: 100%; padding: 10px; background: white; border: 2px solid rgba(137, 207, 216, 0.4); border-radius: 6px; color: black; font-size: 15px;">
                    </div>
                    <div>
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 13px; display: block; margin-bottom: 8px; font-weight: 500;">偏移中心角度 (°)</label>
                        <input type="number" value="0" step="1" style="width: 100%; padding: 10px; background: white; border: 2px solid rgba(137, 207, 216, 0.4); border-radius: 6px; color: black; font-size: 15px;">
                    </div>
                    <div>
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 13px; display: block; margin-bottom: 8px; font-weight: 500;">偏移Pan (°)</label>
                        <input type="number" value="0" step="1" style="width: 100%; padding: 10px; background: white; border: 2px solid rgba(137, 207, 216, 0.4); border-radius: 6px; color: black; font-size: 15px;">
                    </div>
                    <div>
                        <label style="color: rgba(137, 207, 216, 0.7); font-size: 13px; display: block; margin-bottom: 8px; font-weight: 500;">偏移Tilt (°)</label>
                        <input type="number" value="0" step="1" style="width: 100%; padding: 10px; background: white; border: 2px solid rgba(137, 207, 216, 0.4); border-radius: 6px; color: black; font-size: 15px;">
                    </div>
                </div>
            </div>

            <div id="ZoneFullMap_calibration_button_container" style="
                background: rgba(0, 0, 0, 0.4); 
                border-radius: 12px; 
                padding: 25px; 
                display: none;
            ">
                <button id="ZoneFullMap_open_calibration_btn" class="BC200PanTiltBtn_style"
                onclick="ZoneFullMap_openBC200CameraCalibration()"  style="margin-left:200px; width:450px; height:48px; font-size: 20px;">
                    Open BC-200 & Camera Calibration Settings
                </button>
            </div>
        `;

        const bottomBar = document.createElement('div');
        bottomBar.style.cssText = `
            padding: 18px 25px;
            background: rgba(0, 0, 0, 0.6);
            border-top: 2px solid rgba(137, 207, 216, 0.3);
            display: flex;
            justify-content: space-between;
            align-items: center;
        `;

        bottomBar.innerHTML = `
            <div>
                <button class="BC200PanTiltBtn_style" style="margin-right: 12px; width:160px; height:32px; font-size: 18px;">匯出配置</button>
                <button class="BC200PanTiltBtn_style" style="font-size: 18px; width:160px; height:32px;">導入配置</button>
            </div>
            <button class="BC200PanTiltBtn_style" style="width:160px; height:32px;font-size: 18px;">Apply</button>
        `;

        contentArea.appendChild(leftPanel);
        contentArea.appendChild(centerPanel);
        
        configMapWindow.appendChild(titleBar);
        configMapWindow.appendChild(contentArea);
        configMapWindow.appendChild(bottomBar);
        
        document.body.appendChild(configMapWindow);

        // 新增: 拖動功能
        let isDragging = false;
        let currentX = 0, currentY = 0, initialX = 0, initialY = 0;
        let xOffset = 0, yOffset = 0;

        titleBar.addEventListener('mousedown', (e) => {
            if (e.target.id === 'closeDeviceConfigMapBtn' || e.target.closest('#closeDeviceConfigMapBtn')) return;
            
            initialX = e.clientX - xOffset;
            initialY = e.clientY - yOffset;
            
            if (e.target === titleBar || titleBar.contains(e.target)) {
                isDragging = true;
                configMapWindow.style.transition = 'none';
            }
        });

        document.addEventListener('mousemove', (e) => {
            if (isDragging) {
                e.preventDefault();
                currentX = e.clientX - initialX;
                currentY = e.clientY - initialY;
                xOffset = currentX;
                yOffset = currentY;
                configMapWindow.style.top = `calc(10% + ${currentY}px)`;
                configMapWindow.style.right = `calc(5% - ${currentX}px)`;
            }
        });

        document.addEventListener('mouseup', () => {
            initialX = currentX;
            initialY = currentY;
            isDragging = false;
        });

    } catch (error) {
        alert('Failed to open Device Config Map: ' + error.message);
    }
}

var gCalibration_CurrentBC200Id = null;
var gCalibration_CurrentCameraId = null;
var gIsCalibrationMode = false;

function ZoneFullMap_openBC200CameraCalibration() 
{
    const sourceSelect = document.querySelector('#Device_Config_Map_Window select[data-type="source"]');
    const targetSelect = document.querySelector('#Device_Config_Map_Window select[data-type="target"]');
    
    if (!sourceSelect || !targetSelect) {
        alert('Please select devices first');
        return;
    }
    
    const bc200Id = sourceSelect.value;
    const cameraId = targetSelect.value;
    
    if (!bc200Id || !cameraId) {
        alert('Please select BC-200 and Camera devices');
        return;
    }
    
    const bc200Device = ZoneFullMap_devices.find(d => d.id === bc200Id);
    const cameraDevice = ZoneFullMap_devices.find(d => d.id === cameraId);
    
    if (!bc200Device || !cameraDevice) {
        alert('Device information not found');
        return;
    }
    
    // 保存到全局變數
    gCalibration_CurrentBC200Id = bc200Id;
    gCalibration_CurrentCameraId = cameraId;
    gIsCalibrationMode = true;
    
    // 更新全局 IP
    gZoneFullMap_CurrentBC200IP = extractIPFromDisplayName(bc200Device.connectedCamera || bc200Device.name);
    gZoneFullMap_CurrentCameraIP = extractIPFromDisplayName(cameraDevice.connectedCamera || cameraDevice.name);
    
    // 1. 關閉 Device Config Map 視窗
    ZoneFullMap_closeDeviceConfigMap();
    
    // 2. 開啟兩個視窗
    ZoneFullMap_BC200View(bc200Id);
    ZoneFullMap_PTZControl(cameraId);
    
    // 3. 延遲一點再排版，確保視窗已經創建完成
    setTimeout(() => {
        ZoneFullMap_arrangeCalibrationWindows();
    }, 30);

}


function ZoneFullMap_closeDeviceConfigMap() {
    const configMapWindow = document.getElementById('Device_Config_Map_Window');
    if (configMapWindow) {
        configMapWindow.remove();
    }
    
    const configBtn = document.getElementById('ZoneFullMapDeviceConfigBtn');
    if (configBtn) {
        configBtn.classList.remove('active');
    }
}

function ZoneFullMap_arrangeCalibrationWindows() {
    const bc200Window = document.getElementById('cameralist_FullMapBC200ViewWindows');
    const cameraWindow = document.getElementById('cameralist_FullMapCameraPTZWindows');
    
    if (bc200Window && cameraWindow) {
        const windowWidth = window.innerWidth;
        const windowHeight = window.innerHeight;
        const modalWidth = 700;  // 固定寬度
        const gap = 20;  // 兩個視窗之間的間距
        
        // 計算總寬度（兩個視窗 + 間距）
        const totalWidth = modalWidth * 2 + gap;
        
        // 計算起始位置（讓整體居中）
        const startX = (windowWidth - totalWidth) / 2;
        
        // Camera 在左邊
        cameraWindow.style.cssText = `
            position: fixed;
            left: ${startX}px;
            top: 50%;
            transform: translateY(-50%);
            z-index: 10001;
        `;

        // BC200 在右邊
        bc200Window.style.cssText = `
            position: fixed;
            left: ${startX + modalWidth + gap}px;
            top: 50%;
            transform: translateY(-50%);
            z-index: 10001;
        `;
    }
}

// function ZoneFullMap_arrangeCalibrationWindows() {
//     const bc200Window = document.getElementById('cameralist_FullMapBC200ViewWindows');
//     const cameraWindow = document.getElementById('cameralist_FullMapCameraPTZWindows');
    
//     if (bc200Window && cameraWindow) {

//         cameraWindow.style.cssText = `
//             position: fixed;
//             left: 30%;
//             top: 50%;
//         `;

//         bc200Window.style.cssText = `
//             position: fixed;
//             left: 60%;
//             top: 50%;
//         `;
//     }
// }

function ZoneFullMap_setupDeviceButtonDragHandlers() {
    const deviceButtons = document.querySelectorAll('.ZoneFullMap_draggable_device_btn');
    
    deviceButtons.forEach(button => {
        button.removeAttribute('onclick');
        
        button.addEventListener('click', function(e) {
            if (!gZoneFullMap_IsDraggingFromButton) {
                const deviceType = this.dataset.deviceType;
                switch(deviceType) {
                    case 'microphone':
                        ZoneFullMap_createMicrophone();
                        break;
                    case 'camera':
                        ZoneFullMap_addCamera();
                        break;
                    case 'vision':
                        ZoneFullMap_addVision();
                        break;
                    case 'anchor':
                        ZoneFullMap_addAnchorPoint();
                        break;
                }
            }
        });
        
        button.addEventListener('mousedown', function(e) {
            e.preventDefault();
            e.stopPropagation();
            ZoneFullMap_startDeviceButtonDrag(e, this);
        });
    });
}


function ZoneFullMap_startDeviceButtonDrag(e, button) {
    const deviceType = button.dataset.deviceType;
    
    const limits = {
        'microphone': 6,
        'camera': 6,
        'vision': 6,
        'anchor': 1
    };
    
    const currentCount = ZoneFullMap_devices.filter(d => d.type === deviceType).length;
    if (currentCount >= limits[deviceType]) {
        return;
    }
    
    gZoneFullMap_DraggedDeviceType = deviceType;
    gZoneFullMap_DragStartPos = { x: e.clientX, y: e.clientY };
    gZoneFullMap_IsDraggingFromButton = false;
    
    ZoneFullMap_createDeviceDragPreview(deviceType, e.clientX, e.clientY);
    
    button.style.opacity = '0.5';
    button.dataset.draggingState = 'true';
    
    document.addEventListener('mousemove', ZoneFullMap_onDeviceButtonDragMove, { capture: true });
    document.addEventListener('mouseup', ZoneFullMap_onDeviceButtonDragEnd, { capture: true });
}

function ZoneFullMap_checkDeviceLimit(deviceType) {
    const limits = {
        'microphone': 6,
        'camera': 6,
        'vision': 6,
        'anchor': 1
    };
    
    const currentCount = ZoneFullMap_devices.filter(d => d.type === deviceType).length;
    return currentCount < limits[deviceType];
}

function ZoneFullMap_createDeviceDragPreview(deviceType, x, y) {
    
    if (gZoneFullMap_DragPreview) {
        gZoneFullMap_DragPreview.remove();
    }
    
    const scale = gDragSystem?.canvasTransform?.scale || 0.05;
    const predictedNumber = ZoneFullMap_predictNextDeviceNumber(deviceType);
    
    let baseSize;
    
    switch(deviceType) {
        case 'microphone':
            baseSize = 0.6;
            break;
        case 'camera':
            baseSize = ZoneFullMap_DeviceVisualConfig?.camera?.baseCircleSize || 0.3;
            break;
        case 'vision':
            baseSize = ZoneFullMap_DeviceVisualConfig?.bc200?.baseCircleSize || 0.3;
            break;
        case 'anchor':
            baseSize = 0.3;
            break;
        default:
            baseSize = 0.4;
    }
    
    let previewSize_raw = baseSize * scale * 100;
    
    
    if (deviceType === 'microphone') {
        previewSize_raw = previewSize_raw + 16;
    }
    
    const previewSize_clamped = Math.max(20, Math.min(200, previewSize_raw));
    const previewSize = previewSize_clamped;
    
    gZoneFullMap_DragPreview = document.createElement('div');
    gZoneFullMap_DragPreview.className = 'ZoneFullMap_device_drag_preview';
    gZoneFullMap_DragPreview.style.cssText = `
        position: fixed;
        pointer-events: none;
        z-index: 10000;
        opacity: 0.7;
        width: ${previewSize}px;
        height: ${previewSize}px;
        left: ${x - previewSize/2}px;
        top: ${y - previewSize/2}px;
        transform: translate(0, 0);
        will-change: transform;
        overflow: visible;
    `;
    
    gZoneFullMap_DragPreview.dataset.predictedNumber = predictedNumber;
    gZoneFullMap_DragPreview.dataset.deviceType = deviceType;
    gZoneFullMap_DragPreview.dataset.previewSize = previewSize;
    
    if (deviceType === 'microphone') {
        const micId = `mic_${predictedNumber}`;
        const micColorScheme = ZoneFullMap_getMicrophoneColorScheme(micId);
        
        const actualWidthPx = baseSize * scale * 100;
        const imgSize = actualWidthPx;
        const borderSize = previewSize;
        const padding = (borderSize - imgSize) / 2;
        
        const img = document.createElement('img');
        img.src = './images/Web_RMCG.png';
        img.style.cssText = `
            width: ${imgSize}px; 
            height: ${imgSize}px; 
            object-fit: contain; 
            position: absolute; 
            top: ${padding}px; 
            left: ${padding}px;
        `;
        
        const borderDiv = document.createElement('div');
        borderDiv.style.cssText = `
            position: absolute;
            top: 0;
            left: 0;
            width: ${borderSize}px;
            height: ${borderSize}px;
            border: 3px dashed ${micColorScheme.primary};
            pointer-events: none;
            box-sizing: border-box;
        `;
        
        gZoneFullMap_DragPreview.appendChild(borderDiv);
        gZoneFullMap_DragPreview.appendChild(img);
        
        img.onerror = function() {
            ZoneFullMap_createCanvasPreviewWithNumber(gZoneFullMap_DragPreview, deviceType, predictedNumber, previewSize);
        };
    } else {
        ZoneFullMap_createCanvasPreviewWithNumber(gZoneFullMap_DragPreview, deviceType, predictedNumber, previewSize);
    }
    
    document.body.appendChild(gZoneFullMap_DragPreview);
}

function ZoneFullMap_createCanvasPreviewWithNumber(container, deviceType, deviceNumber, containerSize) {
    const existingBorder = container.querySelector('div[style*="border"]');
    if (!existingBorder) 
    {
        container.innerHTML = '';
    }

    const dpr = window.devicePixelRatio || 1;
    
    const canvasSize = containerSize * 6;
    const canvas = document.createElement('canvas');
    canvas.width = canvasSize * dpr;
    canvas.height = canvasSize * dpr;
    const ctx = canvas.getContext('2d');
    
    ctx.scale(dpr, dpr);

    const cx = canvasSize / 2;
    const cy = canvasSize / 2;
    
    canvas.style.position = 'absolute';
    canvas.style.width = canvasSize + 'px';
    canvas.style.height = canvasSize + 'px';
    canvas.style.left = -(canvasSize - containerSize) / 2 + 'px';
    canvas.style.top = -(canvasSize - containerSize) / 2 + 'px';
    
    switch(deviceType) {
        case 'camera':
            const cameraId = `camera_${deviceNumber}`;
            const cameraColorScheme = ZoneFullMap_getCameraColorScheme(cameraId);
            
            const cameraRadius = containerSize / 2;
            const cameraAngle = -Math.PI / 2;
            const arrowLength = containerSize * 2;
            const arrowHeadSize = containerSize * 0.65;
            
            ctx.strokeStyle = cameraColorScheme.primary;
            ctx.lineWidth = 4;
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            const fovEndX = cx + arrowLength * Math.cos(cameraAngle);
            const fovEndY = cy + arrowLength * Math.sin(cameraAngle);
            ctx.lineTo(fovEndX, fovEndY);
            ctx.stroke();
            
            const arrowAngle = Math.PI / 6;
            ctx.beginPath();
            ctx.moveTo(fovEndX, fovEndY);
            ctx.lineTo(
                fovEndX - Math.cos(cameraAngle + arrowAngle) * arrowHeadSize,
                fovEndY - Math.sin(cameraAngle + arrowAngle) * arrowHeadSize
            );
            ctx.moveTo(fovEndX, fovEndY);
            ctx.lineTo(
                fovEndX - Math.cos(cameraAngle - arrowAngle) * arrowHeadSize,
                fovEndY - Math.sin(cameraAngle - arrowAngle) * arrowHeadSize
            );
            ctx.stroke();
            
            ctx.beginPath();
            ctx.arc(cx, cy, cameraRadius, 0, 2 * Math.PI);
            ctx.fillStyle = cameraColorScheme.primary;
            ctx.fill();
            break;
            
        case 'vision':
            const visionId = `vision_${deviceNumber}`;
            const visionColorScheme = ZoneFullMap_getVisionColorScheme(visionId);
            
            const bc200Angle = -Math.PI / 2;
            const bc200ArrowLength = containerSize * 2;
            const bc200ArrowHeadSize = containerSize * 0.65;
            
            ctx.strokeStyle = visionColorScheme.primary;
            ctx.lineWidth = 4;
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            const bc200FovEndX = cx + bc200ArrowLength * Math.cos(bc200Angle);
            const bc200FovEndY = cy + bc200ArrowLength * Math.sin(bc200Angle);
            ctx.lineTo(bc200FovEndX, bc200FovEndY);
            ctx.stroke();
            
            const bc200ArrowAngle = Math.PI / 6;
            ctx.beginPath();
            ctx.moveTo(bc200FovEndX, bc200FovEndY);
            ctx.lineTo(
                bc200FovEndX - Math.cos(bc200Angle + bc200ArrowAngle) * bc200ArrowHeadSize,
                bc200FovEndY - Math.sin(bc200Angle + bc200ArrowAngle) * bc200ArrowHeadSize
            );
            ctx.moveTo(bc200FovEndX, bc200FovEndY);
            ctx.lineTo(
                bc200FovEndX - Math.cos(bc200Angle - bc200ArrowAngle) * bc200ArrowHeadSize,
                bc200FovEndY - Math.sin(bc200Angle - bc200ArrowAngle) * bc200ArrowHeadSize
            );
            ctx.stroke();
            
            const rectWidth = containerSize * 1.2;
            const rectHeight = containerSize;
            const borderRadius = containerSize * 0.2;
            
            const x = cx - rectWidth / 2;
            const y = cy - rectHeight / 2;
            
            ctx.beginPath();
            ctx.moveTo(x + borderRadius, y);
            ctx.lineTo(x + rectWidth - borderRadius, y);
            ctx.quadraticCurveTo(x + rectWidth, y, x + rectWidth, y + borderRadius);
            ctx.lineTo(x + rectWidth, y + rectHeight - borderRadius);
            ctx.quadraticCurveTo(x + rectWidth, y + rectHeight, x + rectWidth - borderRadius, y + rectHeight);
            ctx.lineTo(x + borderRadius, y + rectHeight);
            ctx.quadraticCurveTo(x, y + rectHeight, x, y + rectHeight - borderRadius);
            ctx.lineTo(x, y + borderRadius);
            ctx.quadraticCurveTo(x, y, x + borderRadius, y);
            ctx.closePath();
            
            ctx.fillStyle = visionColorScheme.primary;
            ctx.fill();
            break;
            
        case 'anchor':
            const anchorRadius = containerSize / 2;
            
            ctx.beginPath();
            ctx.arc(cx, cy, anchorRadius * 2.5, 0, 2 * Math.PI);
            ctx.strokeStyle = 'rgba(255, 0, 0, 0.3)';
            ctx.lineWidth = 3;
            ctx.stroke();
            
            ctx.beginPath();
            ctx.arc(cx, cy, anchorRadius, 0, 2 * Math.PI);
            ctx.fillStyle = 'red';
            ctx.fill();
            ctx.closePath();
            break;
            
        case 'microphone':
            const micId = `mic_${deviceNumber}`;
            const micColorScheme = ZoneFullMap_getMicrophoneColorScheme(micId);
            
            const borderSize = containerSize;
            const borderOffset = (canvasSize - borderSize) / 2;
            
            ctx.save();
            ctx.setLineDash([8, 4]);
            ctx.beginPath();
            ctx.rect(borderOffset, borderOffset, borderSize, borderSize);
            ctx.strokeStyle = micColorScheme.primary;
            ctx.lineWidth = 3;
            ctx.stroke();
            ctx.setLineDash([]);
            ctx.restore();
            
            const micSize = containerSize * 0.6;
            ctx.fillStyle = '#6a6a6a';
            ctx.strokeStyle = '#3a3a3a';
            ctx.lineWidth = 1.5;
            
            ctx.beginPath();
            ctx.ellipse(cx, cy - micSize * 0.15, micSize * 0.3, micSize * 0.4, 0, 0, Math.PI * 2);
            ctx.fill();
            ctx.stroke();
            
            ctx.strokeStyle = '#4a4a4a';
            ctx.lineWidth = 0.8;
            for (let i = -4; i <= 4; i++) {
                ctx.beginPath();
                ctx.moveTo(cx + i * (micSize * 0.08), cy - micSize * 0.45);
                ctx.lineTo(cx + i * (micSize * 0.08), cy + micSize * 0.05);
                ctx.stroke();
            }
            
            ctx.fillStyle = '#3a3a3a';
            ctx.fillRect(cx - micSize * 0.08, cy + micSize * 0.12, micSize * 0.16, micSize * 0.2);
            
            ctx.beginPath();
            ctx.ellipse(cx, cy + micSize * 0.35, micSize * 0.2, micSize * 0.08, 0, 0, Math.PI * 2);
            ctx.fill();
            break;
    }
    
    container.appendChild(canvas);
}

function ZoneFullMap_createCanvasPreview(container, deviceType) {
    container.innerHTML = '';
    
    const canvas = document.createElement('canvas');
    canvas.width = 60;
    canvas.height = 60;
    const ctx = canvas.getContext('2d');
    
    const cx = 30;
    const cy = 30;
    
    const nextDeviceNumber = ZoneFullMap_getNextDeviceId(deviceType);
    
    switch(deviceType) {
        case 'camera':
            const cameraId = `camera_${nextDeviceNumber}`;
            const cameraColorScheme = ZoneFullMap_getCameraColorScheme(cameraId);
            const cameraConfig = ZoneFullMap_DeviceVisualConfig?.camera || {
                baseCircleSize: 0.4,
                arrow: {
                    baseLength: 80,
                    lengthMultiplier: 3,
                    headSizeRatio: 0.3,
                    hardwareWidth: 3,
                    fovWidth: 4,
                    dashPattern: [5, 3]
                }
            };
            
            ctx.beginPath();
            ctx.arc(cx, cy, 20, 0, 2 * Math.PI);
            ctx.fillStyle = cameraColorScheme.primary;
            ctx.fill();
            
            const arrowLength = 25;
            
            ctx.save();
            ctx.setLineDash(cameraConfig.arrow.dashPattern);
            ctx.strokeStyle = 'black';
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            ctx.lineTo(cx + arrowLength, cy);
            ctx.stroke();
            ctx.restore();
            
            ctx.strokeStyle = cameraColorScheme.primary;
            ctx.lineWidth = 3;
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            ctx.lineTo(cx + arrowLength - 3, cy - 3);
            ctx.stroke();
            
            ctx.fillStyle = cameraColorScheme.primary;
            ctx.beginPath();
            ctx.moveTo(cx + arrowLength, cy);
            ctx.lineTo(cx + arrowLength - 5, cy - 3);
            ctx.lineTo(cx + arrowLength - 5, cy + 3);
            ctx.closePath();
            ctx.fill();
            break;
            
        case 'vision':
            const visionId = `vision_${nextDeviceNumber}`;
            const visionColorScheme = ZoneFullMap_getVisionColorScheme(visionId);
            const visionConfig = ZoneFullMap_DeviceVisualConfig?.bc200 || {
                baseCircleSize: 0.5,
                arrow: {
                    baseLength: 100,
                    lengthMultiplier: 2.5,
                    headSizeRatio: 0.35,
                    width: 4
                }
            };
            
            const rectWidth = 35;
            const rectHeight = 42;
            const borderRadius = 7;
            
            ctx.save();
            ctx.translate(cx, cy);
            
            const x = -rectWidth / 2;
            const y = -rectHeight / 2;
            
            ctx.beginPath();
            ctx.moveTo(x + borderRadius, y);
            ctx.lineTo(x + rectWidth - borderRadius, y);
            ctx.quadraticCurveTo(x + rectWidth, y, x + rectWidth, y + borderRadius);
            ctx.lineTo(x + rectWidth, y + rectHeight - borderRadius);
            ctx.quadraticCurveTo(x + rectWidth, y + rectHeight, x + rectWidth - borderRadius, y + rectHeight);
            ctx.lineTo(x + borderRadius, y + rectHeight);
            ctx.quadraticCurveTo(x, y + rectHeight, x, y + rectHeight - borderRadius);
            ctx.lineTo(x, y + borderRadius);
            ctx.quadraticCurveTo(x, y, x + borderRadius, y);
            ctx.closePath();
            
            ctx.fillStyle = visionColorScheme.primary;
            ctx.fill();
            
            ctx.restore();
            
            const bc200ArrowLength = 25;
            
            ctx.save();
            ctx.setLineDash([5, 3]);
            ctx.strokeStyle = 'black';
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            ctx.lineTo(cx, cy - bc200ArrowLength);
            ctx.stroke();
            ctx.restore();
            
            ctx.strokeStyle = visionColorScheme.primary;
            ctx.lineWidth = 3;
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            ctx.lineTo(cx + 3, cy - bc200ArrowLength + 3);
            ctx.stroke();
            
            ctx.fillStyle = visionColorScheme.primary;
            ctx.beginPath();
            ctx.moveTo(cx + 3, cy - bc200ArrowLength);
            ctx.lineTo(cx, cy - bc200ArrowLength + 5);
            ctx.lineTo(cx + 6, cy - bc200ArrowLength + 5);
            ctx.closePath();
            ctx.fill();
            break;
            
        case 'anchor':
            ctx.beginPath();
            ctx.arc(cx, cy, 15, 0, 2 * Math.PI);
            ctx.fillStyle = 'red';
            ctx.fill();
            ctx.closePath();
            
            ctx.strokeStyle = 'white';
            ctx.lineWidth = 2;
            ctx.lineCap = 'round';
            ctx.beginPath();
            ctx.moveTo(cx, cy - 10);
            ctx.lineTo(cx, cy + 5);
            ctx.moveTo(cx - 8, cy - 5);
            ctx.lineTo(cx + 8, cy - 5);
            ctx.arc(cx, cy + 3, 6, Math.PI * 0.2, Math.PI * 0.8, false);
            ctx.stroke();
            break;
            
        case 'microphone':
            const micId = `mic_${nextDeviceNumber}`;
            const micColorScheme = ZoneFullMap_getMicrophoneColorScheme(micId);
            
            ctx.save();
            ctx.setLineDash([8, 4]);
            ctx.beginPath();
            ctx.rect(5, 5, 50, 50);
            ctx.strokeStyle = micColorScheme.primary;
            ctx.lineWidth = 3;
            ctx.stroke();
            ctx.setLineDash([]);
            ctx.restore();
            
            ctx.fillStyle = '#6a6a6a';
            ctx.strokeStyle = '#3a3a3a';
            ctx.lineWidth = 1;
            
            ctx.beginPath();
            ctx.ellipse(cx, cy - 5, 10, 14, 0, 0, Math.PI * 2);
            ctx.fill();
            ctx.stroke();
            
            ctx.strokeStyle = '#4a4a4a';
            ctx.lineWidth = 0.5;
            for (let i = -3; i <= 3; i++) {
                ctx.beginPath();
                ctx.moveTo(cx + i * 3, cy - 15);
                ctx.lineTo(cx + i * 3, cy + 5);
                ctx.stroke();
            }
            
            ctx.fillStyle = '#3a3a3a';
            ctx.fillRect(cx - 3, cy + 9, 6, 8);
            
            ctx.beginPath();
            ctx.ellipse(cx, cy + 20, 8, 3, 0, 0, Math.PI * 2);
            ctx.fill();
            break;
    }
    
    container.appendChild(canvas);
}

function ZoneFullMap_onDeviceButtonDragMove(e) {
    if (!gZoneFullMap_DragPreview) return;
    
    const dx = e.clientX - gZoneFullMap_DragStartPos.x;
    const dy = e.clientY - gZoneFullMap_DragStartPos.y;
    const distance = Math.sqrt(dx * dx + dy * dy);
    
    if (distance > 5 && !gZoneFullMap_IsDraggingFromButton) {
        gZoneFullMap_IsDraggingFromButton = true;
    }
    
    const previewSize = parseFloat(gZoneFullMap_DragPreview.dataset.previewSize) || 80;
    const newLeft = e.clientX - previewSize / 2;
    const newTop = e.clientY - previewSize / 2;
    
    gZoneFullMap_DragPreview.style.left = newLeft + 'px';
    gZoneFullMap_DragPreview.style.top = newTop + 'px';
    
    if (!gZoneFullMap_DragPreview._lastLogTime || Date.now() - gZoneFullMap_DragPreview._lastLogTime > 200) {
        gZoneFullMap_DragPreview._lastLogTime = Date.now();
    }
}

function ZoneFullMap_onDeviceButtonDragEnd(e) {
    document.removeEventListener('mousemove', ZoneFullMap_onDeviceButtonDragMove, { capture: true });
    document.removeEventListener('mouseup', ZoneFullMap_onDeviceButtonDragEnd, { capture: true });
    
    const buttons = document.querySelectorAll('[data-dragging-state="true"]');
    buttons.forEach(btn => {
        btn.style.opacity = '1';
        btn.dataset.draggingState = 'false';
    });
    
    const canvas = document.querySelector('.ZoneFullMap_canvas_td');
    if (canvas) {
        
        const rect = canvas.getBoundingClientRect();
        const isOverCanvas = e.clientX >= rect.left && e.clientX <= rect.right &&
                            e.clientY >= rect.top && e.clientY <= rect.bottom;
        
        if (isOverCanvas && gZoneFullMap_IsDraggingFromButton && gZoneFullMap_DraggedDeviceType) {
            const canvasX = e.clientX - rect.left;
            const canvasY = e.clientY - rect.top;
            
            let deviceX, deviceY;
            
            if (gDragSystem && gDragSystem.canvasTransform) {
                const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;

                deviceX = (canvasX - ox) * 100 / scale;
                deviceY = (oy - canvasY) * 100 / scale;

            } else {
                deviceX = canvasX - canvas.width / 2;
                deviceY = canvas.height / 2 - canvasY;
            }
            
            const customPosition = { x: deviceX, y: deviceY };
            let newDevice = null;
            
            switch(gZoneFullMap_DraggedDeviceType) {
                case 'microphone':
                    newDevice = ZoneFullMap_createMicrophone(customPosition);
                    break;
                case 'camera':
                    newDevice = ZoneFullMap_addCamera(customPosition);
                    break;
                case 'vision':
                    newDevice = ZoneFullMap_addVision(customPosition);
                    break;
                case 'anchor':
                    newDevice = ZoneFullMap_addAnchorPoint(customPosition);
                    break;
            }
            
            if (newDevice) {
            
                gSelectedDevice = newDevice;
                
                requestAnimationFrame(() => {
                    ZoneFullMap_updateDevicePositions();
                    
                    const { x: ox, y: oy, scale } = gDragSystem.canvasTransform;
                    const verifyX = ox + newDevice.x * scale / 100;
                    const verifyY = oy - newDevice.y * scale / 100;
                    
                    setTimeout(() => {
                        ZoneFullMap_highlightSelectedDevice();
                    }, 100);
                });
            }
        }
    }
    
    if (gZoneFullMap_DragPreview) {
        gZoneFullMap_DragPreview.remove();
        gZoneFullMap_DragPreview = null;
    }
    
    gZoneFullMap_DraggedDeviceType = null;
    gZoneFullMap_IsDraggingFromButton = false;
    
}

function ZoneFullMap_createDeviceAtDropPosition(deviceType, mathX, mathY) {
    const deviceNumber = ZoneFullMap_getNextDeviceId(deviceType);
    let device = null;
    
    switch(deviceType) {
        case 'microphone':
            const micCount = ZoneFullMap_devices.filter(d => d.type === 'microphone').length;
            if (micCount >= 6) {
                return null;
            }
            
            device = {
                id: `mic_${deviceNumber}`,
                type: 'microphone',
                name: `Microphone ${deviceNumber}`,
                x: mathX,
                y: mathY,
                angle: 0,
                status: 'Connected',
                imgSrc: './images/Web_RMCG.png',
                baseWidthM: 0.6,
                canDelete: true,
                groupColor: ZoneFullMap_getMicrophoneColorScheme(`mic_${deviceNumber}`).primary,
                imageLoaded: false,
                imageElement: null,
                connectedTabIndex: undefined,
                displayNumber: deviceNumber,
            };
            
            ZoneFullMap_devices.push(device);
            
            ZoneFullMap_preloadMicrophoneImage(device, () => {
                requestAnimationFrame(() => {
                    ZoneFullMap_updateDevicePositions();
                    ZoneFullMap_redrawQuadrilaterals();
                    if (gSelectedDevice && gSelectedDevice.id === device.id) {
                        ZoneFullMap_highlightSelectedDevice();
                    }
                    if (gDragSystem && gDragSystem.drawGrid) {
                        gDragSystem.drawGrid();
                    }
                });
            });
            
            const imgEl = document.createElement('img');
            imgEl.className = 'ZoneFullMap_device';
            imgEl.src = device.imgSrc;
            imgEl.setAttribute('data-id', device.id);
            imgEl.setAttribute('data-x', device.x);
            imgEl.setAttribute('data-y', device.y);
            imgEl.style.transformOrigin = 'center';
            imgEl.style.position = 'absolute';
            imgEl.style.zIndex = '3';
            imgEl.style.display = 'none';
            imgEl.style.pointerEvents = 'none';

            const container = document.querySelector('.ZoneFullMap_canvas_td');
            if (container) {
                container.appendChild(imgEl);
            }
            
            device.imageElement = imgEl;
            
            if (micCount === 0) {
                gSelectedDevice = device;
                setTimeout(() => {
                    ZoneFullMap_highlightSelectedDevice();
                }, 100);
            }
            break;
            
        case 'camera':
            const camCount = ZoneFullMap_devices.filter(d => d.type === 'camera').length;
            if (camCount >= 6) {
                return null;
            }
            
            device = {
                id: `camera_${deviceNumber}`,
                type: 'camera',
                name: `Camera ${deviceNumber}`,
                x: mathX,
                y: mathY,
                angle: 0,
                fovAngle: 0,
                offsetAngle: 0,
                status: 'Connected',
                baseWidthM: 0.4,
                isSelectedCamera: false,
                canDelete: true
            };
            
            ZoneFullMap_devices.push(device);
            break;
            
        case 'vision':
            const visCount = ZoneFullMap_devices.filter(d => d.type === 'vision').length;
            if (visCount >= 6) {
                return null;
            }
            
            device = {
                id: `vision_${deviceNumber}`,
                type: 'vision',
                name: `BC-200 ${deviceNumber}`,
                x: mathX,
                y: mathY,
                angle: 0,
                fovAngle: 0,
                offsetAngle: 0,
                offsetX: 0,
                offsetY: 0,
                status: 'Null',
                baseWidthM: 0.5,
                canDelete: true
            };
            
            ZoneFullMap_devices.push(device);
            break;
            
        case 'anchor':
            const anchorCount = ZoneFullMap_devices.filter(d => d.type === 'anchor').length;
            if (anchorCount >= 1) {
                return null;
            }
            
            device = {
                id: `anchor_${deviceNumber}`,
                type: 'anchor',
                name: `Anchor ${deviceNumber}`,
                x: mathX,
                y: mathY,
                status: 'Active',
                baseWidthM: 30,
                canDelete: true
            };
            
            ZoneFullMap_devices.push(device);
            break;
    }
    
    if (device) {
        gSelectedDevice = device;
        
        ZoneFullMap_updateButtonStates();
        ZoneFullMap_updateDevicePositions();
        
        setTimeout(() => {
            ZoneFullMap_highlightSelectedDevice();
        }, 100);
    }

    return device;
}

function ZoneFullMap_createDeviceAtSpecificPosition(deviceType, mathX, mathY) {
    const deviceNumber = ZoneFullMap_getNextDeviceId(deviceType);
    let device = null;
    
    switch(deviceType) {
        case 'microphone':
            device = ZoneFullMap_createMicrophoneAtPosition(deviceNumber, mathX, mathY);
            break;
        case 'camera':
            device = ZoneFullMap_createCameraAtPosition(deviceNumber, mathX, mathY);
            break;
        case 'vision':
            device = ZoneFullMap_createVisionAtPosition(deviceNumber, mathX, mathY);
            break;
        case 'anchor':
            device = ZoneFullMap_createAnchorAtPosition(deviceNumber, mathX, mathY);
            break;
    }
    
    if (device) {
        gSelectedDevice = device;
        ZoneFullMap_updateButtonStates();
        ZoneFullMap_updateDevicePositions();
        
        setTimeout(() => {
            ZoneFullMap_highlightSelectedDevice();
        }, 100);
    }
    
    return device;
}

function ZoneFullMap_createMicrophoneAtPosition(deviceNumber, mathX, mathY) {
    const microphoneCount = ZoneFullMap_devices.filter(d => d.type === 'microphone').length;
    
    if (microphoneCount >= 6) {
        return null;
    }
    
    const device = {
        id: `mic_${deviceNumber}`,
        type: 'microphone',
        name: `Microphone ${deviceNumber}`,
        x: mathX,
        y: mathY,
        angle: 0,
        status: 'Connected',
        imgSrc: './images/Web_RMCG.png',
        baseWidthM: 0.6,
        canDelete: true,
        groupColor: ZoneFullMap_getMicrophoneColorScheme(`mic_${deviceNumber}`).primary,
        imageLoaded: false,
        imageElement: null,
        connectedTabIndex: undefined,
        displayNumber: deviceNumber,
    };
    
    ZoneFullMap_devices.push(device);
    
    ZoneFullMap_preloadMicrophoneImage(device, () => {
        requestAnimationFrame(() => {
            ZoneFullMap_updateDevicePositions();
            ZoneFullMap_redrawQuadrilaterals();
            if (gSelectedDevice) {
                ZoneFullMap_highlightSelectedDevice();
            }
            if (gDragSystem && gDragSystem.drawGrid) {
                gDragSystem.drawGrid();
            }
        });
    });
    
    const imgEl = document.createElement('img');
    imgEl.className = 'ZoneFullMap_device';
    imgEl.src = device.imgSrc;
    imgEl.setAttribute('data-id', device.id);
    imgEl.setAttribute('data-x', device.x);
    imgEl.setAttribute('data-y', device.y);
    imgEl.style.transformOrigin = 'center';
    imgEl.style.position = 'absolute';
    imgEl.style.zIndex = '3';
    imgEl.style.display = 'none';
    imgEl.style.pointerEvents = 'none';

    const container = document.querySelector('.ZoneFullMap_canvas_td');
    if (container) {
        container.appendChild(imgEl);
    }
    
    device.imageElement = imgEl;
    
    if (microphoneCount === 0) {
        gSelectedDevice = device;
        setTimeout(() => {
            ZoneFullMap_highlightSelectedDevice();
        }, 100);
    }
    
    return device;
}

function ZoneFullMap_createCameraAtPosition(deviceNumber, mathX, mathY) {
    const cameraCount = ZoneFullMap_devices.filter(d => d.type === 'camera').length;
    
    if (cameraCount >= 6) {
        return null;
    }
    
    const device = {
        id: `camera_${deviceNumber}`,
        type: 'camera',
        name: `Camera ${deviceNumber}`,
        x: mathX,
        y: mathY,
        angle: 0,
        fovAngle: 0,
        offsetAngle: 0,
        status: 'Connected',
        baseWidthM: 0.4,
        isSelectedCamera: false,
        canDelete: true
    };
    
    ZoneFullMap_devices.push(device);
    return device;
}

function ZoneFullMap_createVisionAtPosition(deviceNumber, mathX, mathY) {
    const visionCount = ZoneFullMap_devices.filter(d => d.type === 'vision').length;
    
    if (visionCount >= 6) {
        return null;
    }
    
    const device = {
        id: `vision_${deviceNumber}`,
        type: 'vision',
        name: `BC-200 ${deviceNumber}`,
        x: mathX,
        y: mathY,
        angle: 0,
        fovAngle: 0,
        offsetAngle: 0,
        offsetX: 0,
        offsetY: 0,
        status: 'Null',
        baseWidthM: 0.5,
        canDelete: true
    };
    
    ZoneFullMap_devices.push(device);
    return device;
}

function ZoneFullMap_createAnchorAtPosition(deviceNumber, mathX, mathY) {
    const anchorCount = ZoneFullMap_devices.filter(d => d.type === 'anchor').length;
    
    if (anchorCount >= 1) {
        return null;
    }
    
    const device = {
        id: `anchor_${deviceNumber}`,
        type: 'anchor',
        name: `Anchor ${deviceNumber}`,
        x: mathX,
        y: mathY,
        status: 'Active',
        baseWidthM: 30,
        canDelete: true
    };
    
    ZoneFullMap_devices.push(device);
    return device;
}

function ZoneFullMap_predictNextDeviceNumber(deviceType) {
    const currentCount = ZoneFullMap_devices.filter(d => d.type === deviceType).length;
    
    let startNumber = 1;
    if (typeof ZoneFullMap_getDeviceStartNumber === 'function') {
        startNumber = ZoneFullMap_getDeviceStartNumber(deviceType);
    }
    
    return currentCount + startNumber;
}

var gZoneFullMap_deviceVisibility = {};
var gZoneFullMap_selectedDevices = ["mic_1", "mic_2","mic_3", "mic_4"];

function ZoneFullMap_DeviceSelect() {
    let FullZoneMapWindow = document.getElementById('Zone_Full_Map_popup_Window');
    let deviceSelectWindow = document.getElementById('Device_Select_Window');
    const selectBtn = document.getElementById('ZoneFullMapDeviceSelectBtn');

    if(!FullZoneMapWindow || window.getComputedStyle(FullZoneMapWindow).display === 'none')
    {
        if (deviceSelectWindow) 
        {
            deviceSelectWindow.remove();
            if (selectBtn) 
            {
                selectBtn.classList.remove('active');
            }
        }
        return;
    }
    
    if (deviceSelectWindow) {
        deviceSelectWindow.remove();
        if (selectBtn) {
            selectBtn.classList.remove('active');
        }
        return;
    }
    
    if (selectBtn) {
        selectBtn.classList.add('active');
    }
    
    const devices = {
        microphones: [],
        cameras: [],
        visions: [],
        total: 0
    };

    if (typeof ZoneFullMap_devices !== 'undefined' && ZoneFullMap_devices.length > 0) {
        ZoneFullMap_devices.forEach(device => {
            if (!(device.id in gZoneFullMap_deviceVisibility)) {
                gZoneFullMap_deviceVisibility[device.id] = true;
            }
            
            switch (device.type) {
                case 'microphone':
                    devices.microphones.push({
                        id: device.id,
                        name: device.name,
                        x: device.x,
                        y: device.y,
                        visible: gZoneFullMap_deviceVisibility[device.id]
                    });
                    break;
                case 'camera':
                    devices.cameras.push({
                        id: device.id,
                        name: device.name,
                        x: device.x,
                        y: device.y,
                        visible: gZoneFullMap_deviceVisibility[device.id]
                    });
                    break;
                case 'vision':
                    devices.visions.push({
                        id: device.id,
                        name: device.name,
                        x: device.x,
                        y: device.y,
                        visible: gZoneFullMap_deviceVisibility[device.id]
                    });
                    break;
            }
        });
    }

    devices.total = devices.microphones.length + devices.cameras.length + devices.visions.length;

    if (devices.total === 0) {
        alert('No devices available on canvas');
        if (selectBtn) {
            selectBtn.classList.remove('active');
        }
        return;
    }

    const styleSheet = document.createElement('style');
    styleSheet.id = 'device-select-styles';
    styleSheet.innerHTML = `
        .Btn_style {
            width: 72px;
            height: 32px;
            border: 0px;
            border-radius: 3px;
            background: linear-gradient(180deg, rgba(26, 34, 41, 1) 0%, rgba(0, 14, 26, 1) 100%);
            color: rgba(137, 207, 216, 1);
            font-family: Arial;
            font-weight: bold;
            font-size: 14px;
            line-height: 26px;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .Btn_style:hover {
            background: linear-gradient(180deg, rgba(137, 207, 216, 1) 0%, rgba(137, 207, 216, 1) 83.33%, rgba(108, 116, 122, 1) 96.35%);
            color: rgba(19, 19, 19, 1);
        }

        .Btn_style:disabled {
            background: rgba(79, 79, 90, 1);
            color: rgba(123, 123, 123, 1);
            cursor: not-allowed;
        }

        .applyButton {
            margin-left: 64px;
            margin-right: 24px;
        }
        
        .device-checkbox {
            width: 24px;
            height: 24px;
            cursor: pointer;
            content: url("../images/checkbox_login_status_unselected.png");
            transition: all 0.2s ease;
        }
        
        .device-checkbox.checked {
            content: url("../images/checkbox_login_status_selected.png");
        }
        
        .device-item {
            display: flex;
            align-items: center;
            padding: 12px 16px;
            margin-bottom: 8px;
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(137, 207, 216, 0.2);
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .device-item:hover {
            background: rgba(137, 207, 216, 0.1);
            border-color: rgba(137, 207, 216, 0.4);
            transform: translateX(4px);
        }
        
        .device-item.selected {
            background: rgba(137, 207, 216, 0.15);
            border-color: #89cfd8;
        }
        
        .device-type-section {
            margin-bottom: 20px;
        }
        
        .device-type-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 12px 16px;
            background: rgba(137, 207, 216, 0.1);
            border-radius: 8px;
            margin-bottom: 12px;
            border: 1px solid rgba(137, 207, 216, 0.3);
        }
        
        .select-all-btn {
            padding: 4px 12px;
            background: rgba(137, 207, 216, 0.2);
            border: 1px solid #89cfd8;
            border-radius: 6px;
            color: #89cfd8;
            cursor: pointer;
            font-size: 12px;
            transition: all 0.2s ease;
        }
        
        .select-all-btn:hover {
            background: #89cfd8;
            color: #000e1a;
        }
    `;
    
    if (!document.getElementById('device-select-styles')) {
        document.head.appendChild(styleSheet);
    }

    deviceSelectWindow = document.createElement('div');
    deviceSelectWindow.id = 'Device_Select_Window';
    deviceSelectWindow.className = 'popup-window';
    deviceSelectWindow.style.cssText = `
        position: fixed;
        top: 10%;
        right: 2%;
        width: 600px;
        max-height: 80vh;
        background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
        border: 2px solid #89cfd8;
        border-radius: 12px;
        z-index: 10000;
        display: flex;
        flex-direction: column;
        box-shadow: 0 25px 70px rgba(0, 0, 0, 0.9);
    `;

    const titleBar = document.createElement('div');
    titleBar.id = 'deviceSelectTitleBar';
    titleBar.style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 18px 25px;
        background: rgba(0, 0, 0, 0.6);
        border-bottom: 2px solid rgba(137, 207, 216, 0.3);
        border-radius: 12px 12px 0 0;
        cursor: move;
        user-select: none;
    `;

    titleBar.innerHTML = `
        <div style="display: flex; align-items: center; gap: 15px;">
            <img src="../imagesaibox/CamConnect.png" style="width: 28px; height: 28px;">
            <span style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;">選擇設備</span>
            <span style="color: rgba(137, 207, 216, 0.7); font-family: Arial; font-size: 20px; font-weight: bold;">
                (${devices.total} devices)
            </span>
        </div>
        <button id="closeDeviceSelectBtn" style="
            background: rgba(255, 255, 255, 0.05);
            border: 2px solid #89cfd8;
            color: #89cfd8;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 20px;
            transition: all 0.3s;
            font-weight: bold;
            display: flex;
            justify-content: center;
            align-items: center;
            line-height: 1;
            padding: 0;
        " onmouseover="this.style.background='#ff4444'; this.style.borderColor='#ff4444'; this.style.color='white'; this.style.transform='rotate(90deg)'" 
           onmouseout="this.style.background='rgba(255,255,255,0.05)'; this.style.borderColor='#89cfd8'; this.style.color='#89cfd8'; this.style.transform='rotate(0)'">✕</button>
    `;

    const contentArea = document.createElement('div');
    contentArea.style.cssText = `
        flex: 1;
        padding: 25px;
        overflow-y: auto;
    `;

    let devicesHTML = '';
    
    if (devices.microphones.length > 0) {
        devicesHTML += `
            <div class="device-type-section">
                <div class="device-type-header">
                    <span style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;">麥克風設備 (${devices.microphones.length})</span>
                    <button style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;" class="select-all-btn" data-type="microphone" data-action="select">全選</button>
                </div>
                <div id="microphone-devices">
                    ${devices.microphones.map(device => `
                        <div class="device-item ${device.visible ? 'selected' : ''}" data-device-id="${device.id}" data-device-type="microphone">
                            <img class="device-checkbox ${device.visible ? 'checked' : ''}" data-device-id="${device.id}">
                            <div style="flex: 1; margin-left: 12px;">
                                <div style="color: #89cfd8; font-weight: bold; font-size: 20px;">${device.name}</div>
                                <div style="color: rgba(137, 207, 216, 0.6); font-weight: bold; font-size: 18px; margin-top: 2px;">位置: (${(device.x / 10000).toFixed(2)}, ${(device.y / 10000).toFixed(2)}) m </div>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }

    if (devices.cameras.length > 0) {
        devicesHTML += `
            <div class="device-type-section">
                <div class="device-type-header">
                    <span style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;">攝影機設備 (${devices.cameras.length})</span>
                    <button style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;" class="select-all-btn" data-type="camera" data-action="select">全選</button>
                </div>
                <div id="camera-devices">
                    ${devices.cameras.map(device => `
                        <div class="device-item ${device.visible ? 'selected' : ''}" data-device-id="${device.id}" data-device-type="camera">
                            <img class="device-checkbox ${device.visible ? 'checked' : ''}" data-device-id="${device.id}">
                            <div style="flex: 1; margin-left: 12px;">
                                <div style="color: #89cfd8; font-weight: bold; font-size: 20px;">${device.name}</div>
                                <div style="color: rgba(137, 207, 216, 0.6); font-weight: bold; font-size: 18px; margin-top: 2px;">位置: (${(device.x / 10000).toFixed(2)}, ${(device.y / 10000).toFixed(2)}) m </div>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }

    if (devices.visions.length > 0) {
        devicesHTML += `
            <div class="device-type-section">
                <div class="device-type-header">
                    <span style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;">BC-200 設備 (${devices.visions.length})</span>
                    <button style="color: #89cfd8; font-family: Arial; font-size: 18px; font-weight: bold;" class="select-all-btn" data-type="vision" data-action="select">全選</button>
                </div>
                <div id="vision-devices">
                    ${devices.visions.map(device => `
                        <div class="device-item ${device.visible ? 'selected' : ''}" data-device-id="${device.id}" data-device-type="vision">
                            <img class="device-checkbox ${device.visible ? 'checked' : ''}" data-device-id="${device.id}">
                            <div style="flex: 1; margin-left: 12px;">
                                <div style="color: #89cfd8; font-weight: bold; font-size: 20px;">${device.name}</div>
                                <div style="color: rgba(137, 207, 216, 0.6); font-weight: bold; font-size: 18px; margin-top: 2px;">位置: (${(device.x / 10000).toFixed(2)}, ${(device.y / 10000).toFixed(2)}) m </div>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }

    contentArea.innerHTML = devicesHTML;

    const bottomBar = document.createElement('div');
    bottomBar.style.cssText = `
        padding: 18px 25px;
        background: rgba(0, 0, 0, 0.6);
        border-top: 2px solid rgba(137, 207, 216, 0.3);
        display: flex;
        justify-content: flex-end;
        align-items: center;
    `;

    const applyButton = document.createElement('button');
    applyButton.id = 'applyDeviceSelectBtn';
    applyButton.type = 'button';
    applyButton.className = 'ZoneFullMapBtn_style applyButton';
    applyButton.innerText = 'Apply';
    applyButton.style.cssText = `
        margin-left: 64px;
        margin-right: 24px;
    `;

    const cancelButton = document.createElement('button');
    cancelButton.id = 'cancelDeviceSelectBtn';
    cancelButton.type = 'button';
    cancelButton.className = 'ZoneFullMapBtn_style';
    cancelButton.innerText = 'Cancel';

    bottomBar.appendChild(applyButton);
    bottomBar.appendChild(cancelButton);

    deviceSelectWindow.appendChild(titleBar);
    deviceSelectWindow.appendChild(contentArea);
    deviceSelectWindow.appendChild(bottomBar);
    
    document.body.appendChild(deviceSelectWindow);

    let isDragging = false;
    let currentX;
    let currentY;
    let initialX;
    let initialY;
    let xOffset = 0;
    let yOffset = 0;

    const dragElement = document.getElementById('deviceSelectTitleBar');

    dragElement.addEventListener('mousedown', dragStart);
    document.addEventListener('mousemove', drag);
    document.addEventListener('mouseup', dragEnd);

    function dragStart(e) {
        if (e.target.id === 'closeDeviceSelectBtn' || e.target.closest('#closeDeviceSelectBtn')) {
            return;
        }

        initialX = e.clientX - xOffset;
        initialY = e.clientY - yOffset;

        if (e.target === dragElement || dragElement.contains(e.target)) {
            isDragging = true;
            deviceSelectWindow.style.transition = 'none';
        }
    }

    function drag(e) {
        if (isDragging) {
            e.preventDefault();
            
            currentX = e.clientX - initialX;
            currentY = e.clientY - initialY;

            xOffset = currentX;
            yOffset = currentY;

            setTranslate(currentX, currentY, deviceSelectWindow);
        }
    }

    function dragEnd(e) {
        initialX = currentX;
        initialY = currentY;

        isDragging = false;
    }

    function setTranslate(xPos, yPos, el) {
        el.style.transform = `translate(calc(-50% + ${xPos}px), calc(-50% + ${yPos}px))`;
    }

    setTimeout(() => {
        document.querySelectorAll('.device-item').forEach(item => {
            item.addEventListener('click', function(e) {
                if (e.target.classList.contains('select-all-btn')) return;
                
                const checkbox = this.querySelector('.device-checkbox');
                checkbox.classList.toggle('checked');
                this.classList.toggle('selected');
            });
        });

        document.querySelectorAll('.select-all-btn').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.stopPropagation();
                const deviceType = this.getAttribute('data-type');
                const action = this.getAttribute('data-action');
                const container = document.getElementById(`${deviceType}-devices`);
                
                if (container) {
                    const items = container.querySelectorAll('.device-item');
                    items.forEach(item => {
                        const checkbox = item.querySelector('.device-checkbox');
                        
                        if (action === 'select') {
                            checkbox.classList.add('checked');
                            item.classList.add('selected');
                        } else {
                            checkbox.classList.remove('checked');
                            item.classList.remove('selected');
                        }
                    });
                    
                    if (action === 'select') {
                        this.textContent = '取消全選';
                        this.setAttribute('data-action', 'deselect');
                    } else {
                        this.textContent = '全選';
                        this.setAttribute('data-action', 'select');
                    }
                }
            });
        });

        document.getElementById('closeDeviceSelectBtn').addEventListener('click', function() {
            ZoneFullMap_DeviceSelect();
        });

        document.getElementById('cancelDeviceSelectBtn').addEventListener('click', function() {
            ZoneFullMap_DeviceSelect();
        });

        document.getElementById('applyDeviceSelectBtn').addEventListener('click', function() {
            gZoneFullMap_selectedDevices = [];
            
            document.querySelectorAll('.device-item').forEach(item => {
                const deviceId = item.getAttribute('data-device-id');
                const checkbox = item.querySelector('.device-checkbox');
                const isChecked = checkbox.classList.contains('checked');
                
                gZoneFullMap_deviceVisibility[deviceId] = isChecked;
                
                if (isChecked) {
                    gZoneFullMap_selectedDevices.push(deviceId);
                }
            });
            
            console.log('Device Visibility Updated:', gZoneFullMap_deviceVisibility);
            console.log('Selected Devices:', gZoneFullMap_selectedDevices);
            
            
            ZoneFullMap_DeviceSelect();

            ZoneFullMap_hideDeviceInfo();

            ZoneFullMap_updateDevicePositions();

            ZoneFullMap_redrawQuadrilaterals();
        });
    }, 100);
}

function ZoneFullMap_MicZoneEdit() 
{
    let FullZoneMapWindow = document.getElementById('Zone_Full_Map_popup_Window');
    let editWindow = document.getElementById('MicZone_Edit_Window');
    const editBtn = document.getElementById('ZoneFullMapMicZoneEditBtn');

    if(!FullZoneMapWindow || window.getComputedStyle(FullZoneMapWindow).display === 'none')
    {
        if (editWindow) 
        {
            editWindow.remove();
            if (editBtn) 
            {
                editBtn.classList.remove('active');
            }
        }
        return;
    }

    if (editWindow) {
        editWindow.remove();
        if (editBtn) {
            editBtn.classList.remove('active');
        }
        return;
    }
    
    if (editBtn) {
        editBtn.classList.add('active');
    }

    if (!document.getElementById('miczone-edit-styles')) {
        const styleSheet = document.createElement('style');
        styleSheet.id = 'miczone-edit-styles';
        styleSheet.innerHTML = `
            .edit-mode-btn {
                width: 100%;
                padding: 15px;
                margin-bottom: 12px;
                background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
                border: 2px solid rgba(137, 207, 216, 0.3);
                border-radius: 8px;
                color: #89cfd8;
                cursor: pointer;
                text-align: center;
                transition: all 0.3s ease;
                font-family: Arial;
                font-size: 15px;
                font-weight: bold;
            }
            
            .edit-mode-btn.active {
                background: linear-gradient(180deg, rgba(137, 207, 216, 1) 0%, rgba(137, 207, 216, 1) 100%);
                border-color: rgba(137, 207, 216, 1);
                color: rgba(19, 19, 19, 1);
                box-shadow: 0 4px 12px rgba(137, 207, 216, 0.3);
            }

            .edit-mode-btn:hover:not(.active):not([disabled]) {
                background: rgba(137, 207, 216, 0.15);
                border-color: rgba(137, 207, 216, 0.5);
            }

            .edit-mode-btn[disabled] {
                background: rgba(79, 79, 90, 1);
                color: rgba(123, 123, 123, 1);
                border-color: rgba(79, 79, 90, 1);
                cursor: not-allowed;
            }
        `;
        document.head.appendChild(styleSheet);
    }

    editWindow = document.createElement('div');
    editWindow.id = 'MicZone_Edit_Window';
    editWindow.style.cssText = `
        position: fixed;
        top: 10%;
        right: 2%;
        width: 380px;
        background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
        border: 2px solid #89cfd8;
        border-radius: 12px;
        z-index: 10000;
        box-shadow: 0 25px 70px rgba(0, 0, 0, 0.9);
    `;

    const titleBar = document.createElement('div');
    titleBar.id = 'micZoneEditTitleBar';
    titleBar.style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 15px 20px;
        background: rgba(0, 0, 0, 0.6);
        border-bottom: 2px solid rgba(137, 207, 216, 0.3);
        border-radius: 12px 12px 0 0;
        cursor: move;
        user-select: none;
    `;

    titleBar.innerHTML = `
        <div style="display: flex; align-items: center; gap: 12px;">
            <img src="../imagesaibox/CamConnect.png" style="width: 24px; height: 24px;">
            <span style="color: #89cfd8; font-family: Arial; font-size: 18px; font-weight: bold;">麥克風區域編輯</span>
        </div>
        <button id="closeMicZoneEditBtn" style="
            background: rgba(255, 255, 255, 0.05);
            border: 2px solid #89cfd8;
            color: #89cfd8;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 18px;
            transition: all 0.3s;
            font-weight: bold;
            display: flex;
            justify-content: center;
            align-items: center;
            line-height: 1;
            padding: 0;
        " onmouseover="this.style.background='#ff4444'; this.style.borderColor='#ff4444'; this.style.color='white'; this.style.transform='rotate(90deg)'" 
           onmouseout="this.style.background='rgba(255,255,255,0.05)'; this.style.borderColor='#89cfd8'; this.style.color='#89cfd8'; this.style.transform='rotate(0)'">✕</button>
    `;

    const contentArea = document.createElement('div');
    contentArea.style.padding = '20px';

    contentArea.innerHTML = `
        <button id="mapEditModeBtn" class="edit-mode-btn ${!gZoneFullMap_EditModeState ? 'active' : ''}">
            地圖編輯模式
        </button>
        
        <button id="zoneEditModeBtn" class="edit-mode-btn ${gZoneFullMap_EditModeState ? 'active' : ''}">
            麥克風區域編輯模式
        </button>
        
        <button id="applyZoneChangesBtn" class="edit-mode-btn" ${!gZoneFullMap_EditModeState ? 'disabled' : ''}>
            麥克風區域變更應用
        </button>
    `;

    editWindow.appendChild(titleBar);
    editWindow.appendChild(contentArea);
    
    document.body.appendChild(editWindow);

    let isDragging = false;
    let currentX = 0, currentY = 0, initialX = 0, initialY = 0;
    let xOffset = 0, yOffset = 0;

    const dragElement = document.getElementById('micZoneEditTitleBar');

    dragElement.addEventListener('mousedown', (e) => {
        if (e.target.id === 'closeMicZoneEditBtn' || e.target.closest('#closeMicZoneEditBtn')) return;
        
        initialX = e.clientX - xOffset;
        initialY = e.clientY - yOffset;
        
        if (e.target === dragElement || dragElement.contains(e.target)) {
            isDragging = true;
            editWindow.style.transition = 'none';
        }
    });

    document.addEventListener('mousemove', (e) => {
        if (isDragging) {
            e.preventDefault();
            currentX = e.clientX - initialX;
            currentY = e.clientY - initialY;
            xOffset = currentX;
            yOffset = currentY;
            editWindow.style.transform = `translate(calc(-50% + ${currentX}px), calc(-50% + ${currentY}px))`;
        }
    });

    document.addEventListener('mouseup', () => {
        initialX = currentX;
        initialY = currentY;
        isDragging = false;
    });

    setTimeout(() => {
        const mapEditBtn = document.getElementById('mapEditModeBtn');
        const zoneEditBtn = document.getElementById('zoneEditModeBtn');
        const applyBtn = document.getElementById('applyZoneChangesBtn');
        const displayButton = document.getElementById('ZoneFullMapZoneDisplayBtn');
        
        mapEditBtn.addEventListener('click', function() {
            if (gZoneFullMap_EditModeState) {
                gZoneFullMap_EditModeState = false;
                
                this.classList.add('active');
                zoneEditBtn.classList.remove('active');
                applyBtn.disabled = true;
                applyBtn.classList.remove('active');
                applyBtn.textContent = '麥克風區域變更應用';
                
                ZoneFullMapMicZoneDrawMode({ checked: false });
                
                if (displayButton) {
                    if (!gZoneFullMap_LastDisplayZoneState) {
                        ZoneFullMap_applyZoneDisplayChanges();  
                    }
                    if (gZoneFullMap_LastDisplayZoneState) {
                        displayButton.classList.add('active');
                    }
                    displayButton.classList.remove('disabled');
                    displayButton.disabled = false;
                }
                
                // 更新 Info Setup 視窗
                const infoZoneBtn = document.getElementById('zoneDisplayBtn');
                if (infoZoneBtn) {
                    infoZoneBtn.disabled = false;
                    infoZoneBtn.classList.toggle('active', gZoneFullMap_DisplayZoneState);
                    const btnText = infoZoneBtn.querySelector('div:first-child');
                    const descText = infoZoneBtn.querySelector('.info-description');
                    if (btnText) {
                        btnText.textContent = gZoneFullMap_DisplayZoneState ? '顯示麥克風區域' : '隱藏麥克風區';
                    }
                    if (descText) {
                        descText.textContent = 'Show or hide microphone zones on the map';
                        descText.style.color = 'rgba(137, 207, 216, 0.7)';
                    }
                }
            }
        });

        zoneEditBtn.addEventListener('click', function() {
            if (!gZoneFullMap_EditModeState) {
                gZoneFullMap_EditModeState = true;

                if(!gZoneFullMap_DisplayZoneState)
                {
                    gZoneFullMap_DisplayZoneState = true;
                }
                
                this.classList.add('active');
                mapEditBtn.classList.remove('active');
                applyBtn.disabled = false;
                
                ZoneFullMapMicZoneDrawMode({ checked: true });
                
                if (displayButton) {
                    if (!gZoneFullMap_DisplayZoneState) {
                        gZoneFullMap_DisplayZoneState = true;
                        displayButton.textContent = '顯示麥克風區域';
                        displayButton.classList.remove('active');
                    } else {            
                        displayButton.classList.remove('active');
                    }
                    displayButton.classList.add('disabled');
                    displayButton.disabled = true;
                }
                
                // 更新 Info Setup 視窗
                const infoZoneBtn = document.getElementById('zoneDisplayBtn');
                if (infoZoneBtn) {
                    infoZoneBtn.disabled = true;
                    infoZoneBtn.classList.add('active');
                    const btnText = infoZoneBtn.querySelector('div:first-child');
                    const descText = infoZoneBtn.querySelector('.info-description');
                    if (btnText) {
                        btnText.textContent = '顯示麥克風區域';
                    }
                    if (descText) {
                        descText.textContent = '在區域編輯模式下停用';
                        descText.style.color = 'rgba(255, 100, 100, 0.8)';
                    }
                }
            }
        });

        applyBtn.addEventListener('click', function() {
            if (!this.disabled) {
                this.classList.add('active');
                this.textContent = 'Applying...';
                this.disabled = true;
                
                ZoneFullMap_applyZoneChanges();
                
                setTimeout(() => {
                    this.textContent = 'Applied';
                    setTimeout(() => {
                        this.classList.remove('active');
                        this.textContent = 'Zone Change Apply';
                        this.disabled = false;
                    }, 1000);
                }, 500);
            }
        });

        document.getElementById('closeMicZoneEditBtn').addEventListener('click', function() {
            ZoneFullMap_MicZoneEdit();
        });
    }, 100);
}

function ZoneFullMap_InfoSetup() 
{
    let FullZoneMapWindow = document.getElementById('Zone_Full_Map_popup_Window');
    let infoWindow = document.getElementById('Info_Setup_Window');
    const infoBtn = document.getElementById('ZoneFullMapInfoSetupBtn');

    if(!FullZoneMapWindow || window.getComputedStyle(FullZoneMapWindow).display === 'none')
    {
        if (infoWindow)
        {
            infoWindow.remove();
            if (infoBtn) 
            {
                infoBtn.classList.remove('active');
            }
        }
        return;
    }
    
    if (infoWindow) {
        infoWindow.remove();
        if (infoBtn) {
            infoBtn.classList.remove('active');
        }
        return;
    }
    
    if (infoBtn) {
        infoBtn.classList.add('active');
    }

    if (!document.getElementById('info-setup-styles')) {
        const styleSheet = document.createElement('style');
        styleSheet.id = 'info-setup-styles';
        styleSheet.innerHTML = `
            .info-toggle-btn {
                width: 100%;
                padding: 20px;
                margin-bottom: 16px;
                background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
                border: 2px solid rgba(137, 207, 216, 0.3);
                border-radius: 10px;
                color: #89cfd8;
                cursor: pointer;
                text-align: center;
                transition: all 0.3s ease;
                font-family: Arial;
                font-size: 18px;
                font-weight: bold;
            }
            
            .info-toggle-btn.active {
                background: linear-gradient(180deg, rgba(137, 207, 216, 1) 0%, rgba(137, 207, 216, 1) 100%);
                border-color: rgba(137, 207, 216, 1);
                color: rgba(19, 19, 19, 1);
                box-shadow: 0 4px 12px rgba(137, 207, 216, 0.3);
            }

            .info-toggle-btn:hover:not([disabled]) {
                border-color: rgba(137, 207, 216, 0.6);
                transform: translateY(-2px);
            }

            .info-toggle-btn[disabled] {
                background: rgba(79, 79, 90, 1);
                color: rgba(123, 123, 123, 1);
                border-color: rgba(79, 79, 90, 1);
                cursor: not-allowed;
                transform: none;
            }

            .info-description {
                color: rgba(137, 207, 216, 0.7);
                font-size: 14px;
                margin-top: 8px;
                font-weight: normal;
                transition: color 0.3s ease;
            }
        `;
        document.head.appendChild(styleSheet);
    }

    infoWindow = document.createElement('div');
    infoWindow.id = 'Info_Setup_Window';
    infoWindow.style.cssText = `
        position: fixed;
        top: 10%;
        right: 2%;
        width: 500px;
        background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
        border: 2px solid #89cfd8;
        border-radius: 12px;
        z-index: 10000;
        box-shadow: 0 25px 70px rgba(0, 0, 0, 0.9);
    `;

    const titleBar = document.createElement('div');
    titleBar.id = 'infoSetupTitleBar';
    titleBar.style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 18px 25px;
        background: rgba(0, 0, 0, 0.6);
        border-bottom: 2px solid rgba(137, 207, 216, 0.3);
        border-radius: 12px 12px 0 0;
        cursor: move;
        user-select: none;
    `;

    titleBar.innerHTML = `
        <div style="display: flex; align-items: center; gap: 15px;">
            <img src="../imagesaibox/CamConnect.png" style="width: 28px; height: 28px;">
            <span style="color: #89cfd8; font-family: Arial; font-size: 20px; font-weight: bold;">資訊顯示設定</span>
        </div>
        <button id="closeInfoSetupBtn" style="
            background: rgba(255, 255, 255, 0.05);
            border: 2px solid #89cfd8;
            color: #89cfd8;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 20px;
            transition: all 0.3s;
            font-weight: bold;
            display: flex;
            justify-content: center;
            align-items: center;
            line-height: 1;
            padding: 0;
        " onmouseover="this.style.background='#ff4444'; this.style.borderColor='#ff4444'; this.style.color='white'; this.style.transform='rotate(90deg)'" 
           onmouseout="this.style.background='rgba(255,255,255,0.05)'; this.style.borderColor='#89cfd8'; this.style.color='#89cfd8'; this.style.transform='rotate(0)'">✕</button>
    `;

    const contentArea = document.createElement('div');
    contentArea.style.padding = '30px';

    const isZoneDisplayDisabled = gZoneFullMap_EditModeState;
    const zoneDisplayText = isZoneDisplayDisabled ? '顯示麥克風區域' : (gZoneFullMap_DisplayZoneState ? '顯示麥克風區域' : '隱藏麥克風區域顯示');
    const zoneDescriptionText = isZoneDisplayDisabled ? 'Disabled in Zone Edit Mode' : 'Show or hide microphone zones on the map';
    const zoneDescriptionColor = isZoneDisplayDisabled ? 'rgba(255, 100, 100, 0.8)' : 'rgba(137, 207, 216, 0.7)';

    contentArea.innerHTML = `
        <div style="margin-bottom: 20px;">
            <h3 style="color: #89cfd8; font-family: Arial; font-size: 18px; font-weight: bold; margin-bottom: 15px;">麥克風聲音資料可見性</h3>
            <button id="dataVisibilityBtn" class="info-toggle-btn ${gZoneFullMap_DataModeState ? 'active' : ''}">
                <div>${gZoneFullMap_DataModeState ? '所有麥克風資料可見' : '單麥克風資料可見'}</div>
                <div class="info-description">在單一麥克風或所有麥克風資料顯示之間切換</div>
            </button>
        </div>
        
        <div>
            <h3 style="color: #89cfd8; font-family: Arial; font-size: 18px; font-weight: bold; margin-bottom: 15px;">麥克風區域顯示</h3>
            <button id="zoneDisplayBtn" class="info-toggle-btn ${isZoneDisplayDisabled ? 'active' : (gZoneFullMap_DisplayZoneState ? 'active' : '')}" ${isZoneDisplayDisabled ? 'disabled' : ''}>
                <div>${zoneDisplayText}</div>
                <div class="info-description" style="color: ${zoneDescriptionColor};">${zoneDescriptionText}</div>
            </button>
        </div>
    `;

    infoWindow.appendChild(titleBar);
    infoWindow.appendChild(contentArea);
    document.body.appendChild(infoWindow);

    let isDragging = false;
    let currentX = 0, currentY = 0, initialX = 0, initialY = 0;
    let xOffset = 0, yOffset = 0;

    titleBar.addEventListener('mousedown', (e) => {
        if (e.target.id === 'closeInfoSetupBtn' || e.target.closest('#closeInfoSetupBtn')) return;
        
        initialX = e.clientX - xOffset;
        initialY = e.clientY - yOffset;
        
        if (e.target === titleBar || titleBar.contains(e.target)) {
            isDragging = true;
            infoWindow.style.transition = 'none';
        }
    });

    document.addEventListener('mousemove', (e) => {
        if (isDragging) {
            e.preventDefault();
            currentX = e.clientX - initialX;
            currentY = e.clientY - initialY;
            xOffset = currentX;
            yOffset = currentY;
            infoWindow.style.transform = `translate(calc(-50% + ${currentX}px), calc(-50% + ${currentY}px))`;
        }
    });

    document.addEventListener('mouseup', () => {
        initialX = currentX;
        initialY = currentY;
        isDragging = false;
    });

    setTimeout(() => {
        document.getElementById('dataVisibilityBtn').addEventListener('click', function() {
            ZoneFullMap_toggleDataMode();
        });

        document.getElementById('zoneDisplayBtn').addEventListener('click', function() {
            if (!this.disabled && !gZoneFullMap_EditModeState) {
                ZoneFullMap_applyZoneDisplayChanges();
            }
        });

        document.getElementById('closeInfoSetupBtn').addEventListener('click', () => {
            ZoneFullMap_InfoSetup();
        });
    }, 100);
}

function ZoneFullMap_toggleDataMode() {
    gZoneFullMap_DataModeState = !gZoneFullMap_DataModeState;
    
    const mainButton = document.getElementById('ZoneFullMapDataModeBtn');

    if(gZoneFullMap_DataModeState)
    {
        gMappingOverViewStatus = 4;
        getMappingOverViewBlockUIforPage();

        setTimeout(
            function()
            {
                getMappingOverViewUnblockUIforPage();
                gMappingOverViewStatus = 0;
            }
        ,600);
    }
    else
    {
        gMappingOverViewStatus = 5;
        getMappingOverViewBlockUIforPage();

        setTimeout(
            function()
            {
                getMappingOverViewUnblockUIforPage();
                gMappingOverViewStatus = 0;
            }
        ,600);
    }

    if (mainButton) {
        mainButton.textContent = gZoneFullMap_DataModeState ? '所有麥克風資料可見' : '單麥克風資料可見';
        mainButton.classList.toggle('active', gZoneFullMap_DataModeState);
    }
    
    const infoButton = document.getElementById('dataVisibilityBtn');
    if (infoButton) {
        infoButton.classList.toggle('active', gZoneFullMap_DataModeState);
        const btnText = infoButton.querySelector('div:first-child');
        if (btnText) {
            btnText.textContent = gZoneFullMap_DataModeState ? '所有麥克風資料可見' : '單麥克風資料可見';
        }
    }
    
    ZoneFullMapMicZoneAllDataShow({ checked: gZoneFullMap_DataModeState });
}

function ZoneFullMap_applyZoneDisplayChanges() {

    console.log('ZoneFullMap_applyZoneDisplayChanges : ',gZoneFullMap_DisplayZoneState,'gZoneFullMap_EditModeState : ',gZoneFullMap_EditModeState);

    if (gZoneFullMap_EditModeState) return;
    
    gZoneFullMap_DisplayZoneState = !gZoneFullMap_DisplayZoneState;
    gZoneFullMap_LastDisplayZoneState = gZoneFullMap_DisplayZoneState;
    
    if(gZoneFullMap_DisplayZoneState)
    {
        gMappingOverViewStatus = 6;
        getMappingOverViewBlockUIforPage();

        setTimeout(
            function()
            {
                getMappingOverViewUnblockUIforPage();
                gMappingOverViewStatus = 0;
            }
        ,600);
    }
    else
    {
        gMappingOverViewStatus = 7;
        getMappingOverViewBlockUIforPage();

        setTimeout(
            function()
            {
                getMappingOverViewUnblockUIforPage();
                gMappingOverViewStatus = 0;
            }
        ,600);
    }

    const mainButton = document.getElementById('ZoneFullMapZoneDisplayBtn');
    if (mainButton) {
        mainButton.textContent = gZoneFullMap_DisplayZoneState ? '顯示麥克風區域' : '隱藏麥克風區域顯示';
        mainButton.classList.toggle('active', gZoneFullMap_DisplayZoneState);
    }
    
    const infoButton = document.getElementById('zoneDisplayBtn');
    if (infoButton) {
        infoButton.classList.toggle('active', gZoneFullMap_DisplayZoneState);
        const btnText = infoButton.querySelector('div:first-child');
        if (btnText) {
            btnText.textContent = gZoneFullMap_DisplayZoneState ? '顯示麥克風區域' : '隱藏麥克風區域顯示';
        }
    }
    
    ZoneFullMap_redrawQuadrilaterals();
}


var gZoneFullMap_CurrentBC200IP = "";


function ZoneFullMap_BC200View(deviceId) {
    const device = ZoneFullMap_findDeviceById(deviceId);
    if (!device || !device.connectedCamera) {
        return;
    }
    
    const cameraIP = extractIPFromDisplayName(device.connectedCamera);
    const cameraName = device.connectedCamera;
    
    gZoneFullMap_CurrentDeviceId = deviceId;
    gZoneFullMap_CurrentBC200IP = cameraIP;

    ZoneFullMap_createBC200ViewWindow();
    
    const BC200ViewWindow = document.getElementById('cameralist_FullMapBC200ViewWindows');
    if (BC200ViewWindow) 
    {
        BC200ViewWindow.style.display = 'block';
        var jsonmsg = {};
        jsonmsg.Command = "GetBC200CameraCalibrationSetting";
        jsonmsg.BC200IPAddress = gZoneFullMap_CurrentBC200IP;
        sendMessageSettings(jsonmsg.Command,jsonmsg);
    
        console.log('sendMessageSettings BC200 IP:', cameraIP);
    }
}

//v1
// function ZoneFullMap_createBC200ViewWindow() {
//     var ZoneMapCameraSettingWindow = document.getElementById('cameralist_FullMapBC200ViewWindows');
//     ZoneMapCameraSettingWindow.innerHTML = '';
//     ZoneMapCameraSettingWindow.className = 'ZoneFullMap_BC200View_Centered';
//     ZoneMapCameraSettingWindow.style.weight = "680px";
//     if (gIsCalibrationMode) {
//         ZoneMapCameraSettingWindow.style.height = 'auto';  // 自動擴展以容納控制面板
//     } else {
//         ZoneMapCameraSettingWindow.style.height = '460px';  // 普通模式固定高度
//     }
    
//     ZoneMapCameraSettingWindow.style.userSelect = 'none';
//     ZoneMapCameraSettingWindow.style.webkitUserSelect = 'none';
//     ZoneMapCameraSettingWindow.style.msUserSelect = 'none';
//     ZoneMapCameraSettingWindow.style.mozUserSelect = 'none';

//     var modalPTZ = document.createElement('div');
//     modalPTZ.className = 'ZoneFullMap_BC200View_Modal';
//     modalPTZ.style.backgroundColor = 'black';
//     modalPTZ.style.display = 'block';
//     modalPTZ.style.width = '100%';
//     modalPTZ.style.height = '100%';
//     modalPTZ.id = 'BC200View_Modal';
//     applyNoDragStyle(modalPTZ);
    
//     var modalDialog = document.createElement('div');
//     applyNoDragStyle(modalDialog);
    
//     var modalContent = document.createElement('div');
//     applyNoDragStyle(modalContent);
    
//     var titleRow = document.createElement('tr');
//     var titleBarRowCell = document.createElement('td');
//     titleBarRowCell.className = 'BC200ViewtitleBarRow';
//     titleBarRowCell.style.cursor = 'move';
//     titleBarRowCell.id = 'BC200View_titleBar';
//     applyNoDragStyle(titleBarRowCell);

//     var logoCell = document.createElement('td');
//     logoCell.className = 'title-bar-cell logo-cell';
//     applyNoDragStyle(logoCell);
    
//     var logoImage = document.createElement('img');
//     logoImage.id = 'BC200Viewlogo_image';
//     logoImage.className = 'mic-popup-Window-logo-Image';
//     logoImage.src = '../imagesaibox/CamConnect.png';
//     logoImage.draggable = false;
//     applyNoDragStyle(logoImage);
//     logoCell.appendChild(logoImage);

//     var titleCell = document.createElement('td');
//     titleCell.className = 'title-bar-cell title-cell';
//     titleCell.style.margin = '10px';
//     applyNoDragStyle(titleCell);
    
//     var titleLabel = document.createElement('label');
//     titleLabel.className = 'Font_Arial_18_bold mic-xy-title';
//     titleLabel.textContent = 'Auxiliary Camera';
//     applyNoDragStyle(titleLabel);
//     titleCell.appendChild(titleLabel);

//     var closeButtonCell = document.createElement('td');
//     closeButtonCell.className = 'title-bar-cell close-button-cell';
//     applyNoDragStyle(closeButtonCell);
    
//     var closeButton = document.createElement('button');
//     closeButton.id = 'ZoneFullMap_close_CameraSetting_Window';
//     closeButton.className = 'close-popup-btn';
//     closeButton.setAttribute("onclick", "ZoneFullMap_closeBC200ViewWindow(this)");
//     applyNoDragStyle(closeButton);
//     closeButtonCell.appendChild(closeButton);

//     titleBarRowCell.appendChild(logoCell);
//     titleBarRowCell.appendChild(titleCell);
//     titleBarRowCell.appendChild(closeButtonCell);
//     titleRow.appendChild(titleBarRowCell);

//     var modalPTZBody = document.createElement('div');
//     modalPTZBody.className = 'modalPTZ-body';
//     modalPTZBody.style.backgroundColor = 'black';
//     applyNoDragStyle(modalPTZBody);
    
//     var imgContents = document.createElement('div');
//     imgContents.id = 'BC200View_imgcontents';
//     imgContents.style.cssText = 'position: relative; width: 100%; height: 100%; display: flex; justify-content: center; align-items: center;';
//     applyNoDragStyle(imgContents);
    
//     var imgLoader = document.createElement('img');
//     imgLoader.id = 'BC200View_loader';
//     imgLoader.className = 'BC200Viewimgloader';
//     imgLoader.style.height = '100%';
//     imgLoader.draggable = false;
//     applyNoDragStyle(imgLoader);
    
//     imgContents.appendChild(imgLoader);
//     modalPTZBody.appendChild(imgContents);

//     modalContent.appendChild(titleRow);
//     modalContent.appendChild(modalPTZBody);

//     // 如果是校準模式，添加控制面板
//     if (gIsCalibrationMode) {
//         // 座標顯示區域
//         var coordPanel = document.createElement('div');
//         coordPanel.id = 'BC200_CoordPanel';
//         coordPanel.style.cssText = `
//             padding: 20px;
//             background: #000000;
//             display: flex;
//             gap: 30px;
//             justify-content: center;
//             align-items: center;
//         `;
//         applyNoDragStyle(coordPanel);
        
//         coordPanel.innerHTML = `
//             <div style="display: flex; align-items: center; gap: 10px;">
//                 <label style="color: #89cfd8; font-size: 16px; font-weight: bold; font-family: Arial; min-width: 90px;">Screen X:</label>
//                 <input type="text" id="BC200_CoordX" readonly style="
//                     width: 120px;
//                     padding: 10px;
//                     background: rgba(26, 34, 41, 0.8);
//                     border: 2px solid #89cfd8;
//                     border-radius: 6px;
//                     color: #89cfd8;
//                     font-size: 16px;
//                     font-family: Arial;
//                     text-align: center;
//                     font-weight: bold;
//                 " value="0" />
//             </div>
//             <div style="display: flex; align-items: center; gap: 10px;">
//                 <label style="color: #89cfd8; font-size: 16px; font-weight: bold; font-family: Arial; min-width: 90px;">Screen Y:</label>
//                 <input type="text" id="BC200_CoordY" readonly style="
//                     width: 120px;
//                     padding: 10px;
//                     background: rgba(26, 34, 41, 0.8);
//                     border: 2px solid #89cfd8;
//                     border-radius: 6px;
//                     color: #89cfd8;
//                     font-size: 16px;
//                     font-family: Arial;
//                     text-align: center;
//                     font-weight: bold;
//                 " value="0" />
//             </div>
//         `;
        
//         modalContent.appendChild(coordPanel);
        
//         // 按鈕區域
//         var buttonRow = document.createElement('div');
//         buttonRow.style.cssText = `
//             padding: 20px;
//             background: #000000;
//             border-top: 1px solid rgba(137, 207, 216, 0.3);
//             display: flex;
//             justify-content: flex-end;
//             gap: 15px;
//         `;
//         applyNoDragStyle(buttonRow);

//         // 統一的按鈕基礎樣式（完全參考原按鈕）
//         const buttonBaseStyle = `
//             padding: 14px 20px;
//             background: linear-gradient(180deg, #1a2229 0%, #000e1a 100%);
//             border: 2px solid #89cfd8;
//             border-radius: 10px;
//             color: #89cfd8;
//             cursor: pointer;
//             font-size: 16px;
//             font-weight: bold;
//             font-family: Arial;
//             transition: all 0.3s ease;
//         `;

//         // Apply 按鈕
//         var applyBtn = document.createElement('button');
//         applyBtn.textContent = 'Apply';
//         applyBtn.className = 'BC200PanTiltBtn_style';
//         applyBtn.style.classList = "width:120px;height:32px;";
//         applyBtn.onclick = function() {
//             const x = document.getElementById('BC200_CoordX').value;
//             const y = document.getElementById('BC200_CoordY').value;
//             console.log('BC200 Apply - X:', x, 'Y:', y);
//             alert('BC200 calibration applied!\nX: ' + x + ', Y: ' + y);
//         };
//         applyNoDragStyle(applyBtn);

//         // Cancel 按鈕
//         var cancelBtn = document.createElement('button');
//         cancelBtn.textContent = 'Cancel';
//         cancelBtn.className = 'BC200PanTiltBtn_style';
//         cancelBtn.style.classList = "width:120px;height:32px;";
//         cancelBtn.onclick = function() {
//             if (confirm('Are you sure you want to cancel BC200 calibration?')) {
//                 gIsCalibrationMode = false;
//                 ZoneFullMap_closeBC200ViewWindow();
//             }
//         };
//         applyNoDragStyle(cancelBtn);

//         buttonRow.appendChild(applyBtn);
//         buttonRow.appendChild(cancelBtn);
//         modalContent.appendChild(buttonRow);
        
//         // 滑鼠移動更新座標
//         imgContents.addEventListener('mousemove', function(e) {
//             const rect = imgContents.getBoundingClientRect();
//             const x = Math.round(e.clientX - rect.left);
//             const y = Math.round(e.clientY - rect.top);
//             const xInput = document.getElementById('BC200_CoordX');
//             const yInput = document.getElementById('BC200_CoordY');
//             if (xInput) xInput.value = x;
//             if (yInput) yInput.value = y;
//         });
//     }


//     modalDialog.appendChild(modalContent);
//     modalPTZ.appendChild(modalDialog);
//     ZoneMapCameraSettingWindow.appendChild(modalPTZ);

//     makeDraggable(ZoneMapCameraSettingWindow, titleBarRowCell);

//     if (gIsCalibrationMode) {
//         const imgLoader = document.getElementById('BC200View_loader');
        
//         function addBC200Crosshair() {
//             const imgContents = document.getElementById('BC200View_imgcontents');
//             if (!imgContents) return;
            
//             // 移除舊的十字架（避免重複）
//             const oldH = imgContents.querySelector('.bc200-crosshair-h');
//             const oldV = imgContents.querySelector('.bc200-crosshair-v');
//             if (oldH) oldH.remove();
//             if (oldV) oldV.remove();
            
//             // 創建新的十字架
//             var crosshairHorizontal = document.createElement('div');
//             crosshairHorizontal.className = 'bc200-crosshair-h';
//             crosshairHorizontal.style.cssText = `
//                 position: absolute;
//                 width: 100px;
//                 height: 2px;
//                 background: limegreen;
//                 top: 50%;
//                 left: 50%;
//                 transform: translate(-50%, -50%);
//                 z-index: 10;
//                 pointer-events: none;
//             `;
//             applyNoDragStyle(crosshairHorizontal);
            
//             var crosshairVertical = document.createElement('div');
//             crosshairVertical.className = 'bc200-crosshair-v';
//             crosshairVertical.style.cssText = `
//                 position: absolute;
//                 width: 2px;
//                 height: 100px;
//                 background: limegreen;
//                 top: 50%;
//                 left: 50%;
//                 transform: translate(-50%, -50%);
//                 z-index: 10;
//                 pointer-events: none;
//             `;
//             applyNoDragStyle(crosshairVertical);
            
//             imgContents.appendChild(crosshairHorizontal);
//             imgContents.appendChild(crosshairVertical);
//         }

//         // if (imgLoader) {
//         //     imgLoader.addEventListener('load', addBC200Crosshair);
            
//         //     if (imgLoader.complete && imgLoader.naturalWidth > 0) {
//         //         addBC200Crosshair();
//         //     }
//         // }
//         addBC200Crosshair();//test
        
//         window.addBC200Crosshair = addBC200Crosshair;
//     }
// }

//v2
function ZoneFullMap_createBC200ViewWindow() {
    var ZoneMapCameraSettingWindow = document.getElementById('cameralist_FullMapBC200ViewWindows');
    ZoneMapCameraSettingWindow.innerHTML = '';
    ZoneMapCameraSettingWindow.className = 'ZoneFullMap_BC200View_Centered';
    // 調整視窗寬度以適應 640px 的內容
    ZoneMapCameraSettingWindow.style.width = "700px";
    if (gIsCalibrationMode) {
        ZoneMapCameraSettingWindow.style.height = 'auto';  // 自動擴展以容納控制面板
    } else {
        ZoneMapCameraSettingWindow.style.height = '460px';  // 普通模式固定高度
    }
    
    ZoneMapCameraSettingWindow.style.userSelect = 'none';
    ZoneMapCameraSettingWindow.style.webkitUserSelect = 'none';
    ZoneMapCameraSettingWindow.style.msUserSelect = 'none';
    ZoneMapCameraSettingWindow.style.mozUserSelect = 'none';

    var modalPTZ = document.createElement('div');
    modalPTZ.className = 'ZoneFullMap_BC200View_Modal';
    modalPTZ.style.backgroundColor = 'black';
    modalPTZ.style.display = 'block';
    modalPTZ.style.width = '100%';
    modalPTZ.style.height = '100%';
    modalPTZ.id = 'BC200View_Modal';
    applyNoDragStyle(modalPTZ);
    
    var modalDialog = document.createElement('div');
    applyNoDragStyle(modalDialog);
    
    var modalContent = document.createElement('div');
    applyNoDragStyle(modalContent);
    
    var titleRow = document.createElement('tr');
    var titleBarRowCell = document.createElement('td');
    titleBarRowCell.className = 'BC200ViewtitleBarRow';
    titleBarRowCell.style.cursor = 'move';
    titleBarRowCell.id = 'BC200View_titleBar';
    applyNoDragStyle(titleBarRowCell);

    var logoCell = document.createElement('td');
    logoCell.className = 'title-bar-cell logo-cell';
    applyNoDragStyle(logoCell);
    
    var logoImage = document.createElement('img');
    logoImage.id = 'BC200Viewlogo_image';
    logoImage.className = 'mic-popup-Window-logo-Image';
    logoImage.src = '../imagesaibox/CamConnect.png';
    logoImage.draggable = false;
    applyNoDragStyle(logoImage);
    logoCell.appendChild(logoImage);

    var titleCell = document.createElement('td');
    titleCell.className = 'title-bar-cell title-cell';
    titleCell.style.margin = '10px';
    applyNoDragStyle(titleCell);
    
    var titleLabel = document.createElement('label');
    titleLabel.className = 'Font_Arial_18_bold mic-xy-title';
    titleLabel.textContent = 'Auxiliary Camera';
    applyNoDragStyle(titleLabel);
    titleCell.appendChild(titleLabel);

    var closeButtonCell = document.createElement('td');
    closeButtonCell.className = 'title-bar-cell close-button-cell';
    applyNoDragStyle(closeButtonCell);
    
    var closeButton = document.createElement('button');
    closeButton.id = 'ZoneFullMap_close_CameraSetting_Window';
    closeButton.className = 'close-popup-btn';
    closeButton.setAttribute("onclick", "ZoneFullMap_closeBC200ViewWindow(this)");
    applyNoDragStyle(closeButton);
    closeButtonCell.appendChild(closeButton);

    titleBarRowCell.appendChild(logoCell);
    titleBarRowCell.appendChild(titleCell);
    titleBarRowCell.appendChild(closeButtonCell);
    titleRow.appendChild(titleBarRowCell);

    var modalPTZBody = document.createElement('div');
    modalPTZBody.className = 'modalPTZ-body';
    modalPTZBody.style.backgroundColor = 'black';
    // 強制設定 modalPTZBody 的高度來容納 640x360 的內容
    modalPTZBody.style.cssText = 'background-color: black; min-height: 360px; padding: 10px; display: flex; justify-content: center; align-items: center;';
    applyNoDragStyle(modalPTZBody);
    
    var imgContents = document.createElement('div');
    imgContents.id = 'BC200View_imgcontents';
    // 強制設定為 640x360 的容器
    imgContents.style.cssText = 'position: relative; width: 640px; height: 360px; display: flex; justify-content: center; align-items: center; margin: 0 auto;';
    applyNoDragStyle(imgContents);
    
    if (gIsCalibrationMode) {
        var canvas = document.createElement('canvas');
        canvas.id = 'BC200_calibrationCanvas';
        canvas.width = 640;
        canvas.height = 360;
        canvas.style.cssText = 'border: 2px solid #89cfd8; cursor: crosshair; background-color: #1a1a1a; width: 640px; height: 360px;';
        applyNoDragStyle(canvas);
        
        imgContents.appendChild(canvas);
        
        // 初始化畫布
        initializeBC200CalibrationCanvas(canvas);
    } else {
        // 非校準模式也強制顯示 640x360 的區域
        var imgLoader = document.createElement('img');
        imgLoader.id = 'BC200View_loader';
        imgLoader.className = 'BC200Viewimgloader';
        imgLoader.style.cssText = 'width: 640px; height: 360px; object-fit: contain; background-color: #1a1a1a;';
        imgLoader.draggable = false;
        applyNoDragStyle(imgLoader);
        
        // 如果沒有串流，顯示佔位符或預設圖片
        imgLoader.onerror = function() {
            // 沒有圖片時顯示黑色背景
            this.style.backgroundColor = '#1a1a1a';
        };
        
        imgContents.appendChild(imgLoader);
    }
    
    modalPTZBody.appendChild(imgContents);

    modalContent.appendChild(titleRow);
    modalContent.appendChild(modalPTZBody);

    if (gIsCalibrationMode) {
        var coordPanel = document.createElement('div');
        coordPanel.id = 'BC200_CoordPanel';
        coordPanel.style.cssText = `
            padding: 20px;
            background: #000000;
            display: flex;
            gap: 30px;
            justify-content: center;
            align-items: center;
        `;
        applyNoDragStyle(coordPanel);
        
        var xContainer = document.createElement('div');
        xContainer.style.cssText = 'display: flex; align-items: center; gap: 10px;';
        
        var xTitle = document.createElement('label');
        xTitle.className = 'Font_Arial_16_bold';
        xTitle.style.cssText = 'color: #89cfd8; min-width: 90px;';
        xTitle.textContent = 'Screen X:';
        applyNoDragStyle(xTitle);
        
        var Label_Calibration_X = document.createElement('label');
        Label_Calibration_X.className = 'Font_Arial_16_bold';
        Label_Calibration_X.id = 'Label_BC200_Camera_X';
        Label_Calibration_X.style.cssText = `
            min-width: 120px;
            padding: 10px;
            background: rgba(26, 34, 41, 0.8);
            border: 2px solid #89cfd8;
            border-radius: 6px;
            color: #89cfd8;
            text-align: center;
            display: inline-block;
        `;
        Label_Calibration_X.textContent = '0';
        applyNoDragStyle(Label_Calibration_X);
        
        xContainer.appendChild(xTitle);
        xContainer.appendChild(Label_Calibration_X);
        
        var yContainer = document.createElement('div');
        yContainer.style.cssText = 'display: flex; align-items: center; gap: 10px;';
        
        var yTitle = document.createElement('label');
        yTitle.className = 'Font_Arial_16_bold';
        yTitle.style.cssText = 'color: #89cfd8; min-width: 90px;';
        yTitle.textContent = 'Screen Y:';
        applyNoDragStyle(yTitle);
        
        var Label_Calibration_Y = document.createElement('label');
        Label_Calibration_Y.className = 'Font_Arial_16_bold';
        Label_Calibration_Y.id = 'Label_BC200_Camera_Y';
        Label_Calibration_Y.style.cssText = `
            min-width: 120px;
            padding: 10px;
            background: rgba(26, 34, 41, 0.8);
            border: 2px solid #89cfd8;
            border-radius: 6px;
            color: #89cfd8;
            text-align: center;
            display: inline-block;
        `;
        Label_Calibration_Y.textContent = '0';
        applyNoDragStyle(Label_Calibration_Y);
        
        yContainer.appendChild(yTitle);
        yContainer.appendChild(Label_Calibration_Y);
        
        coordPanel.appendChild(xContainer);
        coordPanel.appendChild(yContainer);
        
        modalContent.appendChild(coordPanel);
        
        // 按鈕區域
        var buttonRow = document.createElement('div');
        buttonRow.style.cssText = `
            padding: 20px;
            background: #000000;
            border-top: 1px solid rgba(137, 207, 216, 0.3);
            display: flex;
            justify-content: flex-end;
            gap: 15px;
        `;
        applyNoDragStyle(buttonRow);

        // Apply 按鈕
        var applyBtn = document.createElement('button');
        applyBtn.textContent = 'Apply';
        applyBtn.className = 'BC200PanTiltBtn_style';
        applyBtn.style.cssText = "width:120px;height:32px;";
        applyBtn.onclick = function() {
            // 收集點數據並發送到後端
            if (window.gBC200CalibrationState && window.gBC200CalibrationState.currentPoint) {
                var pointsData = collectBC200PointsForBackend();
                console.log('Preparing to send calibration data:', pointsData);
                
                // 檢查是否有設定 BC200 IP
                if (!window.gZoneFullMap_CurrentBC200IP) {
                    alert('Warning: BC200 IP address not set.\nPlease ensure the camera is properly configured.');
                }
                
                // 發送點數據到後端進行最終確認
                sendBC200CalibrationToBackend(pointsData);
            } else {
                alert('Please click on the canvas to set a calibration point before applying.');
            }
        };
        applyNoDragStyle(applyBtn);

        // Cancel 按鈕
        var cancelBtn = document.createElement('button');
        cancelBtn.textContent = 'Cancel';
        cancelBtn.className = 'BC200PanTiltBtn_style';
        cancelBtn.style.cssText = "width:120px;height:32px;";
        cancelBtn.onclick = function() {
            if (confirm('Are you sure you want to cancel BC200 calibration?')) {
                gIsCalibrationMode = false;
                ZoneFullMap_closeBC200ViewWindow();
            }
        };
        applyNoDragStyle(cancelBtn);

        buttonRow.appendChild(applyBtn);
        buttonRow.appendChild(cancelBtn);
        modalContent.appendChild(buttonRow);
    }

    modalDialog.appendChild(modalContent);
    modalPTZ.appendChild(modalDialog);
    ZoneMapCameraSettingWindow.appendChild(modalPTZ);

    makeDraggable(ZoneMapCameraSettingWindow, titleBarRowCell);
}

function collectBC200PointsForBackend() {
    if (!gBC200CalibrationState.currentPoint) {
        return [];
    }
    
    return [{
        originalX: gBC200CalibrationState.currentPoint.x,
        originalY: gBC200CalibrationState.currentPoint.y
    }];
}

function initializeBC200CalibrationCanvas(canvas) {
    var ctx = canvas.getContext('2d');
    
    // 初始化狀態
    if (!window.gBC200CalibrationState) {
        window.gBC200CalibrationState = {};
    }
    
    gBC200CalibrationState.currentPoint = null;
    
    canvas.addEventListener('click', function(e) {
        var rect = canvas.getBoundingClientRect();
        var x = e.clientX - rect.left;
        var y = e.clientY - rect.top;
        
        gBC200CalibrationState.currentPoint = { x: x, y: y };
    
        sendSinglePointToBackend(x, y);
        
        drawBC200CalibrationPoints(canvas);
    });
    
    canvas.setAttribute('tabindex', '0');
    canvas.addEventListener('keydown', function(e) {
        if (e.key === 'Delete' && gBC200CalibrationState.currentPoint !== null) {
            gBC200CalibrationState.currentPoint = null;
            updateBC200CoordinateLabels(0, 0);
            drawBC200CalibrationPoints(canvas);
            console.log('Deleted calibration point');
        }
    });
    
    drawBC200CalibrationPoints(canvas);
    
    addBC200CanvasCrosshair();
}

function addBC200CanvasCrosshair() {
    var canvas = document.getElementById('BC200_calibrationCanvas');
    if (!canvas) return;
    
    var parentDiv = canvas.parentElement;
    if (!parentDiv) return;
    
    var oldH = parentDiv.querySelector('.bc200-canvas-crosshair-h');
    var oldV = parentDiv.querySelector('.bc200-canvas-crosshair-v');
    if (oldH) oldH.remove();
    if (oldV) oldV.remove();
    
    var crosshairHorizontal = document.createElement('div');
    crosshairHorizontal.className = 'bc200-canvas-crosshair-h';
    crosshairHorizontal.style.cssText = `
        position: absolute;
        width: 100px;
        height: 2px;
        background: limegreen;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        z-index: 10;
        pointer-events: none;
    `;
    
    var crosshairVertical = document.createElement('div');
    crosshairVertical.className = 'bc200-canvas-crosshair-v';
    crosshairVertical.style.cssText = `
        position: absolute;
        width: 2px;
        height: 100px;
        background: limegreen;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        z-index: 10;
        pointer-events: none;
    `;
    
    parentDiv.appendChild(crosshairHorizontal);
    parentDiv.appendChild(crosshairVertical);
}

function drawBC200CalibrationPoints(canvas) {
    var ctx = canvas.getContext('2d');
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    
    ctx.fillStyle = '#0a0a0a';
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    
    ctx.strokeStyle = 'limegreen';
    ctx.lineWidth = 2;
    
    ctx.beginPath();
    ctx.moveTo(290, 180);  // 320 - 50
    ctx.lineTo(350, 180);  // 320 + 50
    ctx.stroke();
    
    ctx.beginPath();
    ctx.moveTo(320, 150);  // 180 - 50
    ctx.lineTo(320, 210);  // 180 + 50
    ctx.stroke();
    
    if (!gBC200CalibrationState.currentPoint) {
        ctx.fillStyle = 'rgba(137, 207, 216, 0.5)';
        ctx.font = '16px Arial';
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.fillText('Click to set calibration point', canvas.width / 2, canvas.height / 2 - 70);
        ctx.fillText('(No stream available)', canvas.width / 2, canvas.height / 2 + 70);
    } else {
        var point = gBC200CalibrationState.currentPoint;
        
        ctx.strokeStyle = 'rgba(255, 0, 0, 1)';
        ctx.lineWidth = 2;
        
        ctx.beginPath();
        ctx.moveTo(point.x - 20, point.y);
        ctx.lineTo(point.x - 8, point.y);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.moveTo(point.x + 8, point.y);
        ctx.lineTo(point.x + 20, point.y);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.moveTo(point.x, point.y - 20);
        ctx.lineTo(point.x, point.y - 8);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.moveTo(point.x, point.y + 8);
        ctx.lineTo(point.x, point.y + 20);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.arc(point.x, point.y, 4, 0, 2 * Math.PI);
        ctx.fillStyle = 'rgba(255, 0, 0, 1)';
        ctx.fill();
        
        ctx.beginPath();
        ctx.arc(point.x, point.y, 12, 0, 2 * Math.PI);
        ctx.strokeStyle = 'rgba(255, 0, 0, 0.6)';
        ctx.lineWidth = 1;
        ctx.stroke();
    }
}

function updateBC200CoordinateLabels(x, y) {
    var xLabel = document.getElementById('Label_BC200_Camera_X');
    var yLabel = document.getElementById('Label_BC200_Camera_Y');
    
    if (xLabel) xLabel.textContent = x.toString();
    if (yLabel) yLabel.textContent = y.toString();
}

function sendSinglePointToBackend(x, y) {
    var jsonmsg = {};
    jsonmsg.Command = "GetBC200CameraCalibrationSetting";
    jsonmsg.BC200OriginalX = x;
    jsonmsg.BC200OriginalY = y;
    sendMessageSettings(jsonmsg.Command, jsonmsg);
    
    console.log('Sending point for correction (640x360 coordinates):', jsonmsg);
    console.log('  Coordinates: (' + x + ', ' + y + ')');
}


function applyNoDragStyle(element) {
    element.style.userSelect = 'none';
    element.style.webkitUserSelect = 'none';
    element.style.msUserSelect = 'none';
    element.style.mozUserSelect = 'none';
    element.style.webkitUserDrag = 'none';
    element.style.webkitTouchCallout = 'none';
}

function makeDraggable(element, handle) {
    let pos1 = 0, pos2 = 0, pos3 = 0, pos4 = 0;
    
    handle.onmousedown = dragMouseDown;

    function dragMouseDown(e) {
        e = e || window.event;
        e.preventDefault();
        
        if (element.style.transform && element.style.transform.includes('translate')) {
            const rect = element.getBoundingClientRect();
            element.style.top = rect.top + 'px';
            element.style.left = rect.left + 'px';
            element.style.transform = 'none';
        }
        
        pos3 = e.clientX;
        pos4 = e.clientY;
        document.onmouseup = closeDragElement;
        document.onmousemove = elementDrag;
    }

    function elementDrag(e) {
        e = e || window.event;
        e.preventDefault();
        
        pos1 = pos3 - e.clientX;
        pos2 = pos4 - e.clientY;
        pos3 = e.clientX;
        pos4 = e.clientY;
        
        element.style.top = (element.offsetTop - pos2) + "px";
        element.style.left = (element.offsetLeft - pos1) + "px";
    }

    function closeDragElement() {
        document.onmouseup = null;
        document.onmousemove = null;
    }
}

function ZoneFullMap_closeBC200ViewWindow() 
{
    const BC200ViewWindow = document.getElementById('cameralist_FullMapBC200ViewWindows');
    if (BC200ViewWindow) 
    {
        BC200ViewWindow.style.display = 'none';
        
        var jsonmsg = {};
        jsonmsg.Command = "GetBC200CameraCalibrationSetting";
        jsonmsg.BC200IPAddress = "closeCalibrationWindows";
        sendMessageSettings(jsonmsg.Command,jsonmsg);   

        if(gIsCalibrationMode)
        {
            gIsCalibrationMode = false;
            const cameraPTZWindow = document.getElementById('cameralist_FullMapCameraPTZWindows');
            if (cameraPTZWindow) 
            {
                ZoneFullMap_closePTZWindow();
            }
            setTimeout(function(){
                ZoneFullMap_openDeviceConfigMap();
            },30);
        }
    }
}
